#include "activation_spotter_capability.h"

#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/spotter_types/spotter_types.h>

using namespace quasar;
using namespace YandexIO;

namespace {

    constexpr auto REMOTE_OBJECT_NAME = "ActivationSpotterCapability";

    std::string modelsToString(const std::map<std::string, std::string>& models) {
        std::stringstream ss;
        for (const auto& [type, path] : models) {
            ss << "[" << type << ":" << path << "]";
        }
        return ss.str();
    }

} // namespace

ActivationSpotterCapability::ActivationSpotterCapability(
    std::shared_ptr<quasar::ICallbackQueue> callbackQueue,
    const AliceConfig& aliceConfig,
    std::shared_ptr<QuasarVoiceDialog> voiceDialog,
    AliceDeviceState& deviceState,
    std::weak_ptr<YandexIO::IRemotingRegistry> remotingRegistry)
    : IRemoteObject(std::move(remotingRegistry))
    , callbackQueue_(std::move(callbackQueue))
    , aliceConfig_(aliceConfig)
    , voiceDialog_(std::move(voiceDialog))
    , deviceState_(deviceState)
{
}

ActivationSpotterCapability::~ActivationSpotterCapability() {
    if (auto remotingRegistry = getRemotingRegistry().lock()) {
        remotingRegistry->removeRemoteObject(REMOTE_OBJECT_NAME);
    }
}

void ActivationSpotterCapability::init() {
    Y_ENSURE_THREAD(callbackQueue_);

    if (auto remotingRegistry = getRemotingRegistry().lock()) {
        remotingRegistry->addRemoteObject(REMOTE_OBJECT_NAME, weak_from_this());
    }
}

void ActivationSpotterCapability::start() {
    Y_ENSURE_THREAD(callbackQueue_);

    if (isConfigured()) {
        voiceDialog_->startPhraseSpotter();
    } else {
        pendingStartSpotter_ = true;
    }
}

const std::map<std::string, std::string>& ActivationSpotterCapability::getModelPaths() const {
    Y_ENSURE_THREAD(callbackQueue_);

    return typeToModelPaths_;
}

void ActivationSpotterCapability::onModelError() {
    Y_ENSURE_THREAD(callbackQueue_);

    typeToModelPaths_.clear();
    updateVoiceDialogSettings();

    remoteNotifyModelError();
}

void ActivationSpotterCapability::handleRemotingMessage(
    const quasar::proto::Remoting& message,
    std::shared_ptr<YandexIO::IRemotingConnection> connection)
{
    if (!message.has_spotter_capability_method()) {
        return;
    }

    callbackQueue_->add([this, method{message.spotter_capability_method()}, connection{std::move(connection)}]() mutable {
        connection_ = std::move(connection);

        switch (method.method()) {
            case quasar::proto::Remoting::SpotterCapabilityMethod::SET_MODEL_PATHS:
                processSetModelPaths(method.model_path());
                return;
            case quasar::proto::Remoting::SpotterCapabilityMethod::SET_SPOTTER_WORD:
                processSetSpotterWord(method.spotter_word());
                return;
        }
    });
}

void ActivationSpotterCapability::updateVoiceDialogSettings() {
    auto settings = aliceConfig_.getVoiceServiceSettings(
        deviceState_.getOAuthToken(),
        deviceState_.getPassportUid(),
        typeToModelPaths_[SpotterTypes::ACTIVATION],
        typeToModelPaths_[SpotterTypes::INTERRUPTION],
        typeToModelPaths_[SpotterTypes::ADDITIONAL],
        deviceState_.getWifiList(),
        deviceState_.formatExperiments());

    voiceDialog_->setSettings(std::move(settings));
}

bool ActivationSpotterCapability::isConfigured() const {
    const auto it = typeToModelPaths_.find(SpotterTypes::ACTIVATION);
    return it == typeToModelPaths_.end() ? false : !it->second.empty();
}

void ActivationSpotterCapability::remoteNotifyModelError() {
    if (connection_ == nullptr) {
        return;
    }

    quasar::proto::Remoting remoting;
    remoting.set_remote_object_id(TString(REMOTE_OBJECT_NAME));

    auto method = remoting.mutable_spotter_capability_listener_method();
    method->set_method(quasar::proto::Remoting::SpotterCapabilityListenerMethod::ON_MODEL_ERROR);
    for (const auto& spotterType : {SpotterTypes::ACTIVATION, SpotterTypes::INTERRUPTION, SpotterTypes::ADDITIONAL}) {
        method->add_spotter_type(spotterType);
    }

    connection_->sendMessage(remoting);
}

void ActivationSpotterCapability::remoteNotifyModelSet() {
    if (connection_ == nullptr) {
        return;
    }

    quasar::proto::Remoting remoting;
    remoting.set_remote_object_id(TString(REMOTE_OBJECT_NAME));

    auto method = remoting.mutable_spotter_capability_listener_method();
    method->set_method(quasar::proto::Remoting::SpotterCapabilityListenerMethod::ON_MODEL_SET);
    for (const auto& spotterType : {SpotterTypes::ACTIVATION, SpotterTypes::INTERRUPTION, SpotterTypes::ADDITIONAL}) {
        method->add_spotter_type(TString(spotterType));
    }

    connection_->sendMessage(remoting);
}

void ActivationSpotterCapability::processSetModelPaths(
    const google::protobuf::RepeatedPtrField<quasar::proto::Remoting::SpotterCapabilityMethod::ModelPath>& modelPaths)
{
    typeToModelPaths_.clear();
    for (const auto& modelPath : modelPaths) {
        typeToModelPaths_[modelPath.type()] = modelPath.path();
    }

    YIO_LOG_INFO("New spotter config: " << modelsToString(typeToModelPaths_) << " Updating voice dialog settings");

    updateVoiceDialogSettings();

    if (pendingStartSpotter_ && isConfigured()) {
        YIO_LOG_INFO("Start pending spotter");
        pendingStartSpotter_ = false;
        voiceDialog_->startPhraseSpotter();
    }

    remoteNotifyModelSet();
}

void ActivationSpotterCapability::processSetSpotterWord(const std::string& spotterWord) {
    deviceState_.setSpotterWord(spotterWord);
}
