#include "navi_old_spotter_capability.h"

#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/spotter_types/spotter_types.h>
#include <yandex_io/sdk/interfaces/directive.h>

YIO_DEFINE_LOG_MODULE("navi_old_spotter_capability");

using namespace quasar;

namespace {

    constexpr auto REMOTE_OBJECT_NAME = "NaviOldSpotterCapability";

} // namespace

NaviOldSpotterCapability::NaviOldSpotterCapability(
    std::shared_ptr<YandexIO::IDevice> device,
    std::shared_ptr<SpeechKit::AudioSource> audioSource,
    std::weak_ptr<YandexIO::IDirectiveProcessor> directiveProcessor,
    const AliceConfig& aliceConfig,
    AliceDeviceState& aliceDeviceState,
    std::shared_ptr<quasar::ICallbackQueue> worker,
    std::weak_ptr<YandexIO::IRemotingRegistry> remotingRegistry)
    : YandexIO::IRemoteObject(std::move(remotingRegistry))
    , device_(std::move(device))
    , audioSource_(std::move(audioSource))
    , aliceConfig_(aliceConfig)
    , aliceDeviceState_(aliceDeviceState)
    , directiveProcessor_(std::move(directiveProcessor))
    , worker_(std::move(worker))
{
    recognizedPhraseToDirectiveMap_ = {
        {"дальше", "go_forward"},
        {"назад", "go_backward"},
        {"вверх", "go_up"},
        {"выше", "go_up"},
        {"вниз", "go_down"},
        {"ниже", "go_down"}};
}

NaviOldSpotterCapability::~NaviOldSpotterCapability() {
    if (auto remotingRegistry = getRemotingRegistry().lock()) {
        remotingRegistry->removeRemoteObject(REMOTE_OBJECT_NAME);
    }
}

void NaviOldSpotterCapability::init() {
    if (auto remotingRegistry = getRemotingRegistry().lock()) {
        remotingRegistry->addRemoteObject(REMOTE_OBJECT_NAME, weak_from_this());
    }
}

void NaviOldSpotterCapability::resume()
{
    if (state_ == State::SUSPENDED) {
        start();
    }
}

void NaviOldSpotterCapability::onCapabilityStateChanged(const std::shared_ptr<YandexIO::ICapability>& /*capability*/, const NAlice::TCapabilityHolder& /*state*/) {
    if (aliceDeviceState_.getVideoState().HasCurrentScreen()) {
        if (aliceConfig_.isLongListeningEnabled() && aliceDeviceState_.isLongListenerScreen()) {
            start();
        } else {
            stop();
        }
    }
}

void NaviOldSpotterCapability::onCapabilityEvents(const std::shared_ptr<YandexIO::ICapability>& /*capability*/, const std::vector<NAlice::TCapabilityEvent>& /*events*/) {
    // ¯\_(ツ)_/¯
}

void NaviOldSpotterCapability::logSpotterCommand(
    const std::string& command, std::string_view spotterType, const std::string& spotterModel)
{
    if (!aliceConfig_.getSendNavigationSpotterLog() && spotterType == SpotterTypes::NAVIGATION_OLD) {
        YIO_LOG_DEBUG("Skip logging navigation spotter");
        return;
    }

    if (!aliceConfig_.getSendCommandSpotterLog()) {
        YIO_LOG_DEBUG("Skip logging command spotter");
        return;
    }

    const auto eventType = (spotterType == SpotterTypes::NAVIGATION_OLD) ? "navigationCommand" : "spotterCommand";

    Json::Value args;
    args["device_state"] = aliceDeviceState_.formatJson();
    args["environment_state"] = aliceDeviceState_.getEnvironmentState().formatJson();
    args["spotter_phrase"] = command;
    args["spotter_model"] = spotterModel;
    device_->telemetry()->reportEvent(eventType, jsonToString(args));
}

void NaviOldSpotterCapability::start()
{
    if (state_ == State::STARTED) {
        return;
    }
    state_ = State::STARTED;

    if (modelPath_.empty()) {
        YIO_LOG_INFO("Resetting long dialog spotter due to empty model path");
        phraseSpotter_.reset();
        return;
    }

    YIO_LOG_INFO("start");
    device_->telemetry()->reportEvent("longDialogStart");

    auto settings = ::SpeechKit::PhraseSpotterSettings{""};
    aliceConfig_.setSpotterLoggingSettings(settings);
    settings_.context = SpotterTypes::NAVIGATION_OLD;
    settings_.modelPath = modelPath_;

    phraseSpotter_ = ::SpeechKit::PhraseSpotter::create(settings_, shared_from_this(), audioSource_);
    phraseSpotter_->prepare();
    phraseSpotter_->start();
}

void NaviOldSpotterCapability::stop() {
    if (phraseSpotter_ == nullptr || state_ == State::STOPPED) {
        return;
    }
    YIO_LOG_INFO("stop");

    device_->telemetry()->reportEvent("longDialogStop");

    phraseSpotter_->stop();
    phraseSpotter_.reset();

    state_ = State::STOPPED;
}

void NaviOldSpotterCapability::restart()
{
    if (state_ == State::STARTED) {
        stop();
        start();
    }
}

void NaviOldSpotterCapability::suspend()
{
    if (state_ == State::STARTED) {
        stop();
        state_ = State::SUSPENDED;
    }
}

void NaviOldSpotterCapability::remoteNotifyModelSet() {
    if (connection_ == nullptr) {
        return;
    }

    quasar::proto::Remoting remoting;
    remoting.set_remote_object_id(TString(REMOTE_OBJECT_NAME));

    auto method = remoting.mutable_spotter_capability_listener_method();
    method->set_method(quasar::proto::Remoting::SpotterCapabilityListenerMethod::ON_MODEL_SET);
    method->add_spotter_type(TString(SpotterTypes::NAVIGATION_OLD));

    connection_->sendMessage(remoting);
}

void NaviOldSpotterCapability::remoteNotifyModelError() {
    if (connection_ == nullptr) {
        return;
    }

    quasar::proto::Remoting remoting;
    remoting.set_remote_object_id(TString(REMOTE_OBJECT_NAME));

    auto method = remoting.mutable_spotter_capability_listener_method();
    method->set_method(quasar::proto::Remoting::SpotterCapabilityListenerMethod::ON_MODEL_ERROR);
    method->add_spotter_type(TString(SpotterTypes::NAVIGATION_OLD));

    connection_->sendMessage(remoting);
}

void NaviOldSpotterCapability::onPhraseSpotted(
    SpeechKit::PhraseSpotter::SharedPtr /*phraseSpotter*/,
    const std::string& phrase,
    int /* phraseId */,
    const std::string& /* channelName */)
{
    worker_->add([this, phrase]() {
        if (aliceState_.state() != proto::AliceState::IDLE) {
            return;
        }

        auto iter = recognizedPhraseToDirectiveMap_.find(phrase);
        if (iter == recognizedPhraseToDirectiveMap_.end()) {
            return;
        }

        const std::string& command = iter->second;
        YIO_LOG_INFO("onPhraseSpotted: phrase=" << phrase << ", command=" << command);

        Json::Value event;
        event["spotted_phrase"] = command;
        device_->telemetry()->reportEvent("longDialogCommand", jsonToString(event));

        Json::Value phraseJson;
        phraseJson["phrase"] = phrase;
        device_->telemetry()->reportEvent("longDialogPhraseSpotted", jsonToString(phraseJson));

        logSpotterCommand(command, SpotterTypes::NAVIGATION_OLD, settings_.getModel());

        auto directive = YandexIO::Directive::createClientAction(command);
        if (auto directiveProcessor = directiveProcessor_.lock()) {
            directiveProcessor->addDirectives({std::move(directive)});
        }
    });
}

void NaviOldSpotterCapability::onPhraseSpotterStarted(
    SpeechKit::PhraseSpotter::SharedPtr /*phraseSpotter*/)
{
    worker_->add([this]() {
        device_->telemetry()->reportEvent("longDialogPhraseSpotterStarted");
    });
}

void NaviOldSpotterCapability::onPhraseSpotterError(
    SpeechKit::PhraseSpotter::SharedPtr /*phraseSpotter*/, const SpeechKit::Error& error)
{
    YIO_LOG_ERROR_EVENT("longDialogPhraseSpotterError", error.getString());

    worker_->add([this, error]() {
        Json::Value errorJson;
        errorJson["code"] = error.getCode();
        device_->telemetry()->reportEvent("longDialogPhraseSpotterError", jsonToString(errorJson));

        if (error.isSpotterModelError()) {
            modelPath_.clear();
            restart();

            remoteNotifyModelError();
        }
    });
}

void NaviOldSpotterCapability::onAliceStateChanged(proto::AliceState state)
{
    if (state_ == State::STARTED &&
        state.state() == proto::AliceState::LISTENING) {
        suspend();
    }

    aliceState_ = std::move(state);
}

void NaviOldSpotterCapability::onAliceTtsCompleted()
{
}

void NaviOldSpotterCapability::handleRemotingMessage(
    const quasar::proto::Remoting& message,
    std::shared_ptr<YandexIO::IRemotingConnection> connection)
{
    if (!message.has_spotter_capability_method()) {
        return;
    }

    const auto& method = message.spotter_capability_method();

    if (method.method() != quasar::proto::Remoting::SpotterCapabilityMethod::SET_MODEL_PATHS) {
        return;
    }

    worker_->add([this, method, connection{std::move(connection)}]() mutable {
        connection_ = std::move(connection);

        modelPath_.clear();

        for (const auto& modelPath : method.model_path()) {
            if (modelPath.type() == SpotterTypes::NAVIGATION_OLD) {
                modelPath_ = modelPath.path();
                restart();
                remoteNotifyModelSet();
                return;
            }
        }
    });
}
