#include "stereo_pair_preprocessor.h"

#include "stereo_pair_directives.h"

#include <yandex_io/libs/base/directives.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>

using namespace YandexIO;
using namespace quasar;

YIO_DEFINE_LOG_MODULE("stereo_pair_preprocessor");

StereoPairPreprocessor::StereoPairPreprocessor(
    std::shared_ptr<IStereoPairProvider> stereoPairProvider,
    std::shared_ptr<IUserConfigProvider> userConfigProvider)
    : followerProhibitedDirectives_{
          Directives::AUDIO_PLAY,
          Directives::MUSIC_PLAY,
          Directives::RADIO_PLAY,
          Directives::VIDEO_PLAY,
          Directives::SET_TIMER,
          Directives::RESUME_TIMER,
          Directives::START_BLUETOOTH,
          Directives::START_MUSIC_RECOGNIZER,
          Directives::START_MULTIROOM,
          Directives::BLUETOOTH_PLAYER_PLAY}
    , stereoPairProvider_(std::move(stereoPairProvider))
    , userConfigProvider_(std::move(userConfigProvider))
{
    if (userConfigProvider_) {
        userConfigProvider_->jsonChangedSignal(IUserConfigProvider::ConfigScope::SYSTEM, "stereo_pair").connect([this](const auto& jsonPtr) {
            ignorePartnerReady_ = tryGetBool(*jsonPtr, "ignore_partner_ready", false);
        }, lifetime_);
    }
}

StereoPairPreprocessor::~StereoPairPreprocessor() {
    lifetime_.die();
}

const std::string& StereoPairPreprocessor::getPreprocessorName() const {
    static const std::string s_name = "StereoPairPreprocessor";
    return s_name;
}

void StereoPairPreprocessor::preprocessDirectives(std::list<std::shared_ptr<Directive>>& directives)
{
    if (!stereoPairProvider_) {
        return;
    }

    std::shared_ptr<const StereoPairState> state;
    for (auto& directive : directives) {
        if (!state) {
            state = stereoPairProvider_->stereoPairState().value();
            if (!state || !state->isStereoPair()) {
                return;
            }
        }

        if (state->isFollower()) {
            if (followerProhibitedDirectives_.count(directive->getData().name)) {
                YIO_LOG_ERROR_EVENT("StereoPairPreprocessor.ProhibitedDirective", "Execution of directive \"" << directive->getData().name << "\" on the follower device is prohibited");
                directives.clear();
                return;
            }
        } else if (directive->is(Directives::AUDIO_PLAY) || directive->is(Directives::MUSIC_PLAY)) {
            if (ignorePartnerReady_ && state->stereoPlayerStatus != StereoPairState::StereoPlayerStatus::READY) {
                YIO_LOG_INFO("Stereo pair follower is not ready to play music but \"ignore_partner_ready\" is true. Execute directive \"" << directive->getData().name << "\" anyway.");
            } else if (state->connectivity == StereoPairState::Connectivity::NO_CONNECTION) {
                YIO_LOG_INFO("No connection to follower. Music play canceled. Abort directive " << directive->getData().name);
                directives.clear();
                directives.push_back(YandexIO::Directive::createLocalAction(StereoPairDirectives::NO_CONNECTION_NOTIFICATION));
                return;
            } else if (state->stereoPlayerStatus != StereoPairState::StereoPlayerStatus::READY) {
                YIO_LOG_INFO("Stereo pair follower is not ready to play music. Abort directive " << directive->getData().name);
                directives.clear();
                directives.push_back(YandexIO::Directive::createLocalAction(StereoPairDirectives::PLAYER_NOT_READY_NOTIFICATION));
                return;
            } else {
                YIO_LOG_DEBUG("Stereo pair allow to execute directive " << directive->getData().name);
            }
        }
    }
}
