#include "tandem_capability.h"

#include <yandex_io/interfaces/auth/connector/auth_provider.h>
#include <yandex_io/interfaces/device_state/connector/device_state_provider.h>
#include <yandex_io/libs/base/directives.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/protobuf_utils/proto_trace.h>
#include <yandex_io/libs/base/utils.h>

#include <yandex_io/capabilities/device_state/converters/converters.h>

#include <util/generic/yexception.h>

#include <algorithm>
#include <array>
#include <functional>
#include <memory>
#include <mutex>
#include <utility>
#include <vector>

YIO_DEFINE_LOG_MODULE("tandem_capability");

using namespace quasar;
using namespace YandexIO;

const std::string TandemCapability::LEADER = "leader";

const std::set<std::string> TandemCapability::MEDIA_COMMANDS{
    Directives::AUDIO_REWIND,
    Directives::AUDIO_PLAY,
    Directives::MUSIC_PLAY,
    Directives::RADIO_PLAY,
    Directives::PLAYER_PAUSE,
    Directives::PLAYER_CONTINUE,
    Directives::PLAYER_NEXT_TRACK,
    Directives::PLAYER_PREVIOUS_TRACK,
    Directives::PLAYER_REPLAY,
    Directives::PLAYER_REWIND,
    Directives::VIDEO_PLAY,
    Directives::NEXT_EPISODE_ANNOUNCE,
    Directives::SHOW_GALLERY,
    Directives::SHOW_SEASON_GALLERY,
    Directives::SHOW_TV_GALLERY,
    Directives::SHOW_DESCRIPTION,
    Directives::SHOW_PAY_PUSH_SCREEN,
    Directives::GO_HOME,
    Directives::MORDOVIA_SHOW,
    Directives::MORDOVIA_COMMAND,
};

namespace {
    quasar::FeaturesConfig getFeaturesConfig(const NAlice::TDeviceStateCapability::TState& state) {
        return quasar::FeaturesConfig(state);
    }

    bool getRadioPause(const NAlice::TDeviceState& state) {
        if (state.HasRadio() && state.GetRadio().fields().contains("player")) {
            const auto& playerState = state.GetRadio().fields().at("player");
            if (playerState.has_struct_value() && playerState.struct_value().fields().contains("pause")) {
                const auto& pause = playerState.struct_value().fields().at("pause");
                return pause.has_bool_value() && pause.bool_value();
            }
        }
        return false;
    }

} // unnamed namespace

TandemCapability::TandemCapability(
    std::shared_ptr<YandexIO::IDevice> device,
    std::shared_ptr<ipc::IIpcFactory> ipcFactory,
    std::shared_ptr<IAuthProvider> authProvider,
    std::shared_ptr<quasar::ICallbackQueue> worker,
    std::weak_ptr<IDirectiveProcessor> directiveProcessor,
    std::shared_ptr<quasar::AliceDeviceState> deviceState,
    std::shared_ptr<IDeviceStateCapability> deviceStateCapability,
    std::shared_ptr<IPlaybackControlCapability> playbackControlCapability,
    std::weak_ptr<YandexIO::IEndpointStorage> endpointStorage)
    : device_(std::move(device))
    , authProvider_(std::move(authProvider))
    , server_(ipcFactory->createIpcServer("pilotd"))
    , deviceContext_(ipcFactory, [this]() {
        worker_->add([this]() {
            broadcastDeviceGroupState();
        });
    }, false)
    , worker_(std::move(worker))
    , deviceState_(std::move(deviceState))
    , directiveProcessor_(directiveProcessor)
    , endpointStorage_(std::move(endpointStorage))
    , tandemStateHandler_(std::move(directiveProcessor), deviceState_)
    , localDeviceState_(std::move(deviceStateCapability))
    , playbackControlCapability_(std::move(playbackControlCapability))
{
    groupState_.set_media_state(proto::DeviceGroupState::MediaState::DeviceGroupState_MediaState_UNKNOWN);
    groupState_.set_local_role(proto::DeviceGroupState::Role::DeviceGroupState_Role_STAND_ALONE);

    server_->setClientConnectedHandler([this](auto& connection) {
        worker_->add([this, connection = connection.share()]() {
            broadcastDeviceGroupState();
        });
    });

    server_->listenService();
    deviceContext_.connectToSDK();
}

TandemCapability::~TandemCapability() {
    server_->shutdown();
}

void TandemCapability::init()
{
    if (auto endpointStorage = endpointStorage_.lock()) {
        endpointStorage->addListener(weak_from_this());
    }
    tandemStateHandler_.init();
}

void TandemCapability::handleQuasarMessage(const ipc::SharedMessage& message)
{
    if (message->has_glagold_state()) { // from "glagold"
        glagoldState_ = message->glagold_state();
        updateGroupStateFromGlagolState();
        broadcastDeviceGroupState();
    } else if (message->has_leader_override()) { // from "glagold"
        const auto& leaderOverride = message->leader_override();
        if (leaderOverride.has_music_state_override() ||
            leaderOverride.has_radio_state_override()) {
            if (deviceState_->isMediaPlaying()) {
                YIO_LOG_INFO("pause playback by leaderOverride");
                playbackControlCapability_->pause();
            }
        }
    } else if (message->has_user_config_update()) { // from "syncd"
        const auto& userConfig = message->user_config_update();
        if (userConfig.has_group_config()) {
            tandemConfig_ = TandemConfig::parse(userConfig.group_config());
            updateTandemState();
        }

        updateAliceDeviceState();
    }
}

void TandemCapability::onDeviceStateChanged(std::shared_ptr<const DeviceState> deviceState)
{
    const auto prevConfigurationState = std::exchange(configurationState_, deviceState->configuration);
    if (prevConfigurationState != deviceState->configuration) {
        updateTandemState();
    }
}

const std::string& TandemCapability::getPreprocessorName() const {
    static const std::string s_name = "TandemCapabilityPreprocessor";
    return s_name;
}

void TandemCapability::preprocessDirectives(std::list<std::shared_ptr<Directive>>& directives)
{
    const std::string remoteEndpointId = tandemConfig_.getRemoteEndpointId(device_->deviceId());
    if (remoteEndpointId.empty()) {
        return;
    }

    for (auto iter = directives.begin(); iter != directives.end(); ++iter) {
        auto& directive = *iter;
        const auto& name = directive->getData().name;

        const auto handlerType = tandemStateHandler_.chooseExternalHandlerType(directive);
        if (handlerType == TandemStateHandler::HandlerType::FOLLOWER) {
            auto data = directive->getData();
            data.endpointId = remoteEndpointId;
            directive = std::make_shared<Directive>(std::move(data));

            if (MEDIA_COMMANDS.contains(name)) {
                YandexIO::Directive::Data data(quasar::Directives::GO_HOME, "local_action");
                data.isRouteLocally = true;
                iter = directives.insert(iter, std::make_shared<Directive>(std::move(data)));
                std::advance(iter, 1);
            }
        } else if (handlerType == TandemStateHandler::HandlerType::BROADCAST) {
            auto data = directive->getData();
            data.endpointId = remoteEndpointId;
            auto remoteDirective = std::make_shared<Directive>(std::move(data));

            iter = directives.insert(iter, std::move(remoteDirective));
            std::advance(iter, 1);
        }

        if (MEDIA_COMMANDS.contains(name)) {
            Directive::Data data("tandem_update_media_directive", "client_action");
            if (handlerType == TandemStateHandler::HandlerType::LOCAL) {
                data.payload["sendPlayerOverride"] = true;
                data.payload["mediaState"] = (int)proto::DeviceGroupState::MediaState::DeviceGroupState_MediaState_PLAYING_MUSIC_LOCAL;
                data.payload["pauseConnectedFollower"] = name != Directives::PLAYER_PAUSE;
            } else {
                data.payload["sendPlayerOverride"] = false;
                data.payload["mediaState"] = (int)proto::DeviceGroupState::MediaState::DeviceGroupState_MediaState_PLAYING_MEDIA_TARGET;
                data.payload["pauseConnectedFollower"] = false;
            }

            iter = directives.insert(iter, std::make_shared<Directive>(std::move(data)));
            std::advance(iter, 1);
        }
    }
}

const std::string& TandemCapability::getHandlerName() const {
    static const std::string s_name = "TandemCapabilityHandler";
    return s_name;
}

const std::set<std::string>& TandemCapability::getSupportedDirectiveNames() const {
    static const std::set<std::string> s_names = {
        "tandem_update_media_directive",
    };

    return s_names;
}

void TandemCapability::handleDirective(const std::shared_ptr<Directive>& directive) {
    if (directive->is("tandem_update_media_directive")) {
        try {
            sendPlayerOverride_ = quasar::getBool(directive->getData().payload, "sendPlayerOverride");
            const int mediaState = quasar::getInt(directive->getData().payload, "mediaState");
            groupState_.set_media_state(static_cast<proto::DeviceGroupState::MediaState>(mediaState));
            tandemStateHandler_.setMediaKeerper(sendPlayerOverride_ ? TandemStateHandler::MediaKeeper::LOCAL : TandemStateHandler::MediaKeeper::FOLLOWER);
            broadcastDeviceGroupState();

            const bool pauseConnectedFollower = quasar::getBool(directive->getData().payload, "pauseConnectedFollower");
            if (pauseConnectedFollower && moduleHasRunningMedia()) {
                pauseFollowerMedia();
            }
        } catch (const std::exception& e) {
            YIO_LOG_ERROR_EVENT("TandemCapability.FailedToParsePayload", e.what());
        }
    }
}

void TandemCapability::onEndpointAdded(const std::shared_ptr<IEndpoint>& endpoint) {
    if (isRemoteEndpoint(endpoint)) {
        setRemoteEndpoint(endpoint);
        updateGroupState();
        updateAliceDeviceState();
    }
}

void TandemCapability::onEndpointRemoved(const std::shared_ptr<IEndpoint>& endpoint) {
    if (isRemoteEndpoint(endpoint)) {
        setRemoteEndpoint(nullptr);
        updateGroupState();
        updateAliceDeviceState();
    }
}

void TandemCapability::onEndpointStateChanged(const std::shared_ptr<IEndpoint>& endpoint) {
    Y_UNUSED(endpoint);
}

void TandemCapability::onCapabilityAdded(
    const std::shared_ptr<IEndpoint>& endpoint, const std::shared_ptr<ICapability>& capability) {
    if (isRemoteEndpoint(endpoint) && capability->getId() == localDeviceState_->getId()) {
        setRemoteDeviceStateCapability(capability);
        updateAliceDeviceState();
    }
}

void TandemCapability::onCapabilityRemoved(
    const std::shared_ptr<IEndpoint>& endpoint, const std::shared_ptr<ICapability>& capability) {
    if (isRemoteEndpoint(endpoint) && capability->getId() == localDeviceState_->getId()) {
        setRemoteDeviceStateCapability(nullptr);
        updateAliceDeviceState();
    }
}

void TandemCapability::onCapabilityStateChanged(
    const std::shared_ptr<ICapability>& capability, const NAlice::TCapabilityHolder& /*state*/) {
    if (capability == localDeviceState_ ||
        capability == remoteDeviceState_) {
        updateAliceDeviceState();
    }
}

void TandemCapability::onCapabilityEvents(const std::shared_ptr<ICapability>& capability, const std::vector<NAlice::TCapabilityEvent>& events) {
    Y_UNUSED(capability);
    Y_UNUSED(events);
}

void TandemCapability::updateGroupStateFromGlagolState() {
    if (glagoldState_.has_microphone_group_state()) {
        groupState_.mutable_group_alice_state()->set_microphone_group_state(glagoldState_.microphone_group_state());
    }

    if (groupState_.local_role() == proto::DeviceGroupState_Role::DeviceGroupState_Role_FOLLOWER) {
        for (const auto& connection : glagoldState_.connections()) {
            if (connection.has_type() && connection.type() == proto::GlagoldState::YANDEXIO_DEVICE) {
                groupState_.mutable_leader()->set_connection_state(proto::DeviceGroupState::CONNECTED);
            }
        }
    }
}

void TandemCapability::updateGroupStateFollowerAppState() {
    if (groupState_.local_role() == proto::DeviceGroupState_Role::DeviceGroupState_Role_LEADER &&
        groupState_.has_follower() &&
        remoteEndpoint_ != nullptr) {
        proto::AppState followerAppState;
        groupState_.mutable_follower()->mutable_app_state()->CopyFrom(followerAppState);
    }
}

void TandemCapability::pauseFollowerMedia() {
    if (auto moduleDirectiveHandler = getRemoteDirectiveHandler()) {
        Directive::Data data(Directives::PLAYER_PAUSE, "client_action");
        auto directive = std::make_shared<Directive>(std::move(data));

        moduleDirectiveHandler->handleDirective(directive);
    }
}

NAlice::TDeviceState TandemCapability::getMergedDeviceState() const {
    auto result = localDeviceState_->getState().GetDeviceStateCapability().GetState().GetDeviceState();
    const auto remoteState = remoteDeviceState_->getState().GetDeviceStateCapability().GetState().GetDeviceState();
    if (remoteState.HasRcuState()) {
        result.MutableRcuState()->CopyFrom(remoteState.GetRcuState());
    }
    if (remoteState.HasScreen()) {
        result.MutableScreen()->CopyFrom(remoteState.GetScreen());
    }
    if (remoteState.HasPackagesState()) {
        result.MutablePackagesState()->CopyFrom(remoteState.GetPackagesState());
    }
    if (remoteState.HasVideo()) {
        result.MutableVideo()->CopyFrom(remoteState.GetVideo());
    }
    auto musicState = getEffectiveMusicState(remoteState, result);
    if (musicState) {
        result.MutableMusic()->Swap(&*musicState);
    } else {
        result.ClearMusic();
    }
    auto radioState = getEffectiveRadioState(remoteState, result);
    if (radioState) {
        result.MutableRadio()->Swap(&*radioState);
    } else {
        result.ClearRadio();
    }
    auto lastWatched = getEffectiveLastWatched(remoteState, result);
    result.MutableLastWatched()->Swap(&lastWatched);
    return result;
}

NAlice::TDeviceStateCapability::TState TandemCapability::getTandemState() const {
    NAlice::TDeviceStateCapability::TState result;
    const auto features = getMergedFeaturesConfig();
    result.MutableSupportedFeatures()->Assign(features.getSupportedFeatures().begin(), features.getSupportedFeatures().end());
    result.MutableUnsupportedFeatures()->Assign(features.getUnsupportedFeatures().begin(), features.getUnsupportedFeatures().end());
    result.MutableExperiments()->CopyFrom(features.getExperiments());
    auto state = getMergedDeviceState();
    result.MutableDeviceState()->Swap(&state);
    return result;
}

quasar::FeaturesConfig TandemCapability::getMergedFeaturesConfig() const {
    Y_VERIFY(remoteDeviceState_ != nullptr);

    auto mergedFeaturesConfig = getFeaturesConfig(localDeviceState_->getState().GetDeviceStateCapability().GetState());
    mergedFeaturesConfig.merge(getFeaturesConfig(remoteDeviceState_->getState().GetDeviceStateCapability().GetState()));

    return mergedFeaturesConfig;
}

std::optional<NAlice::TDeviceState::TMusic> TandemCapability::getEffectiveMusicState(const NAlice::TDeviceState& followerState, const NAlice::TDeviceState& localState) {
    const bool moduleHasRunningMusic = followerState.HasMusic() && !followerState.GetMusic().GetPlayer().GetPause();
    const bool speakerHasRunningMusic = localState.HasMusic() && !localState.GetMusic().GetPlayer().GetPause();
    if (!moduleHasRunningMusic || speakerHasRunningMusic) {
        if (localState.HasMusic()) {
            return localState.GetMusic();
        } else {
            return std::nullopt;
        }
    }
    return followerState.GetMusic();
}

std::optional<google::protobuf::Struct> TandemCapability::getEffectiveRadioState(const NAlice::TDeviceState& followerState, const NAlice::TDeviceState& localState) {
    const auto moduleHasRunningRadio = getRadioPause(followerState);
    const auto speakerHasRunningRadio = getRadioPause(localState);
    if (!moduleHasRunningRadio || speakerHasRunningRadio) {
        if (localState.HasRadio()) {
            return localState.GetRadio();
        } else {
            return std::nullopt;
        }
    }

    return followerState.GetRadio();
}

NAlice::TDeviceState::TLastWatched TandemCapability::getEffectiveLastWatched(const NAlice::TDeviceState& followerState, const NAlice::TDeviceState& localState) {
    NAlice::TDeviceState::TLastWatched result = localState.GetLastWatched();
    const auto& videos = followerState.GetLastWatched().GetRawVideos();
    for (const auto& video : videos) {
        result.AddRawVideos()->CopyFrom(video);
    }
    const auto& episodes = followerState.GetLastWatched().GetRawTvShows();
    for (const auto& episode : episodes) {
        result.AddRawTvShows()->CopyFrom(episode);
    }
    const auto& movies = followerState.GetLastWatched().GetRawMovies();
    for (const auto& movie : movies) {
        result.AddRawMovies()->CopyFrom(movie);
    }
    return result;
}

void TandemCapability::broadcastDeviceGroupState() {
    deviceState_->setDeviceGroupState(groupState_);
    deviceState_->getEnvironmentState().updateTandemGroup(groupState_);
    deviceContext_.fireDeviceGroupStateChanged(groupState_);

    auto message = ipc::buildMessage([&](auto& msg) {
        msg.mutable_device_group_state()->CopyFrom(groupState_);
    });
    server_->sendToAll(message);

    tandemStateHandler_.handleDeviceGroupState(groupState_);
}

void TandemCapability::updateAliceDeviceState() {
    if (remoteEndpoint_ != nullptr) {
        deviceState_->getEnvironmentState().updateRemoteDevice(remoteDeviceState_->getState().GetDeviceStateCapability().GetState().GetEnvironmentDeviceInfo());
        deviceState_->setTandemState(getTandemState());
    }
}

bool TandemCapability::moduleHasRunningMedia() const {
    if (remoteEndpoint_ != nullptr) {
        const auto& state = remoteDeviceState_->getState().GetDeviceStateCapability().GetState().GetDeviceState();
        if (state.HasMusic() && !state.GetMusic().GetPlayer().GetPause()) {
            return true;
        }
        if (!getRadioPause(state)) {
            return true;
        }
        if (state.HasVideo() && state.GetVideo().HasPlayer() && !state.GetVideo().GetPlayer().GetPause()) {
            return true;
        }
        if (state.HasAudioPlayer() && state.GetAudioPlayer().GetPlayerState() == NAlice::TDeviceState::TAudioPlayer::Playing) {
            return true;
        }
    }

    return false;
}

void TandemCapability::setRoleToMetrica(proto::DeviceGroupState::Role role) const {
    static constexpr const char* deviceGroupRoleMetricaName = "device_group_role";
    static_assert(proto::DeviceGroupState::Role_ARRAYSIZE == 3, "New DeviceGroupRole added! Handle it");
    constexpr std::array<const char*, proto::DeviceGroupState::Role_ARRAYSIZE> rolesMap = {
        "stand_alone",
        "leader",
        "follower",
    };
    device_->telemetry()->putAppEnvironmentValue(deviceGroupRoleMetricaName, rolesMap[role]);
}

void TandemCapability::updateTandemState() {
    if (!authProvider_->ownerAuthInfo().value()->authToken.empty() &&
        configurationState_ != DeviceState::Configuration::CONFIGURING) {
        updateRemoteEndpoint();
    }

    updateGroupState();
}

void TandemCapability::updateGroupState() {
    auto localRole = tandemConfig_.getRole(device_->deviceId());
    if (configurationState_ == DeviceState::Configuration::CONFIGURING) {
        localRole = proto::DeviceGroupState_Role::DeviceGroupState_Role_STAND_ALONE;
    }

    groupState_.clear_media_state(); // ???
    groupState_.set_local_role(localRole);
    groupState_.set_play_to(TString(LEADER));
    groupState_.mutable_leader()->CopyFrom(tandemConfig_.getLeader());
    groupState_.mutable_follower()->CopyFrom(tandemConfig_.getFollower());

    // FIXME: use Endpoint::Status
    if (remoteEndpoint_) {
        groupState_.mutable_follower()->set_connection_state(quasar::proto::DeviceGroupState::CONNECTED);
    }

    updateGroupStateFromGlagolState();
    updateGroupStateFollowerAppState();

    broadcastDeviceGroupState();
}

void TandemCapability::updateRemoteEndpoint() {
    if (auto endpointStorage = endpointStorage_.lock()) {
        for (const auto& endpoint : endpointStorage->getEndpoints()) {
            if (isRemoteEndpoint(endpoint)) {
                setRemoteEndpoint(endpoint);
                return;
            }
        }
        setRemoteEndpoint(nullptr);
    }
}

bool TandemCapability::isRemoteEndpoint(const std::shared_ptr<IEndpoint>& endpoint) {
    return endpoint->getId() != "" && endpoint->getId() == tandemConfig_.getRemoteEndpointId(device_->deviceId());
}

void TandemCapability::setRemoteDeviceStateCapability(std::shared_ptr<ICapability> capability) {
    if (remoteDeviceState_ == capability) {
        return;
    }

    if (remoteDeviceState_) {
        remoteDeviceState_->removeListener(weak_from_this());
    }
    YIO_LOG_INFO("New remote DSC: " << (capability ? capability->getId() : "null"))
    remoteDeviceState_ = std::move(capability);
    tandemStateHandler_.setFollowerDeviceState(remoteDeviceState_);
    if (remoteDeviceState_) {
        remoteDeviceState_->addListener(weak_from_this());
        updateAliceDeviceState();
    }
}

void TandemCapability::setRemoteEndpoint(std::shared_ptr<IEndpoint> endpoint) {
    if (remoteEndpoint_ == endpoint) {
        // NOTE: There is no case when pointers are equal and endpointIds are not
        //       cause we hold targetEndpoint ptr and it cannot be reused
        return;
    }

    if (remoteEndpoint_ != nullptr) {
        remoteEndpoint_->removeListener(weak_from_this());
        setRemoteDeviceStateCapability(nullptr);
        if (auto directiveHandler = getRemoteDirectiveHandler()) {
            if (const auto dp = directiveProcessor_.lock()) {
                dp->removeDirectiveHandler(directiveHandler);
            }
        }
    }
    YIO_LOG_INFO("New remote endpoint: " << (endpoint ? endpoint->getId() : "null"));
    remoteEndpoint_ = std::move(endpoint);

    if (remoteEndpoint_ != nullptr) {
        remoteEndpoint_->addListener(weak_from_this());
        setRemoteDeviceStateCapability(remoteEndpoint_->findCapabilityById(localDeviceState_->getId()));
        if (auto directiveHandler = getRemoteDirectiveHandler()) {
            if (const auto dp = directiveProcessor_.lock(); dp && !dp->addDirectiveHandler(directiveHandler)) {
                YIO_LOG_ERROR_EVENT("TandemCapability.FailedToRegisterHandler", "Failed to register handler for id: " << directiveHandler->getEndpointId());
            }
        }
    }
}

std::shared_ptr<IDirectiveHandler> TandemCapability::getRemoteDirectiveHandler() const {
    if (!remoteEndpoint_) {
        return nullptr;
    }
    return remoteEndpoint_->getDirectiveHandler();
}

void TandemCapability::onAliceStateChanged(quasar::proto::AliceState state) {
    if (!remoteEndpoint_) {
        return;
    }
    try {
        const auto dh = remoteEndpoint_->getDirectiveHandler();
        Y_ENSURE(dh);
        const auto stateStr = state.SerializeAsStringOrThrow();
        Json::Value payload = Json::objectValue;
        payload["data"] = base64Encode(stateStr.c_str(), stateStr.size());
        Directive::Data data("alice_state_bypass", "client_action", std::move(payload));
        const auto directive = std::make_shared<Directive>(std::move(data));
        YIO_LOG_INFO("Sending alice state to endpoint handler");
        dh->handleDirective(directive);
    } catch (const std::exception& ex) {
        YIO_LOG_ERROR_EVENT("TamdemCapability.FailedToProvideAliceState", ex.what());
    }
}

void TandemCapability::onAliceTtsCompleted() {
}
