#pragma once

#include "tandem_config.h"
#include "tandem_state_handler.h"

#include <yandex_io/capabilities/device_state/interfaces/i_device_state_capability.h>
#include <yandex_io/capabilities/playback_control/interfaces/i_playback_control_capability.h>
#include <yandex_io/capabilities/alice/interfaces/i_alice_capability_listener.h>
#include <yandex_io/interfaces/auth/i_auth_provider.h>
#include <yandex_io/interfaces/device_state/i_device_state_provider.h>
#include <yandex_io/libs/configuration/features_config.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/ipc/i_ipc_factory.h>
#include <yandex_io/protos/model_objects.pb.h>
#include <yandex_io/sdk/interfaces/i_endpoint_storage.h>
#include <yandex_io/sdk/interfaces/i_directive_handler.h>
#include <yandex_io/sdk/private/device_context.h>
#include <yandex_io/services/aliced/device_state/alice_device_state.h>
#include <yandex_io/services/aliced/directive_processor/interface/i_directive_processor.h>
#include <yandex_io/services/aliced/directive_processor/interface/i_directive_preprocessor.h>

#include <memory>
#include <set>
#include <string>

namespace YandexIO {

    class TandemCapability
        : public IDirectiveHandler,
          public IDirectivePreprocessor,
          public IEndpointStorage::IListener,
          public ICapability::IListener,
          public IEndpoint::IListener,
          public IAliceCapabilityListener,
          public std::enable_shared_from_this<TandemCapability> {
    public:
        TandemCapability(
            std::shared_ptr<YandexIO::IDevice> device,
            std::shared_ptr<quasar::ipc::IIpcFactory> ipcFactory,
            std::shared_ptr<quasar::IAuthProvider> authProvider,
            std::shared_ptr<quasar::ICallbackQueue> worker,
            std::weak_ptr<IDirectiveProcessor> directiveProcessor,
            std::shared_ptr<quasar::AliceDeviceState> deviceState,
            std::shared_ptr<IDeviceStateCapability> deviceStateCapability,
            std::shared_ptr<IPlaybackControlCapability> playbackControlCapability,
            std::weak_ptr<YandexIO::IEndpointStorage> endpointStorage);
        ~TandemCapability();

        void init();
        void handleQuasarMessage(const quasar::ipc::SharedMessage& message);
        void onDeviceStateChanged(std::shared_ptr<const quasar::DeviceState> deviceState);

    private:
        // IDirectivePreprocessor implementation
        const std::string& getPreprocessorName() const override;
        void preprocessDirectives(std::list<std::shared_ptr<Directive>>& directives) override;

        // IDirectiveHandler implementation
        const std::string& getHandlerName() const override;
        const std::set<std::string>& getSupportedDirectiveNames() const override;
        void handleDirective(const std::shared_ptr<Directive>& directive) override;

        // IEndpointStorage::IListener implementation
        void onEndpointAdded(const std::shared_ptr<IEndpoint>& endpoint) override;
        void onEndpointRemoved(const std::shared_ptr<IEndpoint>& endpoint) override;

        // IEndpoint::IListener
        void onCapabilityAdded(const std::shared_ptr<IEndpoint>& endpoint, const std::shared_ptr<ICapability>& capability) override;
        void onCapabilityRemoved(const std::shared_ptr<IEndpoint>& endpoint, const std::shared_ptr<ICapability>& capability) override;
        void onEndpointStateChanged(const std::shared_ptr<IEndpoint>& endpoint) override;

        // ICapability::IListener
        void onCapabilityStateChanged(const std::shared_ptr<ICapability>& capability, const NAlice::TCapabilityHolder& state) override;
        void onCapabilityEvents(const std::shared_ptr<ICapability>& capability, const std::vector<NAlice::TCapabilityEvent>& events) override;

        // IAliceCapabilityListener
        void onAliceStateChanged(quasar::proto::AliceState state) override;
        void onAliceTtsCompleted() override;

    private:
        void updateGroupState();
        void updateTandemState();
        void updateAliceDeviceState();
        void updateGroupStateFollowerAppState();
        void updateGroupStateFromGlagolState();
        void updateRemoteEndpoint();
        void setRemoteEndpoint(std::shared_ptr<IEndpoint> endpoint);
        void setRemoteDeviceStateCapability(std::shared_ptr<ICapability> capability);
        bool isRemoteEndpoint(const std::shared_ptr<IEndpoint>& endpoint);

        bool moduleHasRunningMedia() const;
        NAlice::TDeviceStateCapability::TState getTandemState() const;
        NAlice::TDeviceState getMergedDeviceState() const;
        quasar::FeaturesConfig getMergedFeaturesConfig() const;
        std::shared_ptr<IDirectiveHandler> getRemoteDirectiveHandler() const;
        void setRoleToMetrica(quasar::proto::DeviceGroupState::Role role) const;

        void pauseFollowerMedia();
        void broadcastDeviceGroupState();

    private:
        static std::optional<NAlice::TDeviceState::TMusic> getEffectiveMusicState(const NAlice::TDeviceState& followerState, const NAlice::TDeviceState& localState);
        static std::optional<google::protobuf::Struct> getEffectiveRadioState(const NAlice::TDeviceState& followerState, const NAlice::TDeviceState& localState);
        static NAlice::TDeviceState::TLastWatched getEffectiveLastWatched(const NAlice::TDeviceState& followerState, const NAlice::TDeviceState& localState);

        static const std::string LEADER;
        static const std::set<std::string> MEDIA_COMMANDS;

    private:
        std::shared_ptr<YandexIO::IDevice> device_;
        std::shared_ptr<quasar::IAuthProvider> authProvider_;
        std::shared_ptr<quasar::ipc::IServer> server_;

        bool sendPlayerOverride_ = false;
        std::string savedGroupConfigStr_;
        quasar::proto::DeviceGroupState groupState_;
        quasar::proto::GlagoldState glagoldState_;
        quasar::DeviceState::Configuration configurationState_{quasar::DeviceState::Configuration::CONFIGURED};

        YandexIO::DeviceContext deviceContext_;
        std::shared_ptr<quasar::ICallbackQueue> worker_;
        const std::shared_ptr<quasar::AliceDeviceState> deviceState_;
        const std::weak_ptr<YandexIO::IDirectiveProcessor> directiveProcessor_;
        const std::weak_ptr<YandexIO::IEndpointStorage> endpointStorage_;
        TandemConfig tandemConfig_;
        TandemStateHandler tandemStateHandler_;
        std::shared_ptr<IEndpoint> remoteEndpoint_;
        std::shared_ptr<ICapability> remoteDeviceState_;
        std::shared_ptr<ICapability> localDeviceState_;
        const std::shared_ptr<IPlaybackControlCapability> playbackControlCapability_;
    };

} /* namespace YandexIO */
