#include "tandem_state_handler.h"

#include <yandex_io/libs/base/directives.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/protos/model_objects.pb.h>
#include <yandex_io/protos/quasar_proto.pb.h>
#include <yandex_io/protos/enum_names/enum_names.h>

using namespace YandexIO;

TandemStateHandler::TandemStateHandler(
    std::weak_ptr<IDirectiveProcessor> directiveProcessor,
    std::shared_ptr<quasar::AliceDeviceState> deviceState)
    : directiveProcessor_(std::move(directiveProcessor))
    , deviceState_(deviceState)
{
}

void TandemStateHandler::init() {
    // Local directives
    directiveHandlerMap_[quasar::Directives::SET_TIMER] = &executeLocal;
    directiveHandlerMap_[quasar::Directives::CANCEL_TIMER] = &executeLocal;
    directiveHandlerMap_[quasar::Directives::PAUSE_TIMER] = &executeLocal;
    directiveHandlerMap_[quasar::Directives::RESUME_TIMER] = &executeLocal;
    directiveHandlerMap_[quasar::Directives::ALARMS_UPDATE] = &executeLocal;
    directiveHandlerMap_[quasar::Directives::START_MULTIROOM] = &executeLocal;
    directiveHandlerMap_[quasar::Directives::STOP_MULTIROOM] = &executeLocal;
    directiveHandlerMap_[quasar::Directives::START_BLUETOOTH] = &executeLocal;
    directiveHandlerMap_[quasar::Directives::STOP_BLUETOOTH] = &executeLocal;
    directiveHandlerMap_[quasar::Directives::SOUND_SET_LEVEL] = &executeLocal;
    directiveHandlerMap_[quasar::Directives::ALARM_SET_SOUND] = &executeLocal;
    directiveHandlerMap_[quasar::Directives::ALARM_RESET_SOUND] = &executeLocal;
    directiveHandlerMap_[quasar::Directives::THEREMIN_PLAY] = &executeLocal;
    directiveHandlerMap_[quasar::Directives::MESSENGER_CALL] = &executeLocal;
    directiveHandlerMap_[quasar::Directives::DRAW_LED_SCREEN] = &executeLocal;
    directiveHandlerMap_[quasar::Directives::DRAW_SCLED_ANIMATION] = &executeLocal;
    directiveHandlerMap_[quasar::Directives::NOTIFY] = &executeLocal;
    directiveHandlerMap_[quasar::Directives::SUCCESS_STARTING_ONBOARDING] = &executeLocal;
    directiveHandlerMap_[quasar::Directives::IOT_DISCOVERY_START] = &executeLocal;
    directiveHandlerMap_[quasar::Directives::IOT_DISCOVERY_STOP] = &executeLocal;
    directiveHandlerMap_[quasar::Directives::IOT_DISCOVERY_CREDENTIALS] = &executeLocal;
    directiveHandlerMap_[quasar::Directives::SHOW_CLOCK] = &executeLocal;
    directiveHandlerMap_[quasar::Directives::HIDE_CLOCK] = &executeLocal;
    directiveHandlerMap_[quasar::Directives::RADIO_PLAY] = &executeLocal;
    directiveHandlerMap_[quasar::Directives::MUSIC_PLAY] = &executeLocal;
    directiveHandlerMap_[quasar::Directives::PLAYER_NEXT_TRACK] = &executeLocal;
    directiveHandlerMap_[quasar::Directives::PLAYER_PREVIOUS_TRACK] = &executeLocal;
    directiveHandlerMap_[quasar::Directives::PLAYER_LIKE] = &executeLocal;
    directiveHandlerMap_[quasar::Directives::PLAYER_DISLIKE] = &executeLocal;

    // Broadcast directives
    directiveHandlerMap_[quasar::Directives::SEND_BUG_REPORT] = &executeBroadcast;
    directiveHandlerMap_[quasar::Directives::SCREEN_ON] = &executeBroadcast;
    directiveHandlerMap_[quasar::Directives::SCREEN_OFF] = &executeBroadcast;
    directiveHandlerMap_[quasar::Directives::ONBOARDING_PLAY] = &executeBroadcast;
    directiveHandlerMap_[quasar::Directives::ONBOARDING_SKIP] = &executeBroadcast;
    directiveHandlerMap_[quasar::Directives::STOP_LEGACY_PLAYER] = &executeBroadcast;

    // Follower directives
    directiveHandlerMap_[quasar::Directives::VIDEO_PLAY] = &executeFollower;
    directiveHandlerMap_[quasar::Directives::NEXT_EPISODE_ANNOUNCE] = &executeFollower;
    directiveHandlerMap_[quasar::Directives::SHOW_GALLERY] = &executeFollower;
    directiveHandlerMap_[quasar::Directives::SHOW_SEASON_GALLERY] = &executeFollower;
    directiveHandlerMap_[quasar::Directives::SHOW_TV_GALLERY] = &executeFollower;
    directiveHandlerMap_[quasar::Directives::SHOW_DESCRIPTION] = &executeFollower;
    directiveHandlerMap_[quasar::Directives::SHOW_PAY_PUSH_SCREEN] = &executeFollower;
    directiveHandlerMap_[quasar::Directives::GO_HOME] = &executeFollower;
    directiveHandlerMap_[quasar::Directives::GO_FORWARD] = &executeFollower;
    directiveHandlerMap_[quasar::Directives::GO_BACKWARD] = &executeFollower;
    directiveHandlerMap_[quasar::Directives::GO_FORWARD] = &executeFollower;
    directiveHandlerMap_[quasar::Directives::GO_UP] = &executeFollower;
    directiveHandlerMap_[quasar::Directives::GO_DOWN] = &executeFollower;
    directiveHandlerMap_[quasar::Directives::GO_HOME] = &executeFollower;
    directiveHandlerMap_[quasar::Directives::GO_TO_THE_BEGINNING] = &executeFollower;
    directiveHandlerMap_[quasar::Directives::GO_TO_THE_END] = &executeFollower;
    directiveHandlerMap_[quasar::Directives::GO_TOP] = &executeFollower;
    directiveHandlerMap_[quasar::Directives::MORDOVIA_SHOW] = &executeFollower;
    directiveHandlerMap_[quasar::Directives::MORDOVIA_COMMAND] = &executeFollower;

    // Other
    directiveHandlerMap_[quasar::Directives::SOUND_QUITER] = std::bind(&TandemStateHandler::volumeCommands, this, std::placeholders::_1);
    directiveHandlerMap_[quasar::Directives::SOUND_LOUDER] = std::bind(&TandemStateHandler::volumeCommands, this, std::placeholders::_1);
    directiveHandlerMap_[quasar::Directives::SOUND_MUTE] = std::bind(&TandemStateHandler::volumeCommands, this, std::placeholders::_1);
    directiveHandlerMap_[quasar::Directives::SOUND_UNMUTE] = std::bind(&TandemStateHandler::volumeCommands, this, std::placeholders::_1);

    directiveHandlerMap_[quasar::Directives::PLAYER_REWIND] = std::bind(&TandemStateHandler::pauseCommands, this, std::placeholders::_1);
    directiveHandlerMap_[quasar::Directives::PLAYER_REPLAY] = std::bind(&TandemStateHandler::pauseCommands, this, std::placeholders::_1);
    directiveHandlerMap_[quasar::Directives::PLAYER_PAUSE] = std::bind(&TandemStateHandler::pauseCommands, this, std::placeholders::_1);

    directiveHandlerMap_[quasar::Directives::PLAYER_CONTINUE] = [](const std::shared_ptr<Directive>& directive) -> HandlerType {
        constexpr std::string_view VIDEO = "video";
        constexpr std::string_view CURRENT = "current";

        const auto player = quasar::tryGetString(directive->getData().payload, "player");

        if (player == VIDEO || player == CURRENT) {
            return HandlerType::FOLLOWER;
        }
        return HandlerType::LOCAL;
    };
}

TandemStateHandler::HandlerType TandemStateHandler::volumeCommands(const std::shared_ptr<Directive>& /*directive*/) {
    if (followerHasVideoScreen() && !deviceState_->isMediaPlaying()) {
        return HandlerType::FOLLOWER;
    }
    return HandlerType::LOCAL;
}

TandemStateHandler::HandlerType TandemStateHandler::pauseCommands(const std::shared_ptr<Directive>& /*directive*/) {
    if (deviceState_->isMediaPlaying()) {
        YIO_LOG_INFO("Handling pause directive as local: local media is playing");
        return HandlerType::LOCAL;
    } else {
        const auto isFollowerHasRunningMedia = followerHasPlayingVideo();
        if (isFollowerHasRunningMedia || currentMediaKeeper_ == MediaKeeper::FOLLOWER) {
            YIO_LOG_INFO("Handling pause directive as follower. isFollowerHasRunningMedia=" << isFollowerHasRunningMedia << ", currentMediaKeeper_=" << currentMediaKeeper_);
            return HandlerType::FOLLOWER;
        }
    }

    return HandlerType::LOCAL;
}

TandemStateHandler::HandlerType TandemStateHandler::executeLocal(const std::shared_ptr<Directive>& /*directive*/) {
    return TandemStateHandler::HandlerType::LOCAL;
}

TandemStateHandler::HandlerType TandemStateHandler::executeBroadcast(const std::shared_ptr<Directive>& /*directive*/) {
    return TandemStateHandler::HandlerType::BROADCAST;
}

TandemStateHandler::HandlerType TandemStateHandler::executeFollower(const std::shared_ptr<Directive>& /*directive*/) {
    return TandemStateHandler::HandlerType::FOLLOWER;
}

TandemStateHandler::HandlerType TandemStateHandler::chooseExternalHandlerType(const std::shared_ptr<Directive>& directive) {
    if (role_ == quasar::proto::DeviceGroupState::STAND_ALONE || !isFollowerConnected_) {
        return TandemStateHandler::HandlerType::LOCAL;
    }

    if (directive->getData().isRouteLocally) {
        return TandemStateHandler::HandlerType::LOCAL;
    }

    const auto& directiveName = directive->getData().name;
    if (!directiveHandlerMap_.contains(directiveName)) {
        if (auto directiveProcessor = directiveProcessor_.lock()) {
            if (directiveProcessor->getSupportedDirectiveNames().contains(directiveName)) {
                return TandemStateHandler::HandlerType::LOCAL;
            }
        }

        return TandemStateHandler::HandlerType::FOLLOWER;
    }

    return directiveHandlerMap_[directiveName](directive);
}

void TandemStateHandler::setFollowerDeviceState(std::shared_ptr<ICapability> deviceStateCapability) {
    deviceStateCapability_ = std::move(deviceStateCapability);
}

void TandemStateHandler::handleDeviceGroupState(const quasar::proto::DeviceGroupState& deviceGroupState) {
    if (deviceGroupState.has_local_role()) {
        role_ = deviceGroupState.local_role();
    }

    if (deviceGroupState.has_follower()) {
        const auto& follower = deviceGroupState.follower();
        if (follower.has_connection_state()) {
            const auto oldIsFollowerConnected = isFollowerConnected_;
            isFollowerConnected_ = follower.connection_state() == quasar::proto::DeviceGroupState::CONNECTED;
            if (oldIsFollowerConnected != isFollowerConnected_) {
                YIO_LOG_INFO("New connection state: " << isFollowerConnected_);
            }
        }
        if (currentMediaKeeper_ == MediaKeeper::LOCAL && followerHasPlayingVideo()) {
            if (deviceState_->isMediaPlaying()) {
                YandexIO::Directive::Data data;
                data.type = "local_action";
                data.isRouteLocally = true;
                if (deviceState_->isAudioClientPlaying()) {
                    data.name = quasar::Directives::AUDIO_STOP;
                } else {
                    data.name = quasar::Directives::PLAYER_PAUSE;
                }
                currentMediaKeeper_ = MediaKeeper::FOLLOWER;

                if (auto directiveProcessor = directiveProcessor_.lock()) {
                    directiveProcessor->addDirectives({std::make_shared<YandexIO::Directive>(std::move(data))});
                }
            }
        }
    }
}

void TandemStateHandler::setMediaKeerper(MediaKeeper keeper) {
    currentMediaKeeper_ = keeper;
}

bool TandemStateHandler::followerHasVideoScreen() const {
    if (!deviceStateCapability_) {
        return false;
    }
    const auto state = deviceStateCapability_->getState().GetDeviceStateCapability().GetState().GetDeviceState();
    return state.HasVideo() && state.GetVideo().HasCurrentScreen() && state.GetVideo().GetCurrentScreen() == "video_player";
}

bool TandemStateHandler::followerHasPlayingVideo() const {
    if (!deviceStateCapability_) {
        return false;
    }
    const auto state = deviceStateCapability_->getState().GetDeviceStateCapability().GetState().GetDeviceState();
    return state.HasVideo() &&
           state.GetVideo().HasPlayer() &&
           state.GetVideo().GetPlayer().HasPause() &&
           !state.GetVideo().GetPlayer().GetPause();
}

std::ostream& YandexIO::operator<<(std::ostream& out, TandemStateHandler::MediaKeeper type) {
    switch (type) {
        case TandemStateHandler::MediaKeeper::LOCAL: {
            out << "local";
            break;
        }
        case TandemStateHandler::MediaKeeper::FOLLOWER: {
            out << "follower";
            break;
        }
    }
    return out;
}
