#include "alice_device_state.h"

#include <alice/megamind/protos/guest/enrollment_headers.pb.h>

#include <yandex_io/services/aliced/alice_config/alice_config.h>

#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/protobuf_utils/json.h>
#include <yandex_io/libs/protobuf_utils/proto_trace.h>

#include <yandex_io/protos/capabilities/device_state_capability.pb.h>

YIO_DEFINE_LOG_MODULE("alice_device_state");

using namespace quasar;

const char* quasar::audioPlayerTypeName(AudioPlayerType audioPlayerType) {
    switch (audioPlayerType) {
        case AudioPlayerType::MUSIC:
            return "MUSIC";
        case AudioPlayerType::RADIO:
            return "RADIO";
        case AudioPlayerType::BLUETOOTH:
            return "BLUETOOTH";
        case AudioPlayerType::AUDIO_CLIENT:
            return "AUDIO_CLIENT";
        case AudioPlayerType::MULTIROOM:
            return "MULTIROOM";
        case AudioPlayerType::NONE:
            return "NONE";
    }
    return "UNDEFINED";
}

AliceDeviceState::AliceDeviceState(
    std::string deviceId,
    std::shared_ptr<IClockTowerProvider> clockTowerProvider,
    std::shared_ptr<IDeviceStateProvider> deviceStateProvider,
    EnvironmentStateHolder environmentState)
    : deviceId_(std::move(deviceId))
    , clockTowerProvider_(std::move(clockTowerProvider))
    , deviceStateProvider_(std::move(deviceStateProvider))
    , environmentState_(std::move(environmentState))
    , stereoPairState_(std::make_shared<StereoPairState>())
{
}

Json::Value AliceDeviceState::formatJson() const {
    Json::Value deviceState;

    deviceState["device_id"] = deviceId_;

    if (state_.GetDeviceState().HasMicsMuted()) {
        deviceState["mics_muted"] = state_.GetDeviceState().GetMicsMuted();
    }
    deviceState["smart_activation"] = smartActivation_;
    deviceState["dnd_enabled"] = dndEnabled_;

    if (state_.GetDeviceState().HasSoundLevel() && state_.GetDeviceState().HasSoundMuted()) {
        deviceState["sound_level"] = state_.GetDeviceState().GetSoundLevel();
        deviceState["sound_muted"] = state_.GetDeviceState().GetSoundMuted();
        if (state_.GetDeviceState().HasSoundMaxLevel()) {
            deviceState["sound_max_level"] = state_.GetDeviceState().GetSoundMaxLevel();
        }
    }

    if (state_.GetDeviceState().HasBattery()) {
        deviceState["battery"] = convertMessageToJson(state_.GetDeviceState().GetBattery()).value_or(Json::objectValue);
    }

    deviceState["device_config"] = formatDeviceConfig(contentSettings_, childContentSettings_, spotterWord_);

    const bool isFollowerConnected = getIsFollowerConnected();

    deviceState["is_tv_plugged_in"] = state_.GetDeviceState().GetIsTvPluggedIn() || isFollowerConnected;

    deviceState["last_watched"] = convertMessageToJson(getState().GetDeviceState().GetLastWatched()).value_or(Json::objectValue);

    if (getState().GetDeviceState().HasVideo()) {
        deviceState["video"] = convertMessageToJson(getVideoState()).value_or(Json::objectValue);
    }

    const Json::Value actions = formatActions(appState_);
    if (!actions.isNull()) {
        deviceState["actions"] = actions;
    }

    if (state_.GetDeviceState().HasMusic()) {
        deviceState["music"] = convertMessageToJson(state_.GetDeviceState().GetMusic()).value_or(Json::objectValue);
    }

    if (state_.GetDeviceState().HasRadio()) {
        deviceState["radio"] = convertMessageToJson(state_.GetDeviceState().GetRadio()).value_or(Json::objectValue);
    }

    if (state_.GetDeviceState().HasAudioPlayer()) {
        deviceState["audio_player"] = convertMessageToJson(state_.GetDeviceState().GetAudioPlayer()).value_or(Json::objectValue);
    }

    if (state_.GetDeviceState().HasMultiroom()) {
        deviceState["multiroom"] = quasar::convertMessageToJson(state_.GetDeviceState().GetMultiroom()).value_or(Json::objectValue);
    }
    if (getCurrentPlayerType() == AudioPlayerType::MULTIROOM) {
        slaveOverrideMediaState(deviceState);
    }

    Json::Value stereoPair = formatStereoPairState(stereoPairState_);
    if (!stereoPair.isNull()) {
        deviceState["stereo_pair"] = std::move(stereoPair);
    }

    deviceState["bluetooth"] = convertMessageToJson(state_.GetDeviceState().GetBluetooth()).value_or(Json::objectValue);

    if (getState().GetDeviceState().HasRcuState()) {
        deviceState["rcu"] = convertMessageToJson(getState().GetDeviceState().GetRcuState()).value_or(Json::objectValue);
    }

    if (getState().GetDeviceState().HasPackagesState()) {
        deviceState["packages"] = convertMessageToJson(getState().GetDeviceState().GetPackagesState()).value_or(Json::objectValue);
    }

    deviceState["timers"] = convertMessageToJson(state_.GetDeviceState().GetTimers()).value_or(Json::objectValue);
    deviceState["alarms_state"] = state_.GetDeviceState().GetAlarmsState();
    deviceState["alarm_state"] = convertMessageToJson(state_.GetDeviceState().GetAlarmState()).value_or(Json::objectValue);

    deviceState["active_actions"] = formatActiveActions(activeActions_);

    if (getState().GetDeviceState().HasScreen()) {
        deviceState["screen"] = convertMessageToJson(getState().GetDeviceState().GetScreen()).value_or(Json::objectValue);
    }

    const Json::Value calld = formatCalldState(calldState_);
    if (!calld.isNull()) {
        deviceState["messenger_call"] = calld;
    }

    const auto connectionType = formatInternetConnectionType(networkStatus_);
    if (!connectionType.empty()) {
        deviceState["internet_connection"]["type"] = connectionType;
    }

    const Json::Value neighbours = formatNetworkNeighbours();
    if (!neighbours.empty()) {
        deviceState["internet_connection"]["neighbours"] = neighbours;
    }

    if (networkStatus_.has_wifi_status() && networkStatus_.wifi_status().has_current_network()) {
        deviceState["internet_connection"]["current"]["ssid"] = networkStatus_.wifi_status().current_network().ssid();
        deviceState["internet_connection"]["current"]["bssid"] = networkStatus_.wifi_status().current_network().mac();
        deviceState["internet_connection"]["current"]["channel"] = networkStatus_.wifi_status().current_network().channel();
    }

    if (tvPolicyInfo_) {
        deviceState["device_config"] = formatDeviceConfig(tvPolicyInfo_.value());
    }

    if (clockDisplayStateOpt_) {
        const auto& state = *clockDisplayStateOpt_;
        deviceState["clock_display_state"]["clock_enabled"] = state.isClockOn;
        deviceState["clock_display_state"]["brightness"] = state.brightness;
    }

    auto clockTower = formatClockTower(clockTowerProvider_);
    if (!clockTower.isNull()) {
        deviceState["clocks_us"] = std::move(clockTower);
    }

    return deviceState;
}

Json::Value AliceDeviceState::formatExperiments() const {
    Json::Value res = Json::objectValue;
    if (!getState().HasExperiments()) {
        return res;
    }
    for (const auto& [key, val] : getState().GetExperiments().GetStorage()) {
        switch (val.GetValueCase()) {
            case NAlice::TExperimentsProto::TValue::ValueCase::kString:
                res[key] = val.GetString();
                break;
            case NAlice::TExperimentsProto::TValue::ValueCase::kBoolean:
                res[key] = val.GetBoolean();
                break;
            case NAlice::TExperimentsProto::TValue::ValueCase::kInteger:
                res[key] = val.GetInteger();
                break;
            case NAlice::TExperimentsProto::TValue::ValueCase::kNumber:
                res[key] = val.GetNumber();
                break;
            case NAlice::TExperimentsProto::TValue::ValueCase::VALUE_NOT_SET:
            default:
                break;
        }
    }
    return res;
}

Json::Value AliceDeviceState::formatEnrollmentHeaders() const {
    Json::Value result;

    if (state_.HasEnrollmentHeaders()) {
        result = convertMessageToJson(state_.GetEnrollmentHeaders(), true).value_or(Json::objectValue);
    }

    return result;
}

void AliceDeviceState::onConfigUpdate(const AliceConfig& config)
{
    setContentSettings(config.getContentAccess());
    setChildContentSettings(config.getChildContentAccess());
}

const proto::NetworkStatus& AliceDeviceState::getNetworkStatus() const {
    return networkStatus_;
}

const proto::WifiList& AliceDeviceState::getWifiList() const {
    return wifiList_;
}

bool AliceDeviceState::hasActivePlayer() const {
    if (const auto& audioPlayer = state_.GetDeviceState().GetAudioPlayer(); audioPlayer.HasPlayerState()) {
        return true;
    }

    if (const auto& musicState = state_.GetDeviceState().GetMusic(); musicState.GetCurrentlyPlaying().HasTrackId()) {
        return true;
    }

    if (const auto& radioState = state_.GetDeviceState().GetRadio(); radioState.fields().contains("currently_playing")) {
        const auto& currentlyPlaying = radioState.fields().at("currently_playing");
        return currentlyPlaying.has_struct_value() && currentlyPlaying.struct_value().fields().contains("radioId");
    }

    if (const auto& bluetoothState = state_.GetDeviceState().GetBluetooth(); bluetoothState.CurrentConnectionsSize() > 0 && bluetoothState.HasLastPlayTimestamp()) {
        return true;
    }

    return false;
}

void AliceDeviceState::setDndEnabled(bool value)
{
    dndEnabled_ = value;
}

void AliceDeviceState::setDeviceGroupState(const proto::DeviceGroupState& value)
{
    groupState_ = value;
}

void AliceDeviceState::setTandemState(NAlice::TDeviceStateCapability::TState state) {
    tandemState_ = state;
}

void AliceDeviceState::onRadioPlayerStart()
{
    lastRadioPlayTime_ = getNowTimestampMs();
}

int64_t AliceDeviceState::getLastRadioPlayerStart() const {
    return lastRadioPlayTime_;
}

void AliceDeviceState::onLegacyPlayerStart()
{
    lastMusicPlayTime_ = getNowTimestampMs();
}

int64_t AliceDeviceState::getLastLegacyPlayerStart() const {
    return lastMusicPlayTime_;
}

void AliceDeviceState::onLegacyPlayerStop()
{
    lastMusicStopTime_ = getNowTimestampMs();
}

int64_t AliceDeviceState::getLastLegacyPlayerStop() const {
    return lastMusicStopTime_;
}

void AliceDeviceState::setAppState(const proto::AppState& value)
{
    appState_ = value;
    if (appState_.audio_player_state().has_player_descriptor()) {
        audioClientPlayerType_ = appState_.audio_player_state().player_descriptor().type();
        audioClientSlaveMode_ = appState_.audio_player_state().player_descriptor().is_multiroom_slave();
    }
}

const proto::AppState& AliceDeviceState::getAppState() const {
    return appState_;
}

const std::shared_ptr<const StereoPairState>& AliceDeviceState::getStereoPairState() const {
    return stereoPairState_;
}

void AliceDeviceState::setStereoPairState(const std::shared_ptr<const StereoPairState>& stereoPairState)
{
    stereoPairState_ = stereoPairState;
}

void AliceDeviceState::setContentSettings(const std::string& value)
{
    if (contentSettings_ == value) {
        return;
    }

    contentSettings_ = value;
}

void AliceDeviceState::setChildContentSettings(const std::string& value)
{
    if (childContentSettings_ == value) {
        return;
    }

    childContentSettings_ = value;
}

void AliceDeviceState::setSpotterWord(const std::string& value)
{
    if (spotterWord_ == value) {
        return;
    }

    spotterWord_ = value;
}

void AliceDeviceState::setCalldState(const std::optional<proto::CalldSessionState>& value)
{
    calldState_ = value;
}

void AliceDeviceState::setSmartActivation(const Json::Value& value)
{
    smartActivation_ = value;
}

void AliceDeviceState::setNetworkStatus(const proto::NetworkStatus& value)
{
    networkStatus_ = value;
}

void AliceDeviceState::setWifiList(const proto::WifiList& value)
{
    wifiList_ = value;
}

void AliceDeviceState::setTvPolicyInfo(YandexIO::TvPolicyInfo value)
{
    tvPolicyInfo_ = std::move(value);
}

void AliceDeviceState::setClockDisplayState(YandexIO::ClockDisplayState clockDisplayState)
{
    clockDisplayStateOpt_ = clockDisplayState;
}

void AliceDeviceState::setActiveActions(const NAlice::TDeviceState::TActiveActions& payload)
{
    // TODO: setActiveActions conflicts with setActiveActionSemanticFrame
    // Their future will be decided soon
    activeActions_ = payload;
    environmentState_.setActiveActions(payload);
}

void AliceDeviceState::setActiveActionSemanticFrame(const std::optional<std::string>& payload) {
    // TODO: setActiveActions conflicts with setActiveActionSemanticFrame
    // Their future will be decided soon
    if (!payload) {
        activeActionSemanticFrame_ = Json::nullValue;
        return;
    }

    if (auto semanticFrameOptional = tryParseJson(payload.value())) {
        activeActionSemanticFrame_ = std::move(*semanticFrameOptional);
    } else {
        YIO_LOG_ERROR_EVENT("AliceDeviceState.BadJson.ActiveAction", "Can't parse active action payload: " << payload.value());
        activeActionSemanticFrame_ = Json::nullValue;
    }
}

void AliceDeviceState::setMegamindCookie(std::string value)
{
    megamindCookie_ = std::move(value);
}

const std::string& AliceDeviceState::getMegamindCookie() const {
    return megamindCookie_;
}

Json::Value AliceDeviceState::formatNetworkNeighbours() const {
    const Json::Value wifiList = convertWifiListToJson(wifiList_);

    Json::Value result(Json::arrayValue);

    if (wifiList.isArray()) {
        for (const auto& wifi : wifiList) {
            Json::Value item;
            for (const auto& key : wifi.getMemberNames()) {
                if (key == "ssid" || key == "channel") {
                    item[key] = wifi[key];
                } else if (key == "mac") {
                    // Rename "mac" field for device_state
                    item["bssid"] = wifi[key];
                }
            }

            result.append(item);
        }
    }

    return result;
}

std::string AliceDeviceState::formatInternetConnectionType(const proto::NetworkStatus& networkStatus)
{
    if (networkStatus.has_wifi_status() && networkStatus.wifi_status().has_current_network()) {
        // TODO: current network can be found from parsing wifi_list message, not wifi_status.
        //  But right now flag is_connected in wifi_list is not reliable.
        const int frequency = networkStatus.wifi_status().current_network().freq();
        if (frequency > 2400 && frequency < 2500) {
            return "Wifi_2_4GHz";
        } else if (frequency > 5000 && frequency < 6000) {
            return "Wifi_5GHz";
        } else {
            YIO_LOG_ERROR_EVENT("AliceDeviceState.DeviceStateUnknownWifiFrequency", "Unknown wifi frequency " << frequency);
        }
    } else if (networkStatus.has_ethernet_status()) {
        return "Ethernet";
    } else {
        YIO_LOG_WARN("No either wifi nor ethernet status");
    }

    return "";
}

Json::Value AliceDeviceState::formatCalldState(const std::optional<proto::CalldSessionState>& calldState) {
    Json::Value state;

    if (calldState.has_value() && calldState->authorized() && calldState->connected()) {
        switch (calldState->status()) {
            case quasar::proto::CalldSessionState::NEW:
            case quasar::proto::CalldSessionState::DIALING:
            case quasar::proto::CalldSessionState::RINGING:
                switch (calldState->direction()) {
                    case quasar::proto::CalldSessionState::OUTGOING:
                        state["current"]["call_id"] = calldState->call_guid();
                        state["current"]["recipient_id"] = calldState->user_guid();
                        state["current"]["caller_name"] = calldState->user_name();
                        break;
                    case quasar::proto::CalldSessionState::INCOMING:
                        state["incoming"]["call_id"] = calldState->call_guid();
                        state["incoming"]["recipient_id"] = calldState->user_guid();
                        state["incoming"]["caller_name"] = calldState->user_name();
                        break;
                }
                break;
            case quasar::proto::CalldSessionState::ACCEPTING:
            case quasar::proto::CalldSessionState::CONNECTING:
            case quasar::proto::CalldSessionState::CONNECTED:
                state["current"]["call_id"] = calldState->call_guid();
                state["current"]["recipient_id"] = calldState->user_guid();
                break;
            case quasar::proto::CalldSessionState::ENDED:
            case quasar::proto::CalldSessionState::NOCALL:
                break;
        }
    }

    return state;
}

Json::Value AliceDeviceState::formatDeviceConfig(
    const std::string& contentSettings,
    const std::string& childContentSettings,
    const std::string& spotterWord)
{
    Json::Value deviceConfig;
    deviceConfig["content_settings"] = contentSettings;
    if (!childContentSettings.empty()) {
        deviceConfig["child_content_settings"] = childContentSettings;
    }
    deviceConfig["spotter"] = spotterWord;

    return deviceConfig;
}

Json::Value AliceDeviceState::formatActions(const proto::AppState& appState)
{
    if (appState.has_screen_state()) {
        const auto& screenState = appState.screen_state();
        if (screenState.has_actions()) {
            const auto actions = tryParseJson(screenState.actions());
            if (actions.has_value()) {
                return actions.value();
            } else {
                YIO_LOG_ERROR_EVENT("AliceDeviceState.BadJson.ScreenStateActions", "Invalid actions received: " << screenState.actions());
            }
        }
    }

    return Json::Value();
}

void AliceDeviceState::slaveOverrideMediaState(Json::Value& state) const {
    if (state_.GetDeviceState().GetMultiroom().HasMusic()) {
        state["music"] = convertMessageToJson(state_.GetDeviceState().GetMultiroom().GetMusic()).value_or(Json::objectValue);
        auto lastStopTimestamp = tryGetDouble(state["audio_player"], "last_stop_timestamp", 0.);
        auto lastPlayTimestamp = tryGetDouble(state["audio_player"], "last_play_timestamp", 0.);
        state["music"]["last_stop_timestamp"] = (lastStopTimestamp > 0 ? lastStopTimestamp + 1 : 0.);
        state["music"]["last_play_timestamp"] = (lastPlayTimestamp > 0 ? lastPlayTimestamp + 1 : 0.);
        state["music"]["_owner"] = "multiroom";
    }
    state["audio_player"]["_owner"] = "multiroom";
}

Json::Value AliceDeviceState::formatStereoPairState(const std::shared_ptr<const StereoPairState>& stereoPairState)
{
    Json::Value jStereoPair;
    if (stereoPairState && stereoPairState->isStereoPair()) {
        jStereoPair["role"] = StereoPairState::roleName(stereoPairState->role);
        jStereoPair["channel"] = StereoPairState::channelName(stereoPairState->channel);
        jStereoPair["partner_device_id"] = stereoPairState->partnerDeviceId;
        jStereoPair["connected"] = StereoPairState::connectivityName(stereoPairState->connectivity);
        jStereoPair["stereo_player_status"] = StereoPairState::stereoPlayerStatusName(stereoPairState->stereoPlayerStatus);
    }
    return jStereoPair;
}

bool AliceDeviceState::getIsFollowerConnected() const {
    return groupState_.has_follower() &&
           groupState_.follower().has_connection_state() &&
           groupState_.follower().connection_state() == proto::DeviceGroupState::CONNECTED;
}

Json::Value AliceDeviceState::convertWifiListToJson(const proto::WifiList& src)
{
    auto itemToJson = [](const auto& item) {
        Json::Value jsonItem;
        jsonItem["mac"] = item.mac();
        jsonItem["signal_strength"] = item.rssi();
        jsonItem["ssid"] = item.ssid();
        jsonItem["channel"] = item.channel();
        jsonItem["is_connected"] = item.is_connected();
        return jsonItem;
    };

    Json::Value rval(Json::arrayValue);
    for (const auto& item : src.hotspots()) {
        if (!item.mac().empty()) {
            rval.append(itemToJson(item));
        }
    }

    return rval;
}

Json::Value AliceDeviceState::formatDeviceConfig(const YandexIO::TvPolicyInfo& tvPolicyInfo)
{
    Json::Value deviceConfig = Json::nullValue;

    if (tvPolicyInfo.contentSettings) {
        deviceConfig["content_settings"] = *tvPolicyInfo.contentSettings;
    }

    if (tvPolicyInfo.ageLimit) {
        deviceConfig["age_limit"] = *tvPolicyInfo.ageLimit;
    }

    return deviceConfig;
}

Json::Value AliceDeviceState::formatClockTower(const std::shared_ptr<IClockTowerProvider>& clockTowerProvider)
{
    Json::Value clockTower;
    if (clockTowerProvider) {
        auto clockDump = clockTowerProvider->dumpAllClocks();
        if (!clockDump.empty()) {
            for (const auto& [id, ns] : clockDump) {
                clockTower[id] = static_cast<int64_t>(std::chrono::duration_cast<std::chrono::microseconds>(ns).count());
            }
        }
    }
    return clockTower;
}

bool AliceDeviceState::hasMusicOrBluetoothPlayerScreen() const {
    if (state_.GetDeviceState().GetVideo().HasCurrentScreen()) {
        const auto currentScreen = state_.GetDeviceState().GetVideo().GetCurrentScreen();
        if (currentScreen == "music_player" || currentScreen == "bluetooth") {
            return true;
        }
    }
    return false;
}

bool AliceDeviceState::hasVideoPlayerScreen() const {
    return state_.GetDeviceState().HasVideo() && state_.GetDeviceState().GetVideo().HasCurrentScreen() && state_.GetDeviceState().GetVideo().GetCurrentScreen() == "video_player";
}

AudioPlayerType AliceDeviceState::getCurrentlyPlayingPlayerType() const {
    if (const auto& musicState = state_.GetDeviceState().GetMusic(); musicState.GetCurrentlyPlaying().HasTrackId() && musicState.GetPlayer().HasPause() && !musicState.GetPlayer().GetPause()) {
        return AudioPlayerType::MUSIC;
    }

    if (auto player = state_.GetDeviceState().GetRadio().fields().find("player"); player != state_.GetDeviceState().GetRadio().fields().end() && player->second.has_struct_value()) {
        const auto& playerFields = player->second.struct_value().fields();
        if (const auto pause = playerFields.find("pause"); pause != playerFields.end() && pause->second.has_bool_value() && !pause->second.bool_value()) {
            return AudioPlayerType::RADIO;
        }
    }

    if (const auto& bluetoothState = state_.GetDeviceState().GetBluetooth(); bluetoothState.HasPlayer() && !bluetoothState.GetPlayer().GetPause()) {
        return AudioPlayerType::BLUETOOTH;
    }

    if (const auto& audioPlayer = state_.GetDeviceState().GetAudioPlayer(); audioPlayer.HasPlayerState() && audioPlayer.GetPlayerState() == NAlice::TDeviceState::TAudioPlayer::Playing) {
        return (audioClientSlaveMode_ ? AudioPlayerType::MULTIROOM : AudioPlayerType::AUDIO_CLIENT);
    }

    return AudioPlayerType::NONE;
}

bool AliceDeviceState::isMediaPlaying() const {
    return getCurrentlyPlayingPlayerType() != AudioPlayerType::NONE;
}

AudioPlayerType AliceDeviceState::getCurrentPlayerType() const {
    AudioPlayerType resultType = getCurrentlyPlayingPlayerType();
    if (resultType != AudioPlayerType::NONE) {
        return resultType;
    }

    int64_t resultTimestamp = 0;
    if (const auto& musicState = state_.GetDeviceState().GetMusic(); musicState.GetCurrentlyPlaying().HasTrackId()) {
        if (musicState.GetPlayer().HasTimestamp() && musicState.GetPlayer().GetTimestamp() * 1000 > resultTimestamp) {
            resultTimestamp = musicState.GetPlayer().GetTimestamp() * 1000;
            resultType = AudioPlayerType::MUSIC;
        }
    }

    if (auto player = state_.GetDeviceState().GetRadio().fields().find("player"); player != state_.GetDeviceState().GetRadio().fields().end() && player->second.has_struct_value()) {
        const auto& playerFields = player->second.struct_value().fields();
        if (auto pause = playerFields.find("timestamp"); pause != playerFields.end() && pause->second.has_number_value() && pause->second.number_value() * 1000 > resultTimestamp) {
            resultTimestamp = pause->second.number_value() * 1000;
            resultType = AudioPlayerType::RADIO;
        }
    }

    if (const auto& bluetoothState = state_.GetDeviceState().GetBluetooth(); bluetoothState.CurrentConnectionsSize() > 0) {
        if (bluetoothState.GetPlayer().HasTimestamp() && bluetoothState.GetPlayer().GetTimestamp() * 1000 > resultTimestamp) {
            resultTimestamp = bluetoothState.GetPlayer().GetTimestamp() * 1000;
            resultType = AudioPlayerType::BLUETOOTH;
        }
    }

    if (const auto& audioPlayer = state_.GetDeviceState().GetAudioPlayer(); state_.GetDeviceState().HasAudioPlayer()) {
        const int64_t lastPlayTimestamp = audioPlayer.GetLastPlayTimestamp();
        const int64_t lastStopTimestamp = audioPlayer.GetLastStopTimestamp();
        const int64_t lastTimestamp = std::max(lastPlayTimestamp, lastStopTimestamp);

        if (lastTimestamp > resultTimestamp) {
            resultTimestamp = lastTimestamp;
            resultType = (audioClientSlaveMode_ ? AudioPlayerType::MULTIROOM : AudioPlayerType::AUDIO_CLIENT);
        }
    }

    return resultType;
}

std::chrono::seconds AliceDeviceState::getMusicStateProgress() const {
    int32_t result = 0;
    if (appState_.has_music_state()) {
        const auto& state = appState_.music_state();
        if (state.has_progress()) {
            result = static_cast<int32_t>(state.progress());
        }
    }

    return std::chrono::seconds(result);
}

bool AliceDeviceState::isAudioClientPlaying() const {
    return state_.GetDeviceState().GetAudioPlayer().GetPlayerState() == NAlice::TDeviceState::TAudioPlayer::Playing;
}

bool AliceDeviceState::hasMusicPlayerScreen() const {
    return state_.GetDeviceState().GetVideo().HasCurrentScreen() && state_.GetDeviceState().GetVideo().GetCurrentScreen() == "music_player";
}

bool AliceDeviceState::hasPlayerScreen() const {
    return state_.GetDeviceState().GetVideo().HasCurrentScreen();
}

Json::Value AliceDeviceState::formatActiveActions(const NAlice::TDeviceState::TActiveActions& activeActions) const {
    auto converted = convertMessageToJson(activeActions);
    if (converted) {
        auto result = *converted;
        if (result.isMember("semantic_frames")) {
            YIO_LOG_WARN("DeviceState.ActiveActions.SemanticFrames is overriden");
        }
        result["semantic_frames"] = activeActionSemanticFrame_;
        return result;
    } else {
        return Json::Value{};
    }
}

std::string AliceDeviceState::formatAudioClientState(proto::AudioClientState value)
{
    switch (value) {
        case proto::AudioClientState::IDLE:
            return "Idle";
        case proto::AudioClientState::BUFFERING:
            return "Buffering";
        case proto::AudioClientState::PAUSED:
            return "Paused";
        case proto::AudioClientState::PLAYING:
            return "Playing";
        case proto::AudioClientState::STOPPED:
            return "Stopped";
        case proto::AudioClientState::FINISHED:
            return "Finished";
        case proto::AudioClientState::FAILED:
            return "Idle"; // Failed is Idle from backend perspective
    }
}

void AliceDeviceState::setBrickStatus(proto::BrickStatus value)
{
    brickStatus_ = value;
}

proto::BrickStatus AliceDeviceState::getBrickStatus() const {
    return brickStatus_;
}

void AliceDeviceState::setAuthFailed(bool value)
{
    authFailed_ = value;
}

bool AliceDeviceState::getAuthFailed() const {
    return authFailed_;
}

std::shared_ptr<const DeviceState> AliceDeviceState::getProviderDeviceState()
{
    return deviceStateProvider_->deviceState().value();
}

bool AliceDeviceState::isConfiguringOrUpdating(bool isIgnoreCriticalUpdate)
{
    auto deviceState = getProviderDeviceState();
    return (deviceState->configuration != DeviceState::Configuration::CONFIGURED ||
            (deviceState->update == DeviceState::Update::HAS_CRITICAL && !isIgnoreCriticalUpdate));
}

void AliceDeviceState::setLocation(const proto::Location& location)
{
    location_ = location;
}

Json::Value AliceDeviceState::formatLocation() const {
    Json::Value json;

    if (location_.has_longitude() && location_.has_latitude()) {
        Json::Value json;
        json["lat"] = location_.latitude();
        json["lon"] = location_.longitude();

        double precision = 140; // Default Precision is provided by Alice Developers. Check SK-4965
        if (location_.has_precision()) {
            precision = location_.precision();
        }
        json["accuracy"] = precision;
    }

    return json;
}

void AliceDeviceState::setOAuthToken(std::string value)
{
    oauthToken_ = std::move(value);
}

const std::string& AliceDeviceState::getOAuthToken() const {
    return oauthToken_;
}

void AliceDeviceState::setSpeakerCount(int value)
{
    speakersCount_ = value;
}

int AliceDeviceState::getSpeakerCount() const {
    return speakersCount_;
}

EnvironmentStateHolder& AliceDeviceState::getEnvironmentState()
{
    return environmentState_;
}

const EnvironmentStateHolder& AliceDeviceState::getEnvironmentState() const {
    return environmentState_;
}

Json::Value AliceDeviceState::formatAdditionalOptions(
    const AliceConfig& config, Json::Value spotterRms) const {
    Json::Value additionalOptions;

    if (!getOAuthToken().empty()) {
        additionalOptions["oauth_token"] = getOAuthToken();
    }

    if (!config.getBassUrl().empty()) {
        additionalOptions["bass_url"] = config.getBassUrl();
    }

    Json::Value bassOptions;
    bassOptions["user_agent"] = config.getUserAgent();
    additionalOptions["bass_options"] = std::move(bassOptions);

    if (const auto& sfs = getState().GetSupportedFeatures(); !sfs.empty()) {
        additionalOptions["supported_features"] = Json::arrayValue;
        for (const auto& sf : sfs) {
            additionalOptions["supported_features"].append(sf);
        }
    }

    if (const auto& usfs = getState().GetUnsupportedFeatures(); !usfs.empty()) {
        additionalOptions["unsupported_features"] = Json::arrayValue;
        for (const auto& usf : usfs) {
            additionalOptions["unsupported_features"].append(usf);
        }
    }

    Json::Value auxilaryDeviceConfig = config.getAuxiliaryDeviceConfig();
    if (auxilaryDeviceConfig != Json::nullValue) {
        additionalOptions["quasar_auxiliary_config"] = std::move(auxilaryDeviceConfig);
    }

    if (spotterRms != Json::nullValue) {
        additionalOptions["spotter_rms"] = std::move(spotterRms);
        additionalOptions["speakers_count"] = getSpeakerCount();
    }

    return additionalOptions;
}

bool AliceDeviceState::isLongListenerScreen() const {
    if (!getVideoState().HasCurrentScreen()) {
        return false;
    }

    // TODO: refactor this and enable navigation based on another flag/field https://st.yandex-team.ru/TVANDROID-4733
    static const std::unordered_set<TString> longListenerScreen = {
        "gallery",
        "season_gallery",
        "tv_gallery",
        "description",
        "payment",
        "mordovia_webview",
        "search_results",
        "tv_main",
        "content_details",
        "tv_expanded_collection",
    };
    return longListenerScreen.contains(getVideoState().GetCurrentScreen());
}

void AliceDeviceState::setTimezone(std::string value)
{
    timezone_ = std::move(value);
}

const std::string& AliceDeviceState::getTimezone() const {
    return timezone_;
}

void AliceDeviceState::setMetricaUuid(std::string value)
{
    metricaUuid_ = std::move(value);
}

const std::string& AliceDeviceState::getMetricaUuid() const {
    return metricaUuid_;
}

void AliceDeviceState::setPassportUid(std::string value)
{
    passportUid_ = std::move(value);
}

const std::string& AliceDeviceState::getPassportUid() const {
    return passportUid_;
}

Json::Value AliceDeviceState::buildSoundLogExtra(const AliceConfig& config) const {
    Json::Value result;

    result["device_state"] = formatJson();
    result["environment_state"] = getEnvironmentState().formatJson();

    auto& deviceState = result["device_state"];
    deviceState.removeMember("internet_connection");
    deviceState.removeMember("last_watched");
    deviceState.removeMember("music");
    result["experiments"] = formatExperiments();

    const auto quasmodromGroup = config.getQuasmodromGroup();
    if (!quasmodromGroup.empty()) {
        result["quasmodrom_group"] = quasmodromGroup;
        result["quasmodrom_subgroup"] = config.getQuasmodromSubgroup();
    }

    if (const auto testBuckets = config.getTestBuckets(); !testBuckets.empty()) {
        result["test_buckets"] = testBuckets;
    }
    if (const auto testIds = config.getTestIds(); !testIds.empty()) {
        result["testids"] = testIds;
    }

    return result;
}

void AliceDeviceState::onCapabilityStateChanged(const std::shared_ptr<YandexIO::ICapability>& /*capability*/, const NAlice::TCapabilityHolder& state) {
    if (!state.HasDeviceStateCapability()) {
        return;
    }
    state_.CopyFrom(state.GetDeviceStateCapability().GetState());
}

const NAlice::TDeviceState& AliceDeviceState::getDeviceState() const {
    return state_.GetDeviceState();
}

const NAlice::TDeviceState::TVideo& AliceDeviceState::getVideoState() const {
    return getState().GetDeviceState().GetVideo();
}

void AliceDeviceState::onCapabilityEvents(const std::shared_ptr<YandexIO::ICapability>& /*capability*/, const std::vector<NAlice::TCapabilityEvent>& /*events*/) {
    // ¯\_(ツ)_/¯
}

const NAlice::TDeviceStateCapability::TState& AliceDeviceState::getState() const {
    return getIsFollowerConnected() ? tandemState_ : state_;
}

bool AliceDeviceState::hasPlayingAlarm() const {
    const auto& deviceState = getDeviceState();
    if (deviceState.HasAlarmState() && deviceState.GetAlarmState().GetCurrentlyPlaying()) {
        return true;
    }
    for (const auto& timer : deviceState.GetTimers().GetActiveTimers()) {
        if (timer.GetCurrentlyPlaying()) {
            return true;
        }
    }
    return false;
}
