#pragma once

#include "i_alice_device_state.h"
#include "environment_state.h"

#include <yandex_io/interfaces/clock_tower/i_clock_tower_provider.h>
#include <yandex_io/interfaces/multiroom/i_multiroom_provider.h>
#include <yandex_io/interfaces/stereo_pair/i_stereo_pair_provider.h>
#include <yandex_io/protos/model_objects.pb.h>
#include <yandex_io/protos/quasar_proto.pb.h>
#include <yandex_io/sdk/clock_display_state.h>
#include <yandex_io/sdk/tv_set_state.h>
#include <yandex_io/interfaces/device_state/i_device_state_provider.h>
#include <yandex_io/protos/capabilities/device_state_capability.pb.h>

#include <json/json.h>

#include <map>
#include <string>

namespace quasar {

    class AliceConfig;

    enum class AudioPlayerType {
        MUSIC = 0,
        RADIO = 1,
        BLUETOOTH = 2,
        AUDIO_CLIENT = 4,
        MULTIROOM = 5,
        NONE = 6,
    };

    const char* audioPlayerTypeName(AudioPlayerType /*audioPlayerType*/);

    class AliceDeviceState: public IAliceDeviceState, public YandexIO::ICapability::IListener {
    public:
        AliceDeviceState(std::string deviceId,
                         std::shared_ptr<IClockTowerProvider> clockTowerProvider,
                         std::shared_ptr<IDeviceStateProvider> deviceStateProvider,
                         EnvironmentStateHolder environmentState);

        Json::Value formatJson() const override;
        Json::Value formatAdditionalOptions(const AliceConfig& config, Json::Value spotterRms) const;
        Json::Value formatExperiments() const;
        Json::Value formatEnrollmentHeaders() const;

    public:
        // ICapability::IListener
        void onCapabilityStateChanged(const std::shared_ptr<YandexIO::ICapability>& capability, const NAlice::TCapabilityHolder& state) override;
        void onCapabilityEvents(const std::shared_ptr<YandexIO::ICapability>& capability, const std::vector<NAlice::TCapabilityEvent>& events) override;

    public:
        void onConfigUpdate(const AliceConfig& config);

        const NAlice::TDeviceState& getDeviceState() const override;
        const NAlice::TDeviceStateCapability::TState& getState() const;
        const NAlice::TDeviceState::TVideo& getVideoState() const override;
        const proto::NetworkStatus& getNetworkStatus() const;
        const proto::WifiList& getWifiList() const;
        bool hasActivePlayer() const;
        bool hasMusicOrBluetoothPlayerScreen() const;
        bool hasVideoPlayerScreen() const;
        bool hasMusicPlayerScreen() const;
        bool hasPlayerScreen() const;
        AudioPlayerType getCurrentPlayerType() const;
        AudioPlayerType getCurrentlyPlayingPlayerType() const;
        bool isMediaPlaying() const override;
        bool hasPlayingAlarm() const override;
        std::chrono::seconds getMusicStateProgress() const;
        [[nodiscard]] bool isAudioClientPlaying() const;

        void setDndEnabled(bool value);

        void setDeviceGroupState(const proto::DeviceGroupState& value);
        void setAppState(const proto::AppState& value);
        const proto::AppState& getAppState() const;
        void setSmartActivation(const Json::Value& value);

        const std::shared_ptr<const StereoPairState>& getStereoPairState() const;
        void setStereoPairState(const std::shared_ptr<const StereoPairState>& stereoPairState);

        void setCalldState(const std::optional<proto::CalldSessionState>& value);
        void setNetworkStatus(const proto::NetworkStatus& value);
        void setWifiList(const proto::WifiList& value);
        void setTvPolicyInfo(YandexIO::TvPolicyInfo value);

        void setIsScreenActive(bool isScreenActive);
        void setActiveActions(const NAlice::TDeviceState::TActiveActions& payload);
        void setActiveActionSemanticFrame(const std::optional<std::string>& payload);

        void setClockDisplayState(YandexIO::ClockDisplayState clockDisplayState);

        void setMegamindCookie(std::string value);
        const std::string& getMegamindCookie() const;

        void setBrickStatus(proto::BrickStatus value);
        proto::BrickStatus getBrickStatus() const;

        void setAuthFailed(bool value);
        bool getAuthFailed() const;

        std::shared_ptr<const DeviceState> getProviderDeviceState();
        bool isConfiguringOrUpdating(bool isIgnoreCriticalUpdate);

        void setLocation(const proto::Location& location);
        Json::Value formatLocation() const;

        void setOAuthToken(std::string value);
        const std::string& getOAuthToken() const;

        void setSpeakerCount(int value);
        int getSpeakerCount() const;

        EnvironmentStateHolder& getEnvironmentState() override;
        const EnvironmentStateHolder& getEnvironmentState() const override;

        Json::Value formatAdditionalOptions() const;

        bool isLongListenerScreen() const override;

        void setTimezone(std::string value);
        const std::string& getTimezone() const;

        void setMetricaUuid(std::string value);
        const std::string& getMetricaUuid() const;

        void setPassportUid(std::string value);
        const std::string& getPassportUid() const;

        Json::Value buildSoundLogExtra(const AliceConfig& config) const;

        void setSpotterWord(const std::string& value);

        void setTandemState(NAlice::TDeviceStateCapability::TState state);

        void onRadioPlayerStart();
        int64_t getLastRadioPlayerStart() const;

        void onLegacyPlayerStart();
        int64_t getLastLegacyPlayerStart() const;

        void onLegacyPlayerStop();
        int64_t getLastLegacyPlayerStop() const;

    private:
        static Json::Value formatVideoList(const google::protobuf::RepeatedPtrField<proto::VideoState>& source);
        static Json::Value formatMediaItem(
            const proto::MediaItem& item,
            const proto::Progress& progress,
            long timestampSec,
            const std::string& audioLanguage,
            const std::string& subtitlesLanguage);
        static Json::Value formatStereoPairState(const std::shared_ptr<const StereoPairState>& stereoPairState);
        static Json::Value formatCalldState(const std::optional<proto::CalldSessionState>& calldState);
        Json::Value formatNetworkNeighbours() const;

        static Json::Value formatDeviceConfig(
            const std::string& contentSettings,
            const std::string& childContentSettings,
            const std::string& spotterWord);

        static std::string formatInternetConnectionType(const proto::NetworkStatus& networkStatus);
        static Json::Value formatActions(const proto::AppState& appState);

        static Json::Value convertWifiListToJson(const proto::WifiList& src);
        static Json::Value formatDeviceConfig(const YandexIO::TvPolicyInfo& tvPolicyInfo);
        static Json::Value formatClockTower(const std::shared_ptr<IClockTowerProvider>& clockTowerProvider);

        void slaveOverrideMediaState(Json::Value& state) const;

        bool getIsFollowerConnected() const;

        void setContentSettings(const std::string& value);
        void setChildContentSettings(const std::string& value);

        static std::string formatAudioClientState(proto::AudioClientState value);
        Json::Value formatActiveActions(const NAlice::TDeviceState::TActiveActions& activeActions) const;

    private:
        const std::string deviceId_;
        const std::shared_ptr<IClockTowerProvider> clockTowerProvider_;

        bool dndEnabled_{false};

        std::string contentSettings_ = "medium";
        std::string childContentSettings_;
        std::string spotterWord_ = "alisa";

        NAlice::TDeviceState::TActiveActions activeActions_;
        Json::Value activeActionSemanticFrame_;
        Json::Value smartActivation_;

        proto::AppState appState_;
        // FIXME: remove it after removing choosing player type methods.
        proto::AudioPlayerDescriptor::PlayerType audioClientPlayerType_;
        bool audioClientSlaveMode_{false};
        proto::DeviceGroupState groupState_;
        std::optional<proto::CalldSessionState> calldState_;
        proto::NetworkStatus networkStatus_;
        proto::WatchedVideoState watchedVideoState_;
        proto::WifiList wifiList_;
        std::optional<YandexIO::TvPolicyInfo> tvPolicyInfo_;
        std::optional<YandexIO::ClockDisplayState> clockDisplayStateOpt_;

        NAlice::TDeviceStateCapability::TState state_;
        NAlice::TDeviceStateCapability::TState tandemState_;

        std::string megamindCookie_;

        proto::BrickStatus brickStatus_ = proto::BrickStatus::UNKNOWN_BRICK_STATUS;
        bool authFailed_ = false;
        std::shared_ptr<IDeviceStateProvider> deviceStateProvider_;

        proto::Location location_;
        std::string oauthToken_;
        int speakersCount_ = 0;
        EnvironmentStateHolder environmentState_;
        std::string timezone_{"Europe/Moscow"};
        std::string passportUid_;
        std::string metricaUuid_;
        int64_t lastMusicPlayTime_ = 0;
        int64_t lastMusicStopTime_ = 0;
        int64_t lastRadioPlayTime_ = 0;

        std::shared_ptr<const StereoPairState> stereoPairState_;
    };

} // namespace quasar
