#include "environment_state.h"

#include <alice/megamind/protos/common/subscription_state.pb.h>
#include <alice/megamind/protos/common/subscription_state.pb.h>
#include <alice/megamind/protos/common/tandem_state.pb.h>

#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/protobuf_utils/debug.h>
#include <yandex_io/libs/protobuf_utils/json.h>

#include <google/protobuf/any.pb.h>
#include <speechkit/SpeechKit.h>

YIO_DEFINE_LOG_MODULE("environment_state_holder");

using namespace quasar;

const std::string EnvironmentStateHolder::TANDEM_DEVICE_CONNECTED_KEY = "tandem_device_connected";
const std::string EnvironmentStateHolder::TANDEM_DEVICE_ID_KEY = "tandem_device_id";

EnvironmentStateHolder::EnvironmentStateHolder(std::string deviceId,
                                               std::shared_ptr<YandexIO::ITelemetry> telemetry)
    : deviceId_(std::move(deviceId))
    , telemetry_(std::move(telemetry))
{
}

Json::Value EnvironmentStateHolder::formatJson() const {
    NAlice::TEnvironmentState environmentState;

    if (endpointStorage_ != nullptr) {
        for (const auto& endpoint : endpointStorage_->getEndpoints()) {
            if (endpoint->getId() == deviceId_) {
                auto state = endpoint->getState();
                for (auto it = state.GetCapabilities().begin(); it != state.GetCapabilities().end(); it++) {
                    const auto& any = *it;
                    NAlice::TDeviceStateCapability dsc;
                    if (any.UnpackTo(&dsc)) {
                        state.MutableCapabilities()->erase(it);
                        break;
                    }
                }
                environmentState.add_endpoints()->Swap(&state);
                break;
            }
        }
    }

    if (isEnabled_) {
        if (tandemGroup_.devices_size() != 0) {
            environmentState.add_groups()->CopyFrom(tandemGroup_);
        }
        if (localDeviceInfo_.has_application()) {
            environmentState.add_devices()->CopyFrom(localDeviceInfo_);
        }
        if (remoteDeviceInfo_.has_application()) {
            environmentState.add_devices()->CopyFrom(remoteDeviceInfo_);
        }
    }

    return quasar::convertMessageToJson(environmentState, true).value();
}

void EnvironmentStateHolder::enable(bool enabled) {
    isEnabled_ = enabled;
}

void EnvironmentStateHolder::updateLocalDevice() {
    if (!isEnabled_) {
        return;
    }

    updateLocalDeviceImpl();
    if (environmentChangedCallback_) {
        environmentChangedCallback_();
    }
}

void EnvironmentStateHolder::updateLocalSubscriptionType(const proto::SubscriptionState& subscriptionState) {
    if (!isEnabled_) {
        return;
    }

    updateLocalDeviceImpl();

    auto result = NAlice::TSubscriptionState_ESubscriptionType_unknown;

    const auto info = tryParseJsonOrEmpty(subscriptionState.subscription_info());
    const std::string mode = tryGetString(info, "mode", "unknown");
    if (mode == "subscription") {
        result = NAlice::TSubscriptionState_ESubscriptionType_yandex_subscription;
    } else if (mode == "transaction") {
        result = NAlice::TSubscriptionState_ESubscriptionType_none;
    } else {
        YIO_LOG_WARN("unexpected subscription mode: " << mode);
    }

    localDeviceInfo_.mutable_speakerdevicestate()->mutable_devicesubscriptionstate()->set_subscription(result);

    if (environmentChangedCallback_) {
        environmentChangedCallback_();
    }
}

void EnvironmentStateHolder::updateTandemGroup(const proto::DeviceGroupState& groupState) {
    if (!isEnabled_) {
        return;
    }

    updateLocalDeviceImpl();
    savedGroupState_ = groupState;

    tandemGroup_.Clear();
    localDeviceInfo_.mutable_speakerdevicestate()->clear_tandemstate();

    if (!groupState.has_local_role() || groupState.local_role() == proto::DeviceGroupState::Role::DeviceGroupState_Role_STAND_ALONE) {
        // should drop remote device from environment
        remoteDeviceInfo_.Clear();
        telemetry_->deleteAppEnvironmentValue(TANDEM_DEVICE_CONNECTED_KEY);
        telemetry_->deleteAppEnvironmentValue(TANDEM_DEVICE_ID_KEY);
        return;
    }

    tandemGroup_.set_type(NAlice::TEnvironmentGroupInfo::tandem);
    auto currentDevice = tandemGroup_.add_devices();
    currentDevice->set_platform(SpeechKit::SpeechKit::getInstance()->getPlatformInfo()->getDeviceModel());
    currentDevice->set_id(TString(SpeechKit::SpeechKit::getInstance()->getDeviceId()));

    bool tandemConnected = false;
    auto connectedDevice = tandemGroup_.add_devices();
    if (groupState.local_role() == proto::DeviceGroupState::Role::DeviceGroupState_Role_LEADER) {
        currentDevice->set_role(NAlice::TEnvironmentGroupInfo::leader);
        connectedDevice->set_role(NAlice::TEnvironmentGroupInfo::follower);
        connectedDevice->set_platform(groupState.follower().platform());
        connectedDevice->set_id(groupState.follower().device_id());
        tandemConnected = groupState.follower().connection_state() == proto::DeviceGroupState::CONNECTED;

        telemetry_->putAppEnvironmentValue(TANDEM_DEVICE_CONNECTED_KEY, tandemConnected ? "1" : "0");
        telemetry_->putAppEnvironmentValue(TANDEM_DEVICE_ID_KEY, groupState.follower().device_id());
    } else {
        currentDevice->set_role(NAlice::TEnvironmentGroupInfo::follower);
        connectedDevice->set_role(NAlice::TEnvironmentGroupInfo::leader);
        connectedDevice->set_platform(groupState.leader().platform());
        connectedDevice->set_id(groupState.leader().device_id());
        tandemConnected = groupState.leader().connection_state() == proto::DeviceGroupState::CONNECTED;

        telemetry_->putAppEnvironmentValue(TANDEM_DEVICE_CONNECTED_KEY, tandemConnected ? "1" : "0");
        telemetry_->putAppEnvironmentValue(TANDEM_DEVICE_ID_KEY, groupState.leader().device_id());
    }

    localDeviceInfo_.mutable_speakerdevicestate()->mutable_tandemstate()->set_connected(tandemConnected);
    if (!tandemConnected) {
        // should drop remote device from environment
        remoteDeviceInfo_.Clear();
    }

    if (environmentChangedCallback_) {
        environmentChangedCallback_();
    }
}

void EnvironmentStateHolder::updateLocalMediaDeviceIdentifier(const NAlice::TClientInfoProto::TMediaDeviceIdentifier& identifier) {
    localDeviceInfo_.mutable_application()->mutable_mediadeviceidentifier()->CopyFrom(identifier);
}

void EnvironmentStateHolder::updateLocalDeviceImpl() {
    const auto speechKit = SpeechKit::SpeechKit::getInstance();
    localDeviceInfo_.mutable_application()->set_uuid(TString(speechKit->getUuid()));
    localDeviceInfo_.mutable_application()->set_deviceid(TString(speechKit->getDeviceId()));

    const auto platformInfo = speechKit->getPlatformInfo();
    localDeviceInfo_.mutable_application()->set_devicemodel(platformInfo->getDeviceModel());
    localDeviceInfo_.mutable_application()->set_appid(platformInfo->getAppId());
    localDeviceInfo_.mutable_application()->set_appversion(platformInfo->getAppVersion());

    localDeviceInfo_.clear_supportedfeatures();
    const auto supportedFeatures = platformInfo->getSupportedFeatures();
    for (const std::string& feature : supportedFeatures) {
        localDeviceInfo_.add_supportedfeatures(TString(feature));
    }
}

void EnvironmentStateHolder::setActiveActions(const NAlice::TDeviceState::TActiveActions& payload) {
    localDeviceInfo_.mutable_speakerdevicestate()->mutable_activeactions()->CopyFrom(payload);
}

NAlice::TEnvironmentDeviceInfo EnvironmentStateHolder::getLocalDeviceInfo() const {
    return localDeviceInfo_;
}

void EnvironmentStateHolder::setEndpointStorage(std::shared_ptr<YandexIO::IEndpointStorage> endpointStorage)
{
    endpointStorage_ = std::move(endpointStorage);
}

void EnvironmentStateHolder::updateRemoteDevice(const NAlice::TEnvironmentDeviceInfo& device) {
    if (!isEnabled_) {
        return;
    }

    remoteDeviceInfo_.CopyFrom(device);
    updateTandemGroup(savedGroupState_);
}
