#include "directive_processor.h"

#include <yandex_io/libs/base/directives.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/base/persistent_file.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/base/directives.h>

#include <util/system/yassert.h>

YIO_DEFINE_LOG_MODULE("directive_processor");

using namespace YandexIO;

DirectiveProcessor::FlagBlocker::FlagBlocker(bool& block)
    : block_(block)
{
    block_ = true;
}

DirectiveProcessor::FlagBlocker::~FlagBlocker()
{
    block_ = false;
}

void DirectiveProcessor::addListener(std::weak_ptr<IDirectiveProcessorListener> wlistener) {
    if (const auto listener = wlistener.lock()) {
        listener->onSequenceStateChanged();
    }
    listeners_.push_back(std::move(wlistener));
}

void DirectiveProcessor::notifySequenceStateChanged() {
    for (const auto& wlistener : listeners_) {
        if (const auto listener = wlistener.lock()) {
            listener->onSequenceStateChanged();
        }
    }
}

void DirectiveProcessor::notifyDialogChannelIsIdle() {
    for (const auto& wlistener : listeners_) {
        if (const auto listener = wlistener.lock()) {
            listener->onDialogChannelIsIdle();
        }
    }
}

void DirectiveProcessor::notifyDirectiveHandled(const std::shared_ptr<Directive>& directive) {
    for (const auto& wlistener : listeners_) {
        if (const auto listener = wlistener.lock()) {
            listener->onDirectiveHandled(directive);
        }
    }
}

void DirectiveProcessor::notifyDirectiveCompleted(const std::shared_ptr<Directive>& directive, IDirectiveProcessorListener::Result result) {
    for (const auto& wlistener : listeners_) {
        if (const auto listener = wlistener.lock()) {
            listener->onDirectiveCompleted(directive, result);
        }
    }
}

void DirectiveProcessor::start(std::string stateFileName)
{
    Y_VERIFY(stateFileName_.empty());
    stateFileName_ = std::move(stateFileName);

    loadStateFromFile();

    // Restore directives in backward order to correct audio focus apply
    for (int channelIndex = quasar::proto::AudioChannel_MAX; channelIndex >= quasar::proto::AudioChannel_MIN; --channelIndex) {
        const auto& directives = channels_[channelIndex];
        if (!directives.empty()) {
            auto directive = *directives.begin();
            if (const auto& handler = findHandlerForDirective(directive)) {
                handler->restoreDirective(directive);
            }
        }
    }
}

void DirectiveProcessor::clear()
{
    preprocessors_.clear();
    directiveHandlers_.clear();
    listeners_.clear();
    clearChannels();
}

void DirectiveProcessor::addDirectives(std::list<std::shared_ptr<Directive>> directives)
{
    dumpState("addDirectives", directives);

    {
        FlagBlocker blocker(isSelectActivityBlocked_);

        preprocessDirectives(directives);
        processDirectives(std::move(directives));
    }
    selectNextActivity();
}

void DirectiveProcessor::onHandleDirectiveStarted(std::shared_ptr<Directive> directive)
{
    YIO_LOG_INFO("onHandleDirectiveStarted " << directive->format());
    for (const auto& wlistener : listeners_) {
        if (const auto listener = wlistener.lock()) {
            listener->onDirectiveStarted(directive);
        }
    }
}

void DirectiveProcessor::onHandleDirectiveCompleted(std::shared_ptr<Directive> directive, bool success)
{
    YIO_LOG_INFO("onHandleDirectiveCompleted success=" << success << ", directive=" << directive->format());

    {
        FlagBlocker saveToFileGuard(isSaveStateToFileBlocked_);

        if (directive == currentRequestDirective_) {
            currentRequestDirective_.reset();
        }
        const auto audioChannel = directive->getData().channel;
        if (audioChannel.has_value()) {
            freeChannel(audioChannel.value(), !success);
        }
        selectNextActivity();
    }

    onDirectiveStateChanged(directive);

    if (success) {
        notifyDirectiveCompleted(directive, IDirectiveProcessorListener::Result::SUCCESS);
    } else {
        notifyDirectiveCompleted(directive, IDirectiveProcessorListener::Result::FAIL);
    }
}

void DirectiveProcessor::onHandleDirectiveCompleted(
    std::shared_ptr<Directive> directive, std::list<std::shared_ptr<Directive>> directives)
{
    Y_VERIFY(directive != nullptr);
    dumpState("onHandleDirectiveCompleted", directives);

    {
        FlagBlocker saveToFileGuard(isSaveStateToFileBlocked_);

        preprocessDirectives(directives);
        onDirectiveCompletedInternal(directive);
        invalidatePrefetch();
        processDirectives(std::move(directives));
        selectNextActivity();
    }

    onDirectiveStateChanged(directive);
}

void DirectiveProcessor::onPrefetchDirectiveCompleted(
    std::shared_ptr<Directive> directive, std::list<std::shared_ptr<Directive>> directives)
{
    Y_VERIFY(directive != nullptr);
    dumpState("onPrefetchDirectiveCompleted", directives);

    preprocessDirectives(directives);
    directive->setPrefetchResult(std::move(directives));

    onDirectiveCompletedInternal(directive);
    prefetchFirstContentDirective(directive->getPrefetchResult(), nullptr);
    selectNextActivity();
}

void DirectiveProcessor::pumpPendingDirective()
{
    YIO_LOG_INFO("pumpPendingDirective " << pendingDirective_.get() << " " << (pendingDirective_ ? pendingDirective_->getData().name : ""));

    if (pendingDirective_ != nullptr) {
        auto* directives = findChainByDirective(pendingDirective_);

        if (directives != nullptr) {
            if (directives->size() > 1 && (*std::next(directives->begin()))->getPrefetchInProgress()) {
                YIO_LOG_INFO("pumpPendingDirective skipped due to prefetch in progress");
                return;
            }

            directives->erase(directives->begin());
            processDirectives(*directives);
        }

        pendingDirective_.reset();
    }
}

std::list<std::shared_ptr<Directive>>* DirectiveProcessor::findChainByDirective(const std::shared_ptr<Directive>& directive)
{
    for (auto& directives : channels_) {
        if (!directives.empty() && *directives.begin() == directive) {
            return &directives;
        }
    }

    return nullptr;
}

const DirectiveProcessor::State& DirectiveProcessor::getState() const {
    return channels_;
}

void DirectiveProcessor::setDeviceId(std::string deviceId)
{
    deviceId_ = std::move(deviceId);
}

void DirectiveProcessor::setPrefetchEnabled(bool prefetchEnabled)
{
    prefetchEnabled_ = prefetchEnabled;
}

void DirectiveProcessor::setDirectiveFactory(std::shared_ptr<IDirectiveFactory> directiveFactory)
{
    directiveFactory_ = std::move(directiveFactory);
}

void DirectiveProcessor::processDirectives(std::list<std::shared_ptr<Directive>> directives)
{
    YIO_LOG_INFO("processDirectives: " << Directive::formatDirectiveNames(directives));

    bool chainIsStarted = false;
    auto iter = directives.begin();

    while (iter != directives.end()) {
        auto directive = *iter;
        if (!directive->getPrefetchResult().empty()) {
            auto insertIter = directives.erase(iter);
            iter = directives.insert(insertIter, directive->getPrefetchResult().begin(), directive->getPrefetchResult().end());

            YIO_LOG_INFO(directive->getData().name << " replaced with " << Directive::formatDirectiveNames(directive->getPrefetchResult()));
            directive->clearPrefetchResult();
            continue;
        }

        chainIsStarted = chainIsStarted || directive->getFirstInChain();

        std::optional<quasar::proto::AudioChannel> audioChannel = directive->getData().channel;
        if (!audioChannel.has_value()) {
            if (directive->is(quasar::Directives::CLEAR_QUEUE)) {
                clearChannels();
            } else if (directive->is(quasar::Directives::AUDIO_STOP)) {
                clearContentChannel();
            }

            routeDirective(directive);
            iter = directives.erase(iter);
        } else {
            pendingTtsContentDirective();
            auto& channelDirectives = channels_[audioChannel.value()];
            if (!channelDirectives.empty()) {
                cancelDirective(*channelDirectives.begin());
            }

            channelDirectives = std::move(directives);

            routeDirective(directive);
            prefetchFirstContentDirective(channelDirectives, directive);
            if (chainIsStarted) {
                prefetchLastServerActionDirective(directive);
            }

            break;
        }
    }

    updateAudioFocus();
}

void DirectiveProcessor::freeChannel(quasar::proto::AudioChannel channel, bool dropChain)
{
    YIO_LOG_INFO("freeChannel " << quasar::proto::AudioChannel_Name(channel) << ", dropChain=" << dropChain);

    if (!channels_[channel].empty()) {
        auto directives = channels_[channel];
        channels_[channel].clear();
        auto directive = *directives.begin();

        if (dropChain) {
            directives.clear();
        } else {
            const auto curentTtsDirective = findActiveTtsDirective();
            const bool hasAliceActivity = currentRequestDirective_ != nullptr || curentTtsDirective != nullptr;

            YIO_LOG_INFO("hasAliceActivity=" << hasAliceActivity);
            if (hasAliceActivity) {
                pendingDirective_ = directive;
                channels_[channel] = std::move(directives);
            } else {
                directives.erase(directives.begin());
                processDirectives(std::move(directives));
            }
        }

        updateAudioFocus();
    }
}

void DirectiveProcessor::onDirectiveCompletedInternal(std::shared_ptr<Directive> directive)
{
    if (directive == currentRequestDirective_) {
        currentRequestDirective_.reset();
    }

    const auto audioChannel = directive->getData().channel;
    if (audioChannel.has_value()) {
        channels_[audioChannel.value()].clear();
    }
}

void DirectiveProcessor::onBlockPrefetchChanged(std::shared_ptr<Directive> directive)
{
    YIO_LOG_INFO("onBlockPrefetchChanged for " << directive->format());
    prefetchLastServerActionDirective(directive);
}

void DirectiveProcessor::clearChannels()
{
    YIO_LOG_INFO("clearChannels");

    requestQueue_.clear();

    for (auto& directives : channels_) {
        if (!directives.empty()) {
            const auto& directive = *directives.begin();

            cancelDirective(directive);
            directives.clear();
        }
    }
}

void DirectiveProcessor::clearContentChannel()
{
    auto& directives = channels_[quasar::proto::CONTENT_CHANNEL];
    YIO_LOG_INFO("clearContentChannel: " << Directive::formatDirectiveNames(directives));
    if (!directives.empty()) {
        const auto& directive = *directives.begin();

        cancelDirective(directive);
        directives.clear();
    }
}

void DirectiveProcessor::prefetchLastServerActionDirective(std::shared_ptr<Directive> directive)
{
    if (!prefetchEnabled_) {
        return;
    }

    auto* directives = findChainByDirective(directive);
    if (directives == nullptr) {
        return;
    }

    for (const auto& directive : *directives) {
        if (directive->isBlocksSubsequentPrefetch()) {
            YIO_LOG_INFO("Prefetch is blocked by directive " << directive->format());
            return;
        }
    }

    auto lastDirective = *(directives->rbegin());
    if (!lastDirective->getData().isServerAction() || lastDirective->getData().ignoreAnswer) {
        YIO_LOG_INFO("Ignore prefetch due to not server_action and ignore_answer");
        return;
    }

    if (currentRequestDirective_ != nullptr) {
        YIO_LOG_INFO("Ignore prefetch due to currentRequestDirective_ in progress: " << currentRequestDirective_->format());
        return;
    }

    if (const auto& handler = findHandlerForDirective(lastDirective)) {
        YIO_LOG_INFO("Prefetch " << lastDirective->format() << " via handler " << handler.get() << ":" << handler->getHandlerName());
        handler->prefetchDirective(lastDirective);
        currentRequestDirective_ = lastDirective;
    }
}

void DirectiveProcessor::prefetchFirstContentDirective(
    const std::list<std::shared_ptr<Directive>>& directives, const std::shared_ptr<Directive>& directive)
{
    if (!prefetchEnabled_) {
        return;
    }

    auto begin = directives.begin();
    if (directive != nullptr) {
        begin = std::find(directives.begin(), directives.end(), directive);
        if (begin != directives.end()) {
            ++begin;
        }
    }

    auto contentDirective = std::find_if(begin, directives.end(), [](const std::shared_ptr<Directive>& item) {
        const auto channel = item->getData().channel;
        return channel.has_value() && channel.value() == quasar::proto::CONTENT_CHANNEL;
    });

    if (contentDirective != directives.end()) {
        if (const auto& handler = findHandlerForDirective(*contentDirective)) {
            YIO_LOG_INFO("Prefetch " << (*contentDirective)->format() << " via handler " << handler.get() << ":" << handler->getHandlerName());
            handler->prefetchDirective(*contentDirective);
        }
    }
}

void DirectiveProcessor::invalidatePrefetch()
{
    YIO_LOG_INFO("invalidatePrefetch");

    for (auto& directives : channels_) {
        for (auto& directive : directives) {
            if (!directive->getPrefetchResult().empty()) {
                directive->clearPrefetchResult();
            }
        }
    }
}

void DirectiveProcessor::updateAudioFocus()
{
    notifySequenceStateChanged();
}

void DirectiveProcessor::selectNextActivity()
{
    if (isSelectActivityBlocked_) {
        YIO_LOG_INFO("selectNextActivity blocked by isSelectActivityBlocked_");
        return;
    }
    if (currentRequestDirective_ != nullptr) {
        YIO_LOG_INFO("selectNextActivity skipped. currentRequestDirective_=" << currentRequestDirective_->format());
        return;
    }
    if (auto ttsDirective = findActiveTtsDirective(); ttsDirective && ttsDirective != pendingDirective_) {
        YIO_LOG_INFO("selectNextActivity skipped. activeTtsDirective=" << ttsDirective->format());
        return;
    }
    if (!requestQueue_.empty()) {
        auto directive = std::move(requestQueue_.front());
        requestQueue_.pop_front();

        routeDirective(directive);
        return;
    }

    pumpPendingDirective();

    if (auto ttsDirective = findActiveTtsDirective()) {
        YIO_LOG_INFO("selectNextActivity skipped. activeTtsDirective after pumpPendingDirective=" << ttsDirective->format());
        return;
    }

    notifyDialogChannelIsIdle();
}

void DirectiveProcessor::enqueueDirective(std::shared_ptr<Directive> directive)
{
    YIO_LOG_INFO("enqueueDirective " << directive->format());
    requestQueue_.push_back(std::move(directive));
    selectNextActivity();
}

void DirectiveProcessor::dumpState(const std::string& prefix, const std::list<std::shared_ptr<Directive>>& directives)
{
    std::stringstream ss;

    ss << prefix << " " << Directive::formatDirectives(directives) << "; ";
    for (int i = quasar::proto::AudioChannel_MIN; i <= quasar::proto::AudioChannel_MAX; ++i) {
        ss << quasar::proto::AudioChannel_Name(static_cast<quasar::proto::AudioChannel>(i)) << "=[";
        for (auto& directive : channels_[i]) {
            ss << directive->format() << ";";
        }
        ss << "] ";
    }

    ss << "RequestQueue=[";
    for (const auto& directive : requestQueue_) {
        ss << directive->format();
    }
    ss << "]";

    YIO_LOG_INFO(ss.str());
}

bool DirectiveProcessor::addDirectiveHandler(const IDirectiveHandlerPtr& handler)
{
    std::set<IDirectiveHandlerPtr> handlers;
    for (const auto& existingHandler : directiveHandlers_) {
        if (existingHandler->getEndpointId() == handler->getEndpointId()) {
            handlers.insert(existingHandler);
        }
    }

    for (const auto& existingHandler : handlers) {
        const auto intersection = findFirstDirectiveNameIntersection(existingHandler, handler);
        if (intersection.has_value()) {
            YIO_LOG_ERROR_EVENT("DirectiveProcessor.FailedAddDirectiveHandler",
                                "Handler '" << handler->getHandlerName() << "' requests to handle directive '" << intersection.value() << "' "
                                                                                                                                          "which is already handled by '"
                                            << existingHandler->getHandlerName() << "'");
            return false;
        }
    }
    const auto& handlerSupportedDirectives = handler->getSupportedDirectiveNames();
    directiveNames_.insert(handlerSupportedDirectives.begin(), handlerSupportedDirectives.end());
    directiveHandlers_.insert(handler);
    return true;
}

bool DirectiveProcessor::removeDirectiveHandler(const IDirectiveHandlerPtr& handler)
{
    const auto handlerIter = directiveHandlers_.find(handler);
    if (handlerIter != std::end(directiveHandlers_)) {
        for (const auto& name : handler->getSupportedDirectiveNames()) {
            directiveNames_.erase(name);
        }
        directiveHandlers_.erase(handlerIter);
        return true;
    }

    return false;
}

IDirectiveHandlerPtr DirectiveProcessor::findDirectiveHandlerByName(const std::string& name) const {
    for (const auto& handler : directiveHandlers_) {
        if (handler->getHandlerName() == name) {
            return handler;
        }
    }

    return nullptr;
}

void DirectiveProcessor::routeDirective(std::shared_ptr<Directive> directive)
{
    if (const auto& handler = findHandlerForDirective(directive)) {
        if (directive->isRequest()) {
            if (currentRequestDirective_ != nullptr) {
                cancelDirective(currentRequestDirective_);
            }
            currentRequestDirective_ = directive;
        }

        YIO_LOG_INFO("routeDirective " << directive->format() << " via handler " << handler.get() << ":" << handler->getHandlerName());
        handler->handleDirective(directive);

        onDirectiveStateChanged(directive);
    } else {
        YIO_LOG_WARN("Failed to find handler for directive " << directive->format());
    }

    notifyDirectiveHandled(directive);
}

IDirectiveHandlerPtr DirectiveProcessor::findHandlerForDirective(const std::shared_ptr<Directive>& directive) const {
    std::set<IDirectiveHandlerPtr> handlers;
    const auto& directiveEndpointId = (directive->getData().endpointId.empty() ? deviceId_ : directive->getData().endpointId);
    for (const auto& handler : directiveHandlers_) {
        const auto& handlerEndpointId = (handler->getEndpointId().empty() ? deviceId_ : handler->getEndpointId());
        if (handlerEndpointId == directiveEndpointId) {
            handlers.insert(handler);
        }
    }

    const auto& name = directive->getData().name;
    for (const auto& handler : handlers) {
        if (handler->getSupportedDirectiveNames().count(name) > 0) {
            return handler;
        }
    }

    // Treat empty directiveNames as `*` wildcard
    for (const auto& handler : handlers) {
        if (handler->getSupportedDirectiveNames().empty()) {
            return handler;
        }
    }

    return nullptr;
}

const std::vector<IDirectivePreprocessorPtr>& DirectiveProcessor::getDirectivePreprocessors() const {
    return preprocessors_;
}

const std::set<std::string>& DirectiveProcessor::getSupportedDirectiveNames() const {
    return directiveNames_;
}

bool DirectiveProcessor::addDirectivePreprocessor(const IDirectivePreprocessorPtr& value)
{
    if (value == nullptr) {
        YIO_LOG_ERROR_EVENT("DirectiveProcessor.FailedAddDirectivePreprocessor", "null");
        return false;
    }

    if (std::find(preprocessors_.begin(), preprocessors_.end(), value) != preprocessors_.end()) {
        YIO_LOG_ERROR_EVENT("DirectiveProcessor.FailedAddDirectivePreprocessor", "Adding preprocessor twice " << value.get());
        return false;
    }

    preprocessors_.push_back(value);
    return true;
}

bool DirectiveProcessor::removeDirectivePreprocessor(const IDirectivePreprocessorPtr& value)
{
    auto iter = std::find(preprocessors_.begin(), preprocessors_.end(), value);
    if (iter == preprocessors_.end()) {
        YIO_LOG_DEBUG("Failed to remove preprocessor that don't exist " << value.get());
        return false;
    }

    preprocessors_.erase(iter);
    return true;
}

void DirectiveProcessor::preprocessDirectives(std::list<std::shared_ptr<Directive>>& directives)
{
    for (auto& preprocessor : preprocessors_) {
        auto copy = directives;
        preprocessor->preprocessDirectives(directives);
        if (copy != directives) {
            YIO_LOG_INFO("Preprocessor '" << preprocessor->getPreprocessorName() << "' changed directives to: " << Directive::formatDirectives(directives));
        }
    }
}

std::shared_ptr<Directive> DirectiveProcessor::findActiveTtsDirective() const {
    for (const auto& directives : channels_) {
        if (directives.empty()) {
            continue;
        }

        const auto& directive = *directives.begin();
        if (directive->isActive() && directive->is(quasar::Directives::TTS_PLAY_PLACEHOLDER)) {
            return directive;
        }
    }

    return nullptr;
}

void DirectiveProcessor::pendingTtsContentDirective()
{
    // temporal workaround for MorningShow to not stop due to voice requests

    const auto& directives = channels_[quasar::proto::CONTENT_CHANNEL];
    if (!directives.empty()) {
        const auto& directive = directives.front();
        if (directive->isActive() && directive->is(quasar::Directives::TTS_PLAY_PLACEHOLDER)) {
            YIO_LOG_INFO("pending tts content directive");
            // cancelDirective can reset pendingDirective_
            // NOLINTNEXTLINE(performance-unnecessary-copy-initialization)
            const auto pendingDirective = directive;
            pendingDirective_ = pendingDirective;
            cancelDirective(pendingDirective_);
            pendingDirective->setIsActive(false);
        }
    }
}

std::optional<std::string> DirectiveProcessor::findFirstDirectiveNameIntersection(
    const IDirectiveHandlerPtr& handler1, const IDirectiveHandlerPtr& handler2)
{
    for (const auto& name : handler1->getSupportedDirectiveNames()) {
        if (handler2->getSupportedDirectiveNames().count(name) > 0) {
            return name;
        }
    }

    return std::nullopt;
}

void DirectiveProcessor::cancelDirective(std::shared_ptr<Directive> directive)
{
    if (currentRequestDirective_ == directive) {
        currentRequestDirective_.reset();
    }

    if (const auto& handler = findHandlerForDirective(directive)) {
        YIO_LOG_INFO("cancelDirective " << directive->format() << " via handler (" << handler.get() << "):" << handler->getHandlerName());
        handler->cancelDirective(directive);

        notifyDirectiveCompleted(directive, IDirectiveProcessorListener::Result::CANCELED);
    }
}

void DirectiveProcessor::onDirectiveStateChanged(
    const std::shared_ptr<Directive>& causeDirective)
{
    if (stateFileName_.empty() ||
        isSaveStateToFileBlocked_ ||
        !causeDirective->getData().channel.has_value()) {
        return;
    }

    saveStateToFile();
}

void DirectiveProcessor::saveStateToFile() const {
    Y_VERIFY(!stateFileName_.empty());
    Y_VERIFY(!isSaveStateToFileBlocked_);

    quasar::proto::DirectiveSequencerState stateProto;
    for (int channelIndex = quasar::proto::AudioChannel_MAX; channelIndex >= quasar::proto::AudioChannel_MIN; --channelIndex) {
        const auto& directives = channels_[channelIndex];
        if (directives.empty()) {
            continue;
        }
        auto firstDirective = *directives.begin();
        if (firstDirective->is(quasar::Directives::LISTEN) ||
            firstDirective->is(quasar::Directives::TTS_PLAY_PLACEHOLDER)) {
            continue;
        }

        quasar::proto::DirectiveSequencerState::Channel channelProto;
        channelProto.set_channel(static_cast<quasar::proto::AudioChannel>(channelIndex));
        for (const auto& directive : directives) {
            channelProto.add_directives()->CopyFrom(directive->toProtobuf());
        }

        stateProto.add_channels()->Swap(&channelProto);
    }

    quasar::TransactionFile file(stateFileName_, stateFileName_ + "_tmp");
    if (!file.write(stateProto.SerializeAsString())) {
        YIO_LOG_ERROR_EVENT("DirectiveProcessor.FailedToWriteStateToFile", "Failed to write " << stateFileName_);
    }
    if (!file.commit()) {
        YIO_LOG_ERROR_EVENT("DirectiveProcessor.FailedToCommitStateToFile", "Failed to commit " << stateFileName_);
    }
}

void DirectiveProcessor::loadStateFromFile()
{
    YIO_LOG_INFO("loadStateFromFile " << stateFileName_);

    if (stateFileName_.empty() ||
        directiveFactory_ == nullptr ||
        !quasar::fileExists(stateFileName_)) {
        return;
    }

    quasar::proto::DirectiveSequencerState stateProto;
    try {
        stateProto.ParseFromStringOrThrow(quasar::getFileContent(stateFileName_));
    } catch (const std::exception& e) {
        YIO_LOG_ERROR_EVENT("DirectiveProcessor.FailedToParseFileState", e.what());
        return;
    }

    for (const auto& channel : stateProto.channels()) {
        std::list<std::shared_ptr<Directive>> directives;
        for (const auto& directiveProto : channel.directives()) {
            if (auto directive = directiveFactory_->createDirective(directiveProto)) {
                directives.push_back(std::move(directive));
            }
        }

        if (!directives.empty()) {
            auto firstDirective = *directives.begin();
            if (firstDirective->getData().channel.has_value()) {
                channels_[channel.channel()] = std::move(directives);
            }
        }
    }

    dumpState("loadedStateFromFile", {});
}
