#pragma once

#include <yandex_io/sdk/interfaces/i_directive_factory.h>
#include <yandex_io/services/aliced/directive_processor/interface/i_directive_processor.h>
#include <yandex_io/services/aliced/directive_processor/interface/i_directive_processor_listener.h>

#include <yandex_io/protos/quasar_proto.pb.h>
#include <yandex_io/sdk/interfaces/directive.h>

#include <memory>
#include <string>
#include <unordered_set>
#include <vector>

namespace YandexIO {

    class DirectiveProcessor: public IDirectiveProcessor {
    public:
        using Channel = std::list<std::shared_ptr<Directive>>;
        using State = std::vector<Channel>;

    public:
        void start(std::string stateFileName);
        void clear();
        void setDeviceId(std::string deviceId);
        void setPrefetchEnabled(bool prefetchEnabled);
        void setDirectiveFactory(std::shared_ptr<IDirectiveFactory> directiveFactory);

        const std::vector<Channel>& getState() const;
        void addListener(std::weak_ptr<IDirectiveProcessorListener> listener);

        // IDirectiveProcessor implementation
        //
        [[nodiscard]] bool addDirectiveHandler(const IDirectiveHandlerPtr& handler) override;
        bool removeDirectiveHandler(const IDirectiveHandlerPtr& handler) override;
        IDirectiveHandlerPtr findDirectiveHandlerByName(const std::string& name) const override;
        bool addDirectivePreprocessor(const IDirectivePreprocessorPtr& value) override;
        bool removeDirectivePreprocessor(const IDirectivePreprocessorPtr& value) override;
        const std::vector<IDirectivePreprocessorPtr>& getDirectivePreprocessors() const override;
        const std::set<std::string>& getSupportedDirectiveNames() const override;
        void addDirectives(std::list<std::shared_ptr<Directive>> directives) override;
        void onHandleDirectiveStarted(std::shared_ptr<Directive> directive) override;
        void onHandleDirectiveCompleted(std::shared_ptr<Directive> directive, bool success) override;
        void onHandleDirectiveCompleted(std::shared_ptr<Directive> directive, std::list<std::shared_ptr<Directive>> directives) override;
        void onPrefetchDirectiveCompleted(std::shared_ptr<Directive> directive, std::list<std::shared_ptr<Directive>> directives) override;
        void onBlockPrefetchChanged(std::shared_ptr<Directive> directive) override;
        void enqueueDirective(std::shared_ptr<Directive> directive) override;

    private:
        void processDirectives(std::list<std::shared_ptr<Directive>> directives);
        void clearChannels();
        void clearContentChannel();
        void onDirectiveCompletedInternal(std::shared_ptr<Directive> directive);
        std::list<std::shared_ptr<Directive>>* findChainByDirective(const std::shared_ptr<Directive>& directive);

        void prefetchFirstContentDirective(const std::list<std::shared_ptr<Directive>>& directives, const std::shared_ptr<Directive>& directive);
        void prefetchLastServerActionDirective(std::shared_ptr<Directive> directive);

        void invalidatePrefetch();
        void updateAudioFocus();
        void pumpPendingDirective();
        void dumpState(const std::string& prefix, const std::list<std::shared_ptr<Directive>>& directives);
        void selectNextActivity();

        void freeChannel(quasar::proto::AudioChannel channel, bool dropChain);
        void routeDirective(std::shared_ptr<Directive> directive);
        IDirectiveHandlerPtr findHandlerForDirective(const std::shared_ptr<Directive>& directive) const;
        void pendingTtsContentDirective();
        std::shared_ptr<Directive> findActiveTtsDirective() const;

        void cancelDirective(std::shared_ptr<Directive> directive);
        void preprocessDirectives(std::list<std::shared_ptr<Directive>>& directives);
        static std::optional<std::string> findFirstDirectiveNameIntersection(const IDirectiveHandlerPtr& handler1, const IDirectiveHandlerPtr& handler2);

        void notifySequenceStateChanged();
        void notifyDialogChannelIsIdle();
        void notifyDirectiveHandled(const std::shared_ptr<Directive>& directive);
        void notifyDirectiveCompleted(const std::shared_ptr<Directive>& directive, IDirectiveProcessorListener::Result result);

        void loadStateFromFile();
        void saveStateToFile() const;
        void onDirectiveStateChanged(const std::shared_ptr<Directive>& causeDirective);

    private:
        class FlagBlocker {
        public:
            FlagBlocker(bool& block);
            ~FlagBlocker();

        private:
            bool& block_;
        };

    private:
        std::list<std::weak_ptr<IDirectiveProcessorListener>> listeners_;
        std::vector<Channel> channels_{quasar::proto::AudioChannel_ARRAYSIZE};

        // Use this member to enqueue directives executed by chain rules
        // until current alice request is completed
        std::shared_ptr<Directive> pendingDirective_;
        bool prefetchEnabled_ = false;
        std::string stateFileName_;
        std::shared_ptr<IDirectiveFactory> directiveFactory_;

        std::deque<std::shared_ptr<Directive>> requestQueue_;
        std::shared_ptr<Directive> currentRequestDirective_;

        std::set<IDirectiveHandlerPtr> directiveHandlers_;
        std::vector<IDirectivePreprocessorPtr> preprocessors_;
        std::set<std::string> directiveNames_;
        std::string deviceId_;

        bool isSelectActivityBlocked_ = false;
        bool isSaveStateToFileBlocked_ = false;
    };

} // namespace YandexIO
