#pragma once

#include "i_directive_preprocessor.h"

#include <yandex_io/sdk/interfaces/i_directive_handler.h>

#include <list>
#include <memory>
#include <set>
#include <string>
#include <vector>

namespace YandexIO {

    class IDirectiveProcessor {
    public:
        virtual ~IDirectiveProcessor();

        // Adds directive handler
        // @return true if handler was added, false otherwise
        [[nodiscard]] virtual bool addDirectiveHandler(const IDirectiveHandlerPtr& handler) = 0;

        // Removes directive handler
        // @return true if handler was removed, false otherwise
        virtual bool removeDirectiveHandler(const IDirectiveHandlerPtr& handler) = 0;

        // Adds preprocessor to the end of preprocessors list
        // @return true if preprocessor was added, false otherwise
        virtual bool addDirectivePreprocessor(const IDirectivePreprocessorPtr& value) = 0;

        // Removes preprocessor from the preprocessors list
        // @return true if preprocessor was added, false otherwise
        virtual bool removeDirectivePreprocessor(const IDirectivePreprocessorPtr& value) = 0;

        // @return list of preprocessors
        virtual const std::vector<IDirectivePreprocessorPtr>& getDirectivePreprocessors() const = 0;

        // @return set of supported directive names
        virtual const std::set<std::string>& getSupportedDirectiveNames() const = 0;

        virtual IDirectiveHandlerPtr findDirectiveHandlerByName(const std::string& name) const = 0;

        virtual void addDirectives(std::list<std::shared_ptr<Directive>> directives) = 0;
        virtual void onHandleDirectiveStarted(std::shared_ptr<Directive> directive) = 0;
        virtual void onHandleDirectiveCompleted(std::shared_ptr<Directive> directive, bool success) = 0;
        virtual void onHandleDirectiveCompleted(std::shared_ptr<Directive> directive, std::list<std::shared_ptr<Directive>> directives) = 0;
        virtual void onPrefetchDirectiveCompleted(std::shared_ptr<Directive> directive, std::list<std::shared_ptr<Directive>> directives) = 0;
        virtual void onBlockPrefetchChanged(std::shared_ptr<Directive> directive) = 0;
        virtual void enqueueDirective(std::shared_ptr<Directive> directive) = 0;
    };

    using IDirectiveProcessorPtr = std::shared_ptr<IDirectiveProcessor>;
    using IDirectiveProcessorWeakPtr = std::weak_ptr<IDirectiveProcessor>;

} // namespace YandexIO
