#include "speechkit_endpoint.h"

#include "capabilities/alice_capability/directives/alice_request_directive.h"
#include "capabilities/alice_capability/preprocessors/alice_capability_preprocessor.h"
#include "capabilities/alice_capability/vins/vins_response.h"
#include "capabilities/alice_capability/vins/vins_utils.h"
#include "capabilities/bluetooth_capability/preprocessors/bluetooth_capability_preprocessor.h"

#include <yandex_io/services/aliced/capabilities/alarm_capability/alarm_capability.h>
#include <yandex_io/services/aliced/capabilities/alice_capability/audio_client_tts_player.h>
#include <yandex_io/services/aliced/capabilities/alice_capability/speechkit_tts_player.h>
#include <yandex_io/services/aliced/capabilities/audio_player_capability/audio_player_preprocessor.h>
#include <yandex_io/services/aliced/capabilities/bluetooth_capability/bluetooth_capability.h>
#include <yandex_io/services/aliced/capabilities/external_command_capability/external_command_capability.h>
#include <yandex_io/services/aliced/capabilities/file_player_capability/file_player_capability.h>
#include <yandex_io/services/aliced/capabilities/multiroom_capability/multiroom_preprocessor.h>
#include <yandex_io/services/aliced/capabilities/playback_control_capability/playback_control_capability.h>
#include <yandex_io/services/aliced/capabilities/stereo_pair_capability/stereo_pair_capability.h>
#include <yandex_io/services/aliced/capabilities/stereo_pair_capability/stereo_pair_preprocessor.h>
#include <yandex_io/services/aliced/capabilities/testpoint_capability/testpoint_preprocessor.h>
#include <yandex_io/services/aliced/capabilities/legacy_iot_capability/legacy_iot_capability.h>

#include <yandex_io/services/audiosender/info/info.h>

#include <yandex_io/capabilities/device_state/converters/converters.h>

#include <yandex_io/libs/base/directives.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/cryptography/digest.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/ete_metrics/ete_util.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/protobuf_utils/debug.h>
#include <yandex_io/libs/spotter_types/spotter_types.h>
#include <yandex_io/libs/telemetry/telemetry.h>
#include <yandex_io/protos/enum_names/enum_names.h>

#include <json/json.h>

#include <speechkit/Error.h>
#include <speechkit/UniProxy.h>

#include <util/system/yassert.h>

#include <algorithm>
#include <csignal>
#include <cstdlib>
#include <limits>
#include <optional>
#include <unordered_map>

YIO_DEFINE_LOG_MODULE("alice");

using namespace quasar;
using namespace YandexIO;

const std::string SpeechkitEndpoint::SERVICE_NAME = "aliced";
const std::string SpeechkitEndpoint::GET_NEXT_REQUEST_FAILED_KEY = "getNextRequestFailed";

SpeechkitEndpoint::SpeechkitEndpoint(std::shared_ptr<YandexIO::IDevice> device,
                                     NAlice::TEndpoint::EEndpointType endpointType,
                                     std::shared_ptr<YandexIO::SDKInterface> sdk,
                                     std::shared_ptr<ipc::IIpcFactory> ipcFactory,
                                     std::shared_ptr<IAuthProvider> authProvider,
                                     std::shared_ptr<IClockTowerProvider> clockTowerProvider,
                                     std::shared_ptr<IDeviceStateProvider> deviceStateProvider,
                                     std::shared_ptr<IGlagolClusterProvider> glagolClusterProvider,
                                     std::shared_ptr<IMultiroomProvider> multiroomProvider,
                                     std::shared_ptr<IStereoPairProvider> stereoPairProvider,
                                     std::shared_ptr<IUserConfigProvider> userConfigProvider,
                                     SpeechKit::AudioPlayer::SharedPtr audioPlayer,
                                     std::shared_ptr<AliceAudioSource> audioSource,
                                     std::shared_ptr<YandexIO::IAudioSourceClient> audioSourceClient,
                                     std::shared_ptr<VoiceStats> voiceStats,
                                     std::shared_ptr<RandomSoundLogger> randomSoundLogger,
                                     std::shared_ptr<SelfDestroyer> selfDestroyer,
                                     std::shared_ptr<ICallbackQueue> callbackExecutor,
                                     std::unique_ptr<quasar::IBackoffRetries> backoffer,
                                     AliceConfig& aliceConfig,
                                     std::shared_ptr<AliceDeviceState> aliceDeviceState)
    : selfDestroyer_(std::move(selfDestroyer))
    , asyncQueue_(std::move(callbackExecutor))
    , device_(std::move(device))
    , sdk_(std::move(sdk))
    , ipcFactory_(std::move(ipcFactory))
    , authProvider_(std::move(authProvider))
    , clockTowerProvider_(std::move(clockTowerProvider))
    , deviceStateProvider_(std::move(deviceStateProvider))
    , glagolClusterProvider_(std::move(glagolClusterProvider))
    , multiroomProvider_(std::move(multiroomProvider))
    , stereoPairProvider_(std::move(stereoPairProvider))
    , userConfigProvider_(std::move(userConfigProvider))
    , toMediad_(ipcFactory_->createIpcConnector("mediad"))
    , toSyncd_(ipcFactory_->createIpcConnector("syncd"))
    , toWifid_(ipcFactory_->createIpcConnector("wifid"))
    , toNetworkd_(ipcFactory_->createIpcConnector("networkd"))
    , toBrickd_(ipcFactory_->createIpcConnector("brickd"))
    , toAudioClientd_(ipcFactory_->createIpcConnector("audioclient"))
    , toCalld_(ipcFactory_->createIpcConnector("calld"))
    , toIot_(ipcFactory_->createIpcConnector("iot"))
    , toDoNotDisturb_(ipcFactory_->createIpcConnector("do_not_disturb"))
    , toInterfaced_(ipcFactory_->createIpcConnector("interfaced"))
    , toBugReport_(ipcFactory_->createIpcConnector("bug_report"))
    , toNotificationd_(ipcFactory_->createIpcConnector("notificationd"))
    , server_(ipcFactory_->createIpcServer("aliced"))
    , audioPlayer_(std::move(audioPlayer))
    , jingleCustomPlayer_(std::make_shared<JingleAudioClientPlayer>(toAudioClientd_))
    , audioSource_(std::move(audioSource))
    , audioSourceClient_(std::move(audioSourceClient))
    , voiceStats_(std::move(voiceStats))
    , randomSoundLogger_(std::move(randomSoundLogger))
    , brokenMicInfoLogger_(device_->telemetry())
    , connectionStatsSender_(device_->telemetry(), "uniproxy")
    , connectionsStateHolder_({"mediad", "bug_report", "notificationd", "calld", "interfaced", "iot", "glagold"})
    , featuresConfig_(device_->configuration()->getServiceConfig("aliced"))
    , aliceConfig_(aliceConfig)
    , deviceState_(std::move(aliceDeviceState))
    , deviceContext_(ipcFactory_, nullptr /* onConnected*/, false /* autoConnect*/)
    , audioFocusDispatcher_(toMediad_, toInterfaced_, toAudioClientd_)
{
    deviceState_->getEnvironmentState().environmentChangedCallback_ = [this]() {
        asyncQueue_->add([this]() {
            broadcastEnvironmentDeviceInfo();
        });
    };

    deviceController_ = std::make_shared<YandexIO::DeviceController>(
        asyncQueue_, aliceConfig_, std::make_shared<LegacyIotCapability>(toIot_),
        std::move(backoffer));

    deviceController_->init(device_->deviceId(), endpointType);

    getDirectiveProcessor()->setDeviceId(device_->deviceId());
    deviceState_->getEnvironmentState().setEndpointStorage(deviceController_->getEndpointStorage());

    const auto speechkitDumpPath = aliceConfig_.getSpeechkitDumpPath();
    if (!speechkitDumpPath.empty()) {
        SpeechKit::SpeechKit::getInstance()->setDumpPath(speechkitDumpPath);
    }

    for (const auto& serviceName : {"interfaced", "calld", "iot"}) {
        if (!device_->configuration()->hasServiceConfig(serviceName)) {
            connectionsStateHolder_.deleteService(serviceName);
        }
    }
}

SpeechkitEndpoint::~SpeechkitEndpoint()
{
    lifetime_.die();

    server_->shutdown();

    toMediad_->shutdown();
    toSyncd_->shutdown();
    toWifid_->shutdown();
    toNetworkd_->shutdown();
    toBrickd_->shutdown();
    toAudioClientd_->shutdown();
    toCalld_->shutdown();
    toIot_->shutdown();
    toDoNotDisturb_->shutdown();
    if (hasInterfaced()) {
        toInterfaced_->shutdown();
    }
    toBugReport_->shutdown();
    toNotificationd_->shutdown();

    voiceDialog_.reset();
    pendingVinsResponse_.reset();

    activityTracker_.clear();
    getDirectiveProcessor()->clear();
    aliceCapability_.reset();
    multiroomCapability_.reset();
    mrForwarderCapability_.reset();
    audioPlayerCapability_.reset();
    alarmCapability_.reset();
    filePlayerCapability_.reset();
    screenCapability_.reset();
    tandemCapability_.reset();
    deviceController_.reset();
}

void SpeechkitEndpoint::start(std::shared_ptr<QuasarVoiceDialog> voiceDialog)
{
    asyncQueue_->add([this, voiceDialog]() mutable {
        voiceDialog_ = std::move(voiceDialog);
        voiceDialog_->update(aliceConfig_);
        voiceDialog_->setPlayer(getDialogPlayer());

        deviceState_->getEnvironmentState().updateLocalDevice();
        voiceStats_->setRMSCorrection(aliceConfig_.getRMSCorrection());
        voiceStats_->calcOnVqe(aliceConfig_.getRMSOverVqe());

        initConnectors();
        initCapabilities();
        updateDialogSettings();
        getDirectiveProcessor()->addListener(weak_from_this());

        const auto startMoment{std::chrono::steady_clock::now()};
        device_->telemetry()->requestUUID([this, startMoment](const std::string& uuid) {
            asyncQueue_->add([this, uuid, startMoment] {
                const auto delay{std::chrono::duration_cast<std::chrono::milliseconds>(std::chrono::steady_clock::now() - startMoment)};
                onMetricaUuidReceived(uuid, delay);
            });
        });

        asyncQueue_->addDelayed([this] {
            onStartDialogTimeoutExpired();
        }, aliceConfig_.getStartDelay());
    });
}

void SpeechkitEndpoint::onMetricaUuidReceived(const std::string& uuid, std::chrono::milliseconds delay)
{
    YIO_LOG_INFO("onMetricaUuidReceived");
    deviceState_->setMetricaUuid(uuid);

    Json::Value args;
    args["delay_ms"] = Json::Value{static_cast<Json::Int64>(delay.count())};
    device_->telemetry()->reportEvent("MetricaUUIDReceivingDelay", jsonToString(args));

    startVoiceInteraction();
}

void SpeechkitEndpoint::onStartDialogTimeoutExpired() {
    YIO_LOG_INFO("onStartDialogTimeoutExpired");
    if (dialogStarted_) {
        return;
    }

    const auto delaySeconds = aliceConfig_.getStartDelay().count();
    YIO_LOG_INFO("Dialog is not started after " << delaySeconds << " seconds. Forcing startVoiceInteraction()");

    Json::Value args;
    args["timeout_ms"] = Json::Value{static_cast<Json::Int>(delaySeconds * 1000)};
    device_->telemetry()->reportEvent("MetricaUUIDReceivingTimeout", jsonToString(args));

    firstDialogStartTimerExpired_ = true;
    startVoiceInteraction();
}

void SpeechkitEndpoint::initConnectors()
{
    deviceContext_.onVinsResponse = [this](std::string vinsResponse) {
        asyncQueue_->add([this, vinsResponse{std::move(vinsResponse)}]() mutable {
            onDeviceContextVinsResponse(std::move(vinsResponse));
        });
    };

    deviceContext_.onTvPolicyInfoReceived = [this](YandexIO::TvPolicyInfo tvPolicyInfo) {
        asyncQueue_->add([this, tvPolicyInfo{std::move(tvPolicyInfo)}]() mutable {
            deviceState_->setTvPolicyInfo(std::move(tvPolicyInfo));
        });
    };

    deviceContext_.onClockDisplayStateReceived = [this](YandexIO::ClockDisplayState clockDisplayState) {
        asyncQueue_->add([this, clockDisplayState]() mutable {
            deviceState_->setClockDisplayState(clockDisplayState);
        });
    };

    deviceContext_.onActiveActionReceived = [this](const NAlice::TDeviceState::TActiveActions& activeActions) {
        asyncQueue_->add([this, activeActions] {
            deviceState_->setActiveActions(activeActions);
        });
    };

    deviceContext_.onActiveActionSemanticFrameReceived = [this](const std::optional<std::string>& activeActionPayload) {
        asyncQueue_->add([this, activeActionPayload] {
            deviceState_->setActiveActionSemanticFrame(activeActionPayload);
        });
    };

    deviceContext_.onAssistantBlocking = [this](const std::string& source, const std::optional<std::string>& errorSound) {
        asyncQueue_->add([this, source, errorSound] {
            YIO_LOG_INFO("Block voice assistant from: " << source << ", error sound: " << errorSound.value_or("<null>"));
            bool isBlockedNow = aliceCapability_->getAliceBlockers().isBlocked();
            aliceCapability_->getAliceBlockers().block(source, errorSound);
            if (!isBlockedNow) {
                if (aliceCapability_->hasInteraction()) {
                    YIO_LOG_INFO("Stopping dialog because of block message from: " << source);
                    aliceCapability_->cancelDialog();
                }
            }
        });
    };

    deviceContext_.onAssistantUnblocking = [this](const std::string& source) {
        asyncQueue_->add([this, source] {
            YIO_LOG_INFO("Unblock voice assistant from: " << source);
            aliceCapability_->getAliceBlockers().unblock(source);
        });
    };

    deviceContext_.onTimezone = [this](const proto::Timezone& timezone) {
        asyncQueue_->add([this, timezone]() {
            onTimezone(timezone);
        });
    };

    deviceContext_.onLocation = [this](const proto::Location& location) {
        asyncQueue_->add([this, location]() {
            onLocation(location);
        });
    };

    deviceContext_.onWifiList = [this](const proto::WifiList& wifiList) {
        asyncQueue_->add([this, wifiList]() {
            onWifiList(wifiList);
        });
    };

    deviceContext_.onMediaDeviceIdentifier = [this](const NAlice::TClientInfoProto::TMediaDeviceIdentifier& identifier) {
        asyncQueue_->add([this, identifier]() {
            onMediaDeviceIdentifier(identifier);
        });
    };

    deviceContext_.onDeviceStatePart = [this](const yandex_io::proto::TDeviceStatePart& statePart) {
        asyncQueue_->add([this, statePart]() {
            onDeviceStatePart(statePart);
        });
    };

    deviceContext_.onIsScreenActive = [this](bool isActive) {
        asyncQueue_->add([this, isActive]() {
            if (screenCapability_) {
                screenCapability_->onIsScreenActive(isActive);
            } else {
                deviceStateCapability_->setIsTvPluggedIn(isActive);
            }
        });
    };

    deviceContext_.connectToSDK();

    server_->setMessageHandler([this](const auto& message, auto& connection) {
        asyncQueue_->add([this, message, connection = connection.share()]() mutable {
            handleQuasarMessage(message, connection);
        });
    });

    server_->setClientConnectedHandler([this](auto& connection) {
        asyncQueue_->add([this, connection = connection.share()]() mutable {
            proto::QuasarMessage msg;
            *msg.mutable_alice_state() = aliceCapability_->getAliceState();
            connection->send(std::move(msg));

            broadcastEnvironmentDeviceInfo();
            broadcastAppState(deviceStateCapability_->getAppState());
        });
    });

    server_->listenService();

    auto onQuasarMessage = [this](const ipc::SharedMessage& message) {
        asyncQueue_->add([this, message] {
            handleQuasarMessage(message, nullptr);
        });
    };
    toSyncd_->setMessageHandler(onQuasarMessage);
    toBrickd_->setMessageHandler(onQuasarMessage);
    toCalld_->setMessageHandler(onQuasarMessage);
    toWifid_->setMessageHandler(onQuasarMessage);
    toNetworkd_->setMessageHandler(onQuasarMessage);
    toIot_->setMessageHandler(onQuasarMessage);
    toDoNotDisturb_->setMessageHandler(onQuasarMessage);
    toBugReport_->setMessageHandler(onQuasarMessage);
    toNotificationd_->setMessageHandler(onQuasarMessage);

    toMediad_->setConnectHandler([this] {
        asyncQueue_->add([this] {
            handleServiceReady("mediad");
            audioFocusDispatcher_.onMediadConnected();
        });
    });
    toMediad_->setMessageHandler([this](const auto& message) {
        asyncQueue_->add([this, message]() {
            handleMediadMessage(message);
        });
    });

    toAudioClientd_->setConnectHandler([this]() {
        asyncQueue_->add([this]() mutable {
            handleAudioClientConnect();
        });
    });
    toAudioClientd_->setMessageHandler([this](const auto& message) {
        asyncQueue_->add([this, message] {
            handleAudioClientMessage(message);
        });
    });
    toAudioClientd_->setConnectionErrorHandler([this](const std::string& /*error*/) {
        asyncQueue_->add([this]() {
            handleAudioClientDisconnect();
        });
    });

    toBugReport_->setConnectHandler([this]() {
        asyncQueue_->add([this] { handleServiceReady("bug_report"); });
    });
    toCalld_->setConnectHandler([this]() {
        asyncQueue_->add([this] { handleServiceReady("calld"); });
    });
    toIot_->setConnectHandler([this]() {
        asyncQueue_->add([this] { handleServiceReady("iot"); });
    });
    toNotificationd_->setConnectHandler([this]() {
        asyncQueue_->add([this] { handleServiceReady("notificationd"); });
    });

    if (hasInterfaced()) {
        toInterfaced_->setConnectHandler([this]() {
            asyncQueue_->add([this]() mutable {
                handleServiceReady("interfaced");
                audioFocusDispatcher_.onInterfacedConnected();
            });
        });
        toInterfaced_->setMessageHandler([this](const auto& message) {
            asyncQueue_->add([this, message]() {
                handleInterfacedMessage(message);
            });
        });
    }

    toGlagold_ = ipcFactory_->createIpcConnector("glagold");
    toGlagold_->setMessageHandler([this](const auto& message) {
        asyncQueue_->add([this, message] {
            onGlagoldMessage(message);
        });
    });
    toGlagold_->setConnectHandler([this] {
        asyncQueue_->add([this] { handleServiceReady("glagold"); });
    });

    if (device_->configuration()->hasServiceConfig("videod")) {
        connectionsStateHolder_.addService("videod");
        toVideod_ = ipcFactory_->createIpcConnector("videod");
        toVideod_->setMessageHandler(onQuasarMessage);
        toVideod_->setConnectHandler([this] {
            asyncQueue_->add([this] { handleServiceReady("videod"); });
        });
        toVideod_->connectToService();
    }

    toMediad_->connectToService();
    toSyncd_->connectToService();
    toWifid_->connectToService();
    toNetworkd_->connectToService();
    toGlagold_->connectToService();
    toDoNotDisturb_->connectToService();
    toBrickd_->tryConnectToService();
    toAudioClientd_->tryConnectToService();
    toCalld_->tryConnectToService();
    toIot_->tryConnectToService();
    if (hasInterfaced()) {
        toInterfaced_->connectToService();
    }
    toBugReport_->connectToService();
    toNotificationd_->connectToService();

    if (stereoPairProvider_) {
        stereoPairProvider_->stereoPairState().connect(
            [this](const auto& state) {
                const bool oldIsFollower = deviceState_->getStereoPairState()->isFollower();
                deviceState_->setStereoPairState(state);
                const bool newIsFollower = deviceState_->getStereoPairState()->isFollower();

                if (oldIsFollower != newIsFollower) {
                    YIO_LOG_INFO("Switch stereo pair follower flag from " << oldIsFollower << " to " << newIsFollower);
                    if (newIsFollower) {
                        aliceCapability_->cancelDialog();
                    }
                }
            }, lifetime_, asyncQueue_);
    }

    authProvider_->ownerAuthInfo().connect(
        [this](const auto& authInfo) {
            if (deviceState_->getOAuthToken() != authInfo->authToken ||
                deviceState_->getPassportUid() != authInfo->passportUid) {
                YIO_LOG_INFO("oauthToken or passportUid changed");
                deviceState_->setOAuthToken(authInfo->authToken);
                deviceState_->setPassportUid(authInfo->passportUid);

                if (!authInfo->passportUid.empty()) {
                    playbackCapability_->onPassportUidChanged();
                }

                aliceCapability_->cancelDialogAndClearQueue();
                updateDialogSettings();
                startVoiceInteraction();
            }
        }, lifetime_, asyncQueue_);

    deviceStateProvider_->configurationChangedSignal().connect(
        [this](const auto& deviceState)
        {
            if (tandemCapability_ != nullptr) {
                tandemCapability_->onDeviceStateChanged(deviceState);
            }

            if (deviceState->configuration != DeviceState::Configuration::CONFIGURED) {
                aliceCapability_->cancelDialogAndClearQueue();
            }
        }, lifetime_, asyncQueue_);
}

void SpeechkitEndpoint::onPhraseSpotterBegin(SpeechKit::VoiceService::SharedPtr /*voiceDialog*/) {
    asyncQueue_->add([this] {
        YIO_LOG_INFO("onPhraseSpotterBegin");

        proto::QuasarMessage quasarMessage;
        quasarMessage.mutable_speech_kit_event()->mutable_on_phrase_spotter_begin();
        server_->sendToAll(std::move(quasarMessage));

        naviOldSpotterCapability_->resume();

        if (aliceConfig_.getSpottersEnabled()) {
            onSpottingStarted();
        }
    });
}

void SpeechkitEndpoint::onPhraseSpotted(SpeechKit::VoiceService::SharedPtr /* voiceDialog */, const std::string& phrase)
{
    asyncQueue_->add([this, phrase] {
        YIO_LOG_INFO("==========================");
        YIO_LOG_INFO("onPhraseSpotted: " << phrase);
        device_->telemetry()->reportEvent("phraseSpotted");

        aliceCapability_->startVoiceInput(VinsRequest::createSoftwareSpotterEventSource());
    });
}

void SpeechkitEndpoint::onPhraseSpotterError(SpeechKit::VoiceService::SharedPtr /* voiceDialog */, const SpeechKit::Error& error)
{
    asyncQueue_->add([this, error] {
        YIO_LOG_ERROR_EVENT("SpeechkitEndpoint.PhraseSpotterError", "onPhraseSpotterError: " << error.getString());

        /*
         * Нет смысла перезапускать диалог, если у нас ошибка в модели споттера. Это невосстанавливаемая ошибка, а значит
         * он будет падать снова и снова.
         */
        if (error.isSpotterModelError()) {
            activationSpotterCapability_->onModelError();
        } else {
            aliceCapability_->cancelDialog();
        }
    });
}

void SpeechkitEndpoint::onInterruptionPhraseSpotted(
    SpeechKit::VoiceService::SharedPtr /* voiceDialog */, const std::string& phrase)
{
    asyncQueue_->add([this, phrase] {
        YIO_LOG_INFO("=======================================");
        YIO_LOG_INFO("onInterruptionPhraseSpotted: " << phrase);

        aliceCapability_->startVoiceInput(VinsRequest::createSoftwareSpotterEventSource());
    });
}

void SpeechkitEndpoint::onCommandSpotterBegin(SpeechKit::VoiceService::SharedPtr voiceDialog)
{
    Y_UNUSED(voiceDialog);
    asyncQueue_->add([this] {
        commandSpotterCapability_->onCommandSpotterBegin();
    });
}

void SpeechkitEndpoint::onCommandPhraseSpotted(SpeechKit::VoiceService::SharedPtr voiceDialog, const std::string& phrase)
{
    Y_UNUSED(voiceDialog);
    asyncQueue_->add([this, phrase] {
        commandSpotterCapability_->onCommandPhraseSpotted(phrase);
    });
}

void SpeechkitEndpoint::onCommandSpotterError(SpeechKit::VoiceService::SharedPtr voiceDialog, const SpeechKit::Error& error)
{
    Y_UNUSED(voiceDialog);
    asyncQueue_->add([this, error] {
        commandSpotterCapability_->onCommandSpotterError(error);
    });
}

void SpeechkitEndpoint::onRecognitionBegin(SpeechKit::VoiceService::SharedPtr /* voiceDialog */, const std::string& /*requestId*/)
{
}

void SpeechkitEndpoint::onRecognitionResults(
    SpeechKit::VoiceService::SharedPtr /* voiceDialog */,
    const std::string& requestId,
    const SpeechKit::Recognition& recognition,
    bool endOfUtterance) {
    asyncQueue_->add([this, requestId, recognition, endOfUtterance] {
        aliceCapability_->onRecognitionResults(requestId, recognition, endOfUtterance);
    });
}

void SpeechkitEndpoint::onRecognitionEnd(SpeechKit::VoiceService::SharedPtr /* voiceDialog */, const std::string& /*requestId*/) {
    // do nothing
}

void SpeechkitEndpoint::onRecognitionError(SpeechKit::VoiceService::SharedPtr /* voiceDialog */, const std::string& requestId, const SpeechKit::Error& error) {
    asyncQueue_->add([this, requestId, error] {
        auto request = aliceCapability_->findAliceRequest(requestId);
        if (request == nullptr) {
            YIO_LOG_INFO("onRecognitionError: request with id='" << requestId << "' not found");
            return;
        }
        YIO_LOG_ERROR_EVENT("SpeechkitEndpoint.RecognitionError", "requestId=" << requestId << ", errorText: " << error.getString() << ", error.code() = " << error.getCode());

        if (error.getCode() == SpeechKit::Error::ErrorNetwork || error.getCode() == SpeechKit::Error::ErrorServer ||
            error.getCode() == SpeechKit::Error::ErrorPongTimeoutElapsed || error.getCode() == SpeechKit::Error::ErrorTimeout) {
            deviceContext_.fireConversationError();
            if (error.getCode() == SpeechKit::Error::ErrorServer) {
                playSound("vins_error.wav");
                YIO_LOG_INFO("RecognitionError on server error");

                Json::Value speechkitError;
                speechkitError["code"] = error.getCode();
                speechkitError["message"] = error.getString();

                device_->telemetry()->reportEvent("phraseSpottedWithServerError", jsonToString(speechkitError));
            } else {
                // ErrorTimeout - is a grey area for error. It can be Network error or Server error.
                // Notify user about "server" error if Wifi/Ethernet is connected. But it is not always true
                if (error.getCode() == SpeechKit::Error::ErrorTimeout && deviceState_->getNetworkStatus().status() == proto::NetworkStatus::CONNECTED) {
                    playSound("vins_error.wav");
                } else {
                    playSound("no_internet.wav");
                }

                YIO_LOG_INFO("RecognitionError without internet");
                sendMetricaPhraseSpottedNoInternet(error);
            }
        }

        aliceCapability_->onRequestError(request, error);
    });
}

void SpeechkitEndpoint::sendMetricaPhraseSpottedNoInternet(const SpeechKit::Error& error) const {
    Json::Value speechkitError;
    speechkitError["code"] = error.getCode();
    speechkitError["message"] = error.getString();

    Json::Value networkAdditionalInfo;
    networkAdditionalInfo["speechkitError"] = speechkitError;
    networkAdditionalInfo["wifiStatus"] = networkStatusStatusName(deviceState_->getNetworkStatus().status());
    std::string networkType = "none";
    if (deviceState_->getNetworkStatus().has_wifi_status()) {
        networkType = "wifi";
    } else if (deviceState_->getNetworkStatus().has_ethernet_status()) {
        networkType = "ethernet";
    }
    networkAdditionalInfo["networkType"] = networkType;

    device_->telemetry()->reportEvent("phraseSpottedWithoutInternet", jsonToString(networkAdditionalInfo));
}

void SpeechkitEndpoint::onVinsRequestBegin(SpeechKit::VoiceService::SharedPtr /* voiceDialog */, const std::string& requestId) {
    asyncQueue_->add([this, requestId] {
        aliceCapability_->onVinsRequestBegin(requestId);
    });
}

void SpeechkitEndpoint::onVinsResponse(
    SpeechKit::VoiceService::SharedPtr /* voiceDialog */,
    const ::SpeechKit::VinsResponse& response,
    SpeechKit::TTSDataProvider::SharedPtr ttsDataProvider)
{
    asyncQueue_->add([this, response, ttsDataProvider{std::move(ttsDataProvider)}] {
        auto request = aliceCapability_->findAliceRequest(response.vinsRequestId);
        if (request == nullptr) {
            YIO_LOG_INFO("onVinsResponse: request with id='" << response.vinsRequestId << "' not found");
            return;
        }
        YIO_LOG_INFO("onVinsResponse: header=" << response.jsonHeader << ", requestId=" << response.vinsRequestId);

        try {
            auto jsonPayload = response.jsonPayload;
            if (aliceConfig_.getPreprocessVinsResponse()) {
                jsonPayload = deviceContext_.preprocessVinsResponse(jsonPayload);
            }

            Json::Value payload = parseJson(jsonPayload);
            const auto header = SpeechKit::UniProxy::Header::fromJson(response.jsonHeader);
            request->setMessageId(header.refMessageId);

            std::shared_ptr<IPlayer> ttsPlayer;
            if (ttsDataProvider) {
                if (aliceConfig_.getUseAudioClientTtsPlayer() && toAudioClientd_->isConnected()) {
                    ttsPlayer = std::make_shared<AudioClientTtsPlayer>(asyncQueue_, ttsDataProvider, toAudioClientd_, aliceConfig_.getSilenceTimeAfterTts(), request);
                } else {
                    ttsPlayer = std::make_shared<SpeechkitTtsPlayer>(asyncQueue_, audioPlayer_, ttsDataProvider, request);
                }
            }

            auto vinsResponse = VinsResponse::parse(request, header, payload, ttsPlayer, aliceConfig_.getBlockShowCardDirectiveNames());

            auto event = makeETEEvent(header.refMessageId);
            device_->telemetry()->reportEvent("vinsResponse", jsonToString(event));

            if (vinsResponse.hasMegamindError && request->isGetNext()) {
                device_->telemetry()->reportEvent(GET_NEXT_REQUEST_FAILED_KEY);
            }

            aliceCapability_->onRequestCompleted(request, std::move(vinsResponse), payload);
        } catch (const std::exception& e) {
            YIO_LOG_ERROR_EVENT("SpeechkitEndpoint.FailedVinsResponse", e.what());
            aliceCapability_->onRequestError(request, SpeechKit::Error(SpeechKit::Error::ErrorClient, e.what()));
        }
    });
}

void SpeechkitEndpoint::onVinsError(SpeechKit::VoiceService::SharedPtr /* voiceDialog */, const std::string& requestId, const SpeechKit::Error& error)
{
    asyncQueue_->add([this, requestId, error] {
        auto request = aliceCapability_->findAliceRequest(requestId);
        if (request == nullptr) {
            YIO_LOG_INFO("onVinsError: request with id='" << requestId << "' not found");
            return;
        }
        YIO_LOG_ERROR_EVENT("SpeechkitEndpoint.VinsError", "requestId=" << requestId << ", error: " << error.getString());

        if (error.getCode() == SpeechKit::Error::ErrorSpottingRejected) {
            device_->telemetry()->reportEvent("errorSpotingRejected");
        } else {
            device_->telemetry()->reportEvent("vinsError");
            if (!request->getIsSilent() && !request->getIsParallel()) {
                deviceContext_.fireConversationError();
                if (error.getCode() == SpeechKit::Error::ErrorNetwork ||
                    error.getCode() == SpeechKit::Error::ErrorTimeout && deviceState_->getNetworkStatus().status() == proto::NetworkStatus::CONNECTED) {
                    playSound("no_internet.wav");
                } else {
                    playSound("vins_error.wav");
                }
            }
        }

        if (request->isGetNext()) {
            device_->telemetry()->reportEvent(GET_NEXT_REQUEST_FAILED_KEY);
        }

        aliceCapability_->onRequestError(request, error);
    });
}

void SpeechkitEndpoint::onConnectionStateChanged(SpeechKit::VoiceService::SharedPtr /* voiceDialog */, bool isConnected) {
    YIO_LOG_DEBUG("onConnectionStateChanged, updating local device in environment state. connected: " << isConnected);
    asyncQueue_->add([this, isConnected] {
        connectionStatsSender_.onConnectionStateChanged(isConnected);
        deviceState_->getEnvironmentState().updateLocalDevice();
    });
}

void SpeechkitEndpoint::onInvalidOAuthToken(SpeechKit::VoiceService::SharedPtr /* voiceService */)
{
    asyncQueue_->add([this] {
        authProvider_->requestAuthTokenUpdate("SpeechkitEndpoint onInvalidOAuthToken");
    });
}

void SpeechkitEndpoint::startSpotting()
{
    YIO_LOG_INFO("startSpotting");

    if (aliceConfig_.getSpottersEnabled() && !deviceState_->getStereoPairState()->isFollower()) {
        activationSpotterCapability_->start();
    } else {
        onSpottingStarted();
    }

    reportAllStartupSettings();
}

void SpeechkitEndpoint::onSequenceStateChanged()
{
    if (!audioFocusDispatcher_.getAudioFocus().has_value() ||
        audioFocusDispatcher_.getAudioFocus() != activityTracker_.getCurrentFocusChannel()) {
        const auto newFocus = activityTracker_.getCurrentFocusChannel().value_or(proto::CONTENT_CHANNEL);
        audioFocusDispatcher_.setAudioFocus(newFocus);
    }
}

void SpeechkitEndpoint::onDialogChannelIsIdle()
{
    YIO_LOG_INFO("onDialogChannelIsIdle");

    startSpotting();
}

void SpeechkitEndpoint::onDirectiveHandled(const std::shared_ptr<YandexIO::Directive>& directive)
{
    if (directive == nullptr) {
        YIO_LOG_ERROR_EVENT("SpeechkitEndpoint.OnDirectiveHandled", "null directive handled");
        return;
    }

    const auto& endpointId = directive->getData().endpointId;
    if (endpointId.empty() || endpointId == device_->deviceId()) {
        const auto& data = directive->getData();
        deviceContext_.fireExternalCommand(data.name, data.requestId, jsonToString(data.payload));
    }
}

void SpeechkitEndpoint::onDirectiveStarted(const std::shared_ptr<YandexIO::Directive>& directive) {
    Y_UNUSED(directive);
}

void SpeechkitEndpoint::onDirectiveCompleted(const std::shared_ptr<YandexIO::Directive>& directive, YandexIO::IDirectiveProcessorListener::Result result) {
    Y_UNUSED(directive);
    Y_UNUSED(result);
}

void SpeechkitEndpoint::playSound(const std::string& wavName)
{
    filePlayerCapability_->playSoundFile(wavName, proto::DIALOG_CHANNEL);
}

void SpeechkitEndpoint::onUniProxyDirective(
    SpeechKit::VoiceService::SharedPtr voiceDialog, const std::string& jsonHeader, const std::string& jsonPayload) {
    asyncQueue_->add([this, voiceDialog, jsonHeader, jsonPayload] {
        try {
            const auto header = SpeechKit::UniProxy::Header::fromJson(jsonHeader);
            if (!header.is("System", "Push") && !header.is("System", "MatchVoicePrint") && !header.is("System", "UpdateVoicePrints")) {
                YIO_LOG_DEBUG("Skip unsupported directive: " << header.toString());
                return;
            }

            const auto payload = parseJson(jsonPayload);
            auto vinsResponse = VinsResponse::parseAsyncDirectives(header, payload);
            if (!vinsResponse.directives.empty()) {
                if (connectionsStateHolder_.allConnected()) {
                    getDirectiveProcessor()->addDirectives(vinsResponse.directives);
                } else {
                    pendingVinsResponse_ = std::move(vinsResponse);
                }
            }

            if (header.is("System", "Push") && payload.isMember("push_ids")) {
                Json::Value eventPayload;
                eventPayload["push_ids"] = payload["push_ids"];
                device_->telemetry()->reportEvent("systemPushReceived", jsonToString(eventPayload));
            }
        } catch (const std::exception& e) {
            YIO_LOG_ERROR_EVENT("SpeechkitEndpoint.FailedUniProxyDirective", "Failed to handle uniProxy directive: " << e.what());
        }
    });
}

void SpeechkitEndpoint::onDeviceContextVinsResponse(std::string jsonPayload)
{
    YIO_LOG_INFO("onDeviceContextVinsResponse");

    try {
        if (aliceConfig_.getPreprocessVinsResponse()) {
            jsonPayload = deviceContext_.preprocessVinsResponse(jsonPayload);
        }

        Json::Value payload = parseJson(jsonPayload);
        auto vinsResponse = VinsResponse::parse(nullptr, SpeechKit::UniProxy::Header{}, payload, nullptr, aliceConfig_.getBlockShowCardDirectiveNames());

        getDirectiveProcessor()->addDirectives(vinsResponse.directives);
    } catch (const std::exception& e) {
        YIO_LOG_ERROR_EVENT("SpeechkitEndpoint.FailedDeviceContextVinsResponse", "exception on onVinsResponse " << e.what());
    }
}

void SpeechkitEndpoint::setReceivedConfig(const std::string& receivedConfig)
{
    YIO_LOG_INFO("setReceivedConfig");

    const auto oldSingleDialogMode = aliceConfig_.getSingleDialogMode();
    const bool oldIotCapabilityEnabled = aliceConfig_.getIotCapabilityEnabled();

    const bool isFirstConfig = !aliceConfig_.hasReceivedConfig();
    const bool configValid = aliceConfig_.setReceivedConfig(receivedConfig);

    const auto newSingleDialogMode = aliceConfig_.getSingleDialogMode();
    const bool newIotCapabilityEnabled = aliceConfig_.getIotCapabilityEnabled();

    const bool restartDialog = isFirstConfig || (oldSingleDialogMode != newSingleDialogMode);

    if (configValid) {
        deviceState_->getEnvironmentState().enable(aliceConfig_.environmentStateEnabled());
        deviceState_->onConfigUpdate(aliceConfig_);

        if (aliceConfig_.hasSystemConfig()) {
            getDirectiveProcessor()->setPrefetchEnabled(aliceConfig_.getPrefetchEnabled());

            updateLocalVinsPreprocessor();

            voiceDialog_->setPlayer(getDialogPlayer());

            naviOldSpotterCapability_->onCapabilityStateChanged(deviceStateCapability_, deviceStateCapability_->getState());
            commandSpotterCapability_->onConfigUpdated();

            voiceStats_->setRMSCorrection(aliceConfig_.getRMSCorrection());
            voiceStats_->calcOnVqe(aliceConfig_.getRMSOverVqe());
            processAudiosenderConfig();

            randomSoundLogger_->applyConfig(aliceConfig_.getRandomSoundLogger());
            featuresConfig_.processNewConfig(aliceConfig_.getSystemConfig());
            deviceStateCapability_->setSupportedFeatures(featuresConfig_.getSupportedFeatures());
            deviceStateCapability_->setUnsupportedFeatures(featuresConfig_.getUnsupportedFeatures());
            deviceStateCapability_->setExperiments(featuresConfig_.getExperiments());
        }
        deviceState_->setSmartActivation(aliceConfig_.getSmartActivation());
    }

    if (oldIotCapabilityEnabled != newIotCapabilityEnabled) {
        deviceController_->onIotCapabilityEnabledChanged();
    }

    voiceDialog_->update(aliceConfig_);
    updateDialogSettings();

    if (restartDialog) {
        startVoiceInteraction();
    }

    updateUniProxyPinger();

    aliceCapability_->onHasStartupInfo(arePrerequisitesFulfilled());
}

void SpeechkitEndpoint::updateUniProxyPinger() {
    const auto uniProxyPingerConfig = aliceConfig_.getUniProxyPingerConfig();
    auto uniProxyPingerSettings = UniProxyPinger::parseSettings(uniProxyPingerConfig);
    if (!uniProxyPingerSettings.has_value()) {
        uniProxyPinger_.reset();
        return;
    } else if (uniProxyPinger_ != nullptr && uniProxyPinger_->getSettings() == *uniProxyPingerSettings) {
        return;
    }

    uniProxyPinger_.reset(); // Prevent coexisting 2 UniProxyPingers.
    uniProxyPinger_ = std::make_unique<UniProxyPinger>(std::move(*uniProxyPingerSettings), device_);
}

void SpeechkitEndpoint::processAudiosenderConfig() {
    bool audiosenderMode = aliceConfig_.getAudioSenderMode();
    if (!audiosenderMode) {
        // we are already in no-audiosender mode
        return;
    }

    YIO_LOG_INFO("Switching to audiosender mode.");
    AudioSender::switchLaunchCondition(device_, true);

    // suicide in order to be restarted by launcher in new config state
    std::raise(SIGKILL);
}

bool SpeechkitEndpoint::arePrerequisitesFulfilled() const {
    YIO_LOG_DEBUG("requireAuthorization_: " << aliceConfig_.getRequireAuthorization()
                                            << ", passportUid=" << !deviceState_->getPassportUid().empty() << ", oauthToken=" << !deviceState_->getOAuthToken().empty()
                                            << ", metricaUuid=" << !deviceState_->getMetricaUuid().empty() << ", configReceived=" << aliceConfig_.hasReceivedConfig());
    const bool authIsOk = !aliceConfig_.getRequireAuthorization() || (!deviceState_->getPassportUid().empty() && !deviceState_->getOAuthToken().empty());
    return authIsOk && !deviceState_->getMetricaUuid().empty() && aliceConfig_.hasReceivedConfig();
}

::SpeechKit::AudioPlayer::SharedPtr SpeechkitEndpoint::getDialogPlayer() const {
    if (aliceConfig_.getUseAudioClientTtsPlayer() && aliceConfig_.getUseCustomJinglePlayer()) {
        return jingleCustomPlayer_;
    }

    return audioPlayer_;
}

void SpeechkitEndpoint::updateDialogSettings()
{
    YIO_LOG_INFO("updateDialogSettings");

    auto modelPaths = activationSpotterCapability_->getModelPaths();
    auto settings = aliceConfig_.getVoiceServiceSettings(
        deviceState_->getOAuthToken(),
        deviceState_->getPassportUid(),
        modelPaths[SpotterTypes::ACTIVATION],
        modelPaths[SpotterTypes::INTERRUPTION],
        modelPaths[SpotterTypes::ADDITIONAL],
        deviceState_->getWifiList(),
        deviceState_->formatExperiments());

    voiceDialog_->setSettings(std::move(settings));
}

void SpeechkitEndpoint::startVoiceInteraction()
{
    const bool hasPrerequirments = arePrerequisitesFulfilled();
    const bool isReadyToStart = hasPrerequirments || firstDialogStartTimerExpired_;
    if (!isReadyToStart) {
        return;
    }

    dialogStarted_ = true;
    selfDestroyer_->destroySelfIfNeed();

    std::string metricaUuid = deviceState_->getMetricaUuid();
    if (metricaUuid.empty()) {
        metricaUuid = calcMD5Digest(device_->deviceId());
        device_->telemetry()->reportError("alicedNoMetricaUid");
    }

    const std::string timezone = deviceState_->getTimezone();
    voiceDialog_->prepare(metricaUuid, deviceState_->getPassportUid(), timezone,
                          aliceConfig_.getQuasmodromGroup(), aliceConfig_.getQuasmodromSubgroup());

    if (!singleDialogStarted_ && !aliceConfig_.getSingleDialogMode().empty()) {
        singleDialogStarted_ = true;

        auto event = VinsRequest::buildNewDialogSessionEvent(aliceConfig_.getSingleDialogMode());
        auto request = std::make_shared<VinsRequest>(std::move(event), VinsRequest::createSoftwareAutoDirectiveEventSource());
        aliceCapability_->startRequest(std::move(request), nullptr);
    } else {
        startSpotting();
    }

    if (hasPrerequirments && !directiveProcessorStarted_) {
        directiveProcessorStarted_ = true;
        getDirectiveProcessor()->start(aliceConfig_.getDirectiveSequencerStatePath());
    }
}

void SpeechkitEndpoint::setVqeInfo(YandexIO::ChannelData::VqeInfo vqeInfo) {
    asyncQueue_->add([this, newVqeInfo{std::move(vqeInfo)}] {
        if (aliceConfig_.getVqeInfo() != newVqeInfo) {
            YIO_LOG_INFO("Update dialog settings after getting new vqe info. "
                         "Type: "
                         << newVqeInfo.type << " Preset: " << newVqeInfo.preset);

            aliceConfig_.setVqeInfo(newVqeInfo);
            updateDialogSettings();
        }
    });
}

void SpeechkitEndpoint::handleAudioClientMessage(const ipc::SharedMessage& message)
{
    jingleCustomPlayer_->onAudioClientEvent(message->audio_client_event());
    audioPlayerCapability_->onQuasarMessage(message);
    if (multiroomCapability_) {
        multiroomCapability_->onAudioClientEvent(message->audio_client_event());
    }
    aliceCapability_->handleAudioClientMessage(message);
}

void SpeechkitEndpoint::handleAudioClientConnect() {
    audioFocusDispatcher_.onAudioClientdConnected();
    aliceCapability_->handleAudioClientConnectionStatus(true);
}

void SpeechkitEndpoint::handleAudioClientDisconnect() {
    aliceCapability_->handleAudioClientConnectionStatus(false);
}

void SpeechkitEndpoint::onGlagoldMessage(const ipc::SharedMessage& message)
{
    if (tandemCapability_ != nullptr) {
        tandemCapability_->handleQuasarMessage(message);
    }
}

void SpeechkitEndpoint::handleInterfacedMessage(const ipc::SharedMessage& message)
{
    try {
        if (message->has_on_media_error()) {
            deviceContext_.fireMediaError(message->on_media_error());
            return;
        }
        if (message->has_vins_server_action()) {
            YIO_LOG_INFO("handleInterfacedMessage.vins_server_action");
            startAliceRequest(message->vins_server_action(), nullptr);
            return;
        }
        if (message->has_app_state() && screenCapability_) {
            screenCapability_->onAppState(message->app_state());
        }

        playbackCapability_->onQuasarMessage(message);
        legacyPlayerCapability_->onInterfacedMessage(message);
    } catch (const std::exception& e) {
        YIO_LOG_ERROR_EVENT("SpeechkitEndpoint.FailedHandleInterfacedMessage", e.what());
    }

    if (onQuasarMessageReceivedCallback) {
        onQuasarMessageReceivedCallback(*message); // Testing purposes only!
    }
}

void SpeechkitEndpoint::handleMediadMessage(const ipc::SharedMessage& message)
{
    try {
        audioPlayerCapability_->onQuasarMessage(message);
        playbackCapability_->onQuasarMessage(message);
        legacyPlayerCapability_->handleMediadMessage(message);
    } catch (const std::exception& e) {
        YIO_LOG_ERROR_EVENT("SpeechkitEndpoint.FailedHandleMediaDirective", e.what());
    }

    if (onQuasarMessageReceivedCallback) {
        onQuasarMessageReceivedCallback(*message); // Testing purposes only!
    }
}

void SpeechkitEndpoint::handleQuasarMessage(const ipc::SharedMessage& sharedMessage, std::shared_ptr<ipc::IServer::IClientConnection> connection)
{
    const auto& message = *sharedMessage;

    if (message.has_account_devices_list()) {
        deviceState_->setSpeakerCount(message.account_devices_list().account_devices_size());
    }

    if (message.has_tandem_state()) {
        auto resultFeatures = FeaturesConfig(deviceState_->getState());
        auto tandemFeatures = FeaturesConfig(message.tandem_state());
        resultFeatures.merge(tandemFeatures);
        auto tandemStateCopy = message.tandem_state();
        tandemStateCopy.MutableSupportedFeatures()->Assign(resultFeatures.getSupportedFeatures().begin(), resultFeatures.getSupportedFeatures().end());
        tandemStateCopy.MutableUnsupportedFeatures()->Assign(resultFeatures.getUnsupportedFeatures().begin(), resultFeatures.getUnsupportedFeatures().end());
        tandemStateCopy.MutableExperiments()->CopyFrom(resultFeatures.getExperiments());
        deviceState_->setTandemState(std::move(tandemStateCopy));
        naviOldSpotterCapability_->onCapabilityStateChanged(deviceStateCapability_, deviceStateCapability_->getState());
        commandSpotterCapability_->onStateUpdated();
    }

    if (message.has_vins_server_action()) {
        YIO_LOG_INFO("handleQuasarMessage.vins_server_action");
        startAliceRequest(message.vins_server_action(), nullptr);
    }
    if (message.has_network_status()) {
        deviceState_->setNetworkStatus(message.network_status());
    }
    if (message.has_wifi_list()) {
        onWifiList(message.wifi_list());
    }

    if (message.has_user_config_update()) {
        deviceState_->setAuthFailed(false);
        setReceivedConfig(message.user_config_update().config());

        legacyPlayerCapability_->onUserConfig(message.user_config_update());
    }
    if (message.has_auth_failed()) {
        YIO_LOG_INFO("Auth failed");
        deviceState_->setAuthFailed(true);
    }

    if (message.has_brick_status()) {
        deviceState_->setBrickStatus(message.brick_status());
        if (message.brick_status() == proto::BrickStatus::BRICK ||
            message.brick_status() == proto::BrickStatus::BRICK_BY_TTL) {
            aliceCapability_->cancelDialogAndClearQueue();
        }
    }

    if (message.has_subscription_state()) {
        deviceState_->getEnvironmentState().updateLocalSubscriptionType(message.subscription_state());
    }

    if (message.has_broken_mic_info()) {
        brokenMicInfoLogger_.log(message.broken_mic_info());
    }

    if (message.has_calld_state_changed()) {
        deviceState_->setCalldState(message.calld_state_changed());
    }

    if (message.has_iot_state()) {
        handleIotState(message.iot_state());
    }

    if (message.has_service_name_ready()) {
        handleServiceReady(message.service_name_ready());
    }

    if (message.has_do_not_disturb_event() && message.do_not_disturb_event().has_is_dnd_enabled()) {
        deviceState_->setDndEnabled(message.do_not_disturb_event().is_dnd_enabled());
    }

    if (message.has_external_alice_state()) {
        aliceCapability_->onExternalAliceState(message.external_alice_state());
    }

    if (message.has_directive()) {
        getDirectiveProcessor()->addDirectives({YandexIO::Directive::createDirectiveFromProtobuf(message.directive())});
    }

    if (message.has_control_request()) {
        reportNavigationRequest(message.control_request());
        if (hasInterfaced()) {
            auto interfaceMessage = message;
            toInterfaced_->sendMessage(std::move(interfaceMessage));
        }
    }

    if (message.has_environment_message()) {
        const auto& env = message.environment_message();
        if (env.has_device_info()) {
            YIO_LOG_DEBUG("environment message with device info received");
            deviceState_->getEnvironmentState().updateRemoteDevice(env.device_info());
        }
    }

    if (message.has_remoting()) {
        YIO_LOG_DEBUG("Protobuf=" << shortUtf8DebugString(message));
        deviceController_->getRemotingMessageRouter()->routeRemotingMessage(message.remoting(), connection);
    }

    if (message.has_watched_video()) {
        deviceStateCapability_->setLastWatched(YandexIO::convertLastWatched(message.watched_video()));
    }

    audioPlayerCapability_->onQuasarMessage(sharedMessage);
    playbackCapability_->onQuasarMessage(sharedMessage);

    if (tandemCapability_ != nullptr) {
        tandemCapability_->handleQuasarMessage(sharedMessage);
    }

    if (onQuasarMessageReceivedCallback) {
        onQuasarMessageReceivedCallback(message); // Testing purposes only!
    }
}

void SpeechkitEndpoint::handleServiceReady(const std::string& serviceName) {
    YIO_LOG_INFO("service ready: " << serviceName);
    connectionsStateHolder_.connect(serviceName);
    checkServicesForNotifications();
}

void SpeechkitEndpoint::checkServicesForNotifications() {
    if (!connectionsStateHolder_.allConnected()) {
        return;
    }

    if (pendingVinsResponse_) {
        getDirectiveProcessor()->addDirectives((*pendingVinsResponse_).directives);
        pendingVinsResponse_.reset();
    }
}

void SpeechkitEndpoint::startAliceRequest(
    const proto::VinsServerAction& vinsServerAction,
    std::shared_ptr<YandexIO::IAliceRequestEvents> events)
{
    YIO_LOG_INFO("startAliceRequest");

    try {
        std::optional<std::string> requestId;
        if (vinsServerAction.has_request_id()) {
            requestId = vinsServerAction.request_id();
        }
        auto event = parseJson(std::string(vinsServerAction.server_action()));

        auto request = VinsRequest::createEventRequest(event, VinsRequest::createSoftwareDirectiveEventSource(), requestId);
        request->setResetSession(vinsServerAction.reset_session());
        request->setIsParallel(vinsServerAction.is_parallel());
        request->setEnqueued(vinsServerAction.enqueued());
        request->setIsSilent(vinsServerAction.is_silent());

        aliceCapability_->startRequest(std::move(request), std::move(events));
    } catch (const std::exception& e) {
        YIO_LOG_ERROR_EVENT("SpeechkitEndpoint.FailedToStartAliceRequest", e.what());
    }
}

void SpeechkitEndpoint::handleIotState(const proto::IotState& iotState) {
    // Message has both current and prev state if and only if when state is transitioned
    if (iotState.has_current_state() && iotState.has_prev_state()) {
        if (iotState.current_state() == proto::IotState::STARTING_DISCOVERY) {
            if (iotState.current_discovery_info().has_device_type() && iotState.current_discovery_info().has_timeout_ms() && iotState.current_discovery_info().has_ssid() && iotState.current_discovery_info().has_password()) {
                IotSemanticFrameParams params;
                params.purpose = "send_iot_pairing_token";
                params.broadcastFieldName = "iot_discovery_start";
                params.deviceType = iotState.current_discovery_info().device_type();
                params.ssid = iotState.current_discovery_info().ssid();
                params.password = iotState.current_discovery_info().password();
                params.timeout = std::chrono::milliseconds(iotState.current_discovery_info().timeout_ms());

                auto semanticFrame = buildIotSemanticFrame(params);
                YIO_LOG_INFO("Sending semantic frame: " << quasar::transform(semanticFrame, [pwd = params.password](const std::string& val) -> std::string {
                                 if (val == pwd) {
                                     return "*";
                                 }
                                 return val;
                             }));
                auto request = VinsRequest::createEventRequest(semanticFrame, VinsRequest::createSoftwareDirectiveEventSource());
                request->setEnqueued(true);
                aliceCapability_->startRequest(std::move(request), nullptr);
            } else {
                device_->telemetry()->reportError("iotDiscoveryStartedWithoutInfo");
            }

        } else if (iotState.current_state() == proto::IotState::IDLE) {
            // We notify only when discovery is unsuccessful. Successful discovery notification is made by push
            if (iotState.has_result() && iotState.result().has_code() && iotState.current_discovery_info().has_timeout_ms()) {
                if (iotState.result().code() != proto::IotDiscoveryResult::SUCCESS) {
                    IotSemanticFrameParams params;
                    params.purpose = "notify_about_broadcast_failure";
                    params.broadcastFieldName = "iot_discovery_failure";
                    params.deviceType = iotState.current_discovery_info().device_type();
                    params.reason = convertIotResultCode(iotState.result().code());
                    params.timeout = std::chrono::milliseconds(iotState.current_discovery_info().timeout_ms());

                    auto semanticFrame = buildIotSemanticFrame(params);
                    YIO_LOG_INFO("Sending semantic frame: " << semanticFrame);
                    auto request = VinsRequest::createEventRequest(semanticFrame, VinsRequest::createSoftwareDirectiveEventSource());
                    request->setEnqueued(true);
                    aliceCapability_->startRequest(std::move(request), nullptr);
                }
            } else {
                device_->telemetry()->reportError("iotDiscoveryStoppedWithoutInfo");
            }
        }
    } else {
        YIO_LOG_WARN("Iot state message does not contain state transition");
    }
}

std::string SpeechkitEndpoint::convertIotResultCode(proto::IotDiscoveryResult::ResultCode resultCode) {
    switch (resultCode) {
        case proto::IotDiscoveryResult::SUCCESS:
            return "SUCCESS";
        case proto::IotDiscoveryResult::FAILURE:
            return "FAILURE";
        case proto::IotDiscoveryResult::TIMEOUT:
            return "TIMEOUT";
        case proto::IotDiscoveryResult::SUPERCEDED:
            return "SUPERCEDED";
        case proto::IotDiscoveryResult::NOT_STARTED:
            return "NOT_STARTED";
        case proto::IotDiscoveryResult::UNKNOWN:
        default:
            return "UNKNOWN";
    }
}

Json::Value SpeechkitEndpoint::buildIotSemanticFrame(const IotSemanticFrameParams& params) {
    Json::Value semanticFrame;
    Json::Value semanticFramePayload;
    semanticFramePayload["utterance"] = ""; // Megamind doesn't work properly without utterance field
    auto& analytics = semanticFramePayload["analytics"];
    analytics["product_scenario"] = "iot.voice_discovery";
    analytics["origin"] = "SmartSpeaker";
    analytics["purpose"] = params.purpose;

    auto& iotBroadcastStart = semanticFramePayload["typed_semantic_frame"][params.broadcastFieldName];
    if (params.ssid) {
        iotBroadcastStart["ssid"]["string_value"] = *params.ssid;
    }
    if (params.password) {
        iotBroadcastStart["password"]["string_value"] = *params.password;
    }
    if (params.deviceType) {
        iotBroadcastStart["device_type"]["string_value"] = *params.deviceType;
    }
    if (params.timeout) {
        iotBroadcastStart["timeout_ms"]["uint32_value"] = (int)params.timeout->count();
    }
    if (params.reason) {
        iotBroadcastStart["reason"]["string_value"] = *params.reason;
    }
    semanticFrame["name"] = "@@mm_semantic_frame";
    semanticFrame["type"] = "server_action";
    semanticFrame["payload"] = semanticFramePayload;
    return semanticFrame;
}

void SpeechkitEndpoint::updateLocalVinsPreprocessor() {
    if (aliceConfig_.getWithLocalVins()) {
        if (localVinsPreprocessor_ == nullptr) {
            localVinsPreprocessor_ = std::make_shared<LocalVinsPreprocessor>(device_);
            getDirectiveProcessor()->addDirectivePreprocessor(localVinsPreprocessor_);
        }
    } else {
        getDirectiveProcessor()->removeDirectivePreprocessor(localVinsPreprocessor_);
        localVinsPreprocessor_.reset();
    }
}

bool SpeechkitEndpoint::hasInterfaced() const {
    return device_->configuration()->hasServiceConfig("interfaced");
}

void SpeechkitEndpoint::reportAllStartupSettings()
{
    proto::QuasarMessage message;
    message.mutable_all_startup_settings()->set_all_startup_info(arePrerequisitesFulfilled());
    message.mutable_all_startup_settings()->set_auth_token(TString(deviceState_->getOAuthToken()));
    message.mutable_all_startup_settings()->set_passport_uid(TString(deviceState_->getPassportUid()));
    server_->sendToAll(std::move(message));
}

void SpeechkitEndpoint::onSpottingStarted() {
    std::string modelVersion;
    for (const auto& [type, modelPath] : activationSpotterCapability_->getModelPaths()) {
        if (type == SpotterTypes::ACTIVATION) {
            modelVersion = SpeechKit::PhraseSpotterSettings(modelPath).getModel();
            break;
        }
    }

    aliceCapability_->onStartSpotting(modelVersion, arePrerequisitesFulfilled());
}

void SpeechkitEndpoint::onTimezone(const proto::Timezone& timezone) {
    if (deviceState_->getTimezone() != timezone.timezone_name()) {
        YIO_LOG_INFO("onTimezone: " << deviceState_->getTimezone() << " -> " << timezone.timezone_name());
        deviceState_->setTimezone(timezone.timezone_name());

        startVoiceInteraction();
    }
}

void SpeechkitEndpoint::onLocation(const proto::Location& location) {
    deviceState_->setLocation(location);
}

void SpeechkitEndpoint::onWifiList(const proto::WifiList& wifiList) {
    deviceState_->setWifiList(wifiList);
    voiceDialog_->setSynchronizeStatePayload(aliceConfig_.buildSynchronizeStatePayload(deviceState_->getWifiList(), deviceState_->formatExperiments()));
}

void SpeechkitEndpoint::onDeviceStatePart(const yandex_io::proto::TDeviceStatePart& statePart)
{
    playbackCapability_->onDeviceStatePart(statePart);
    deviceStateCapability_->onDeviceStatePart(statePart);
}

void SpeechkitEndpoint::onMediaDeviceIdentifier(const NAlice::TClientInfoProto::TMediaDeviceIdentifier& identifier) {
    deviceState_->getEnvironmentState().updateLocalMediaDeviceIdentifier(identifier);
}

void SpeechkitEndpoint::broadcastEnvironmentDeviceInfo()
{
    const auto deviceInfo = deviceState_->getEnvironmentState().getLocalDeviceInfo();

    auto message = ipc::buildMessage([&](auto& msg) {
        msg.mutable_environment_message()->mutable_device_info()->CopyFrom(deviceInfo);
    });
    server_->sendToAll(message);
}

void SpeechkitEndpoint::broadcastAppState(const quasar::proto::AppState& appState)
{
    // Consumers:
    // - glagold (interdevice state compatibility)
    // - interfaced (media state for screens)
    // - braind
    // - audiosenderd (recording tool)
    // - multiroomd
    auto message = ipc::buildMessage([&](auto& msg) {
        msg.mutable_app_state()->CopyFrom(appState);
    });
    server_->sendToAll(message);
    if (toInterfaced_) {
        toInterfaced_->sendMessage(message);
    }
}

void SpeechkitEndpoint::initCapabilities()
{
    YIO_LOG_INFO("initCapabilities");

    deviceStateCapability_ = std::make_shared<YandexIO::DeviceStateCapabilityHost>(asyncQueue_, deviceController_->getRemotingMessageRouter());
    deviceStateCapability_->addListener(deviceState_);
    deviceStateCapability_->addListener(randomSoundLogger_);
    deviceStateCapability_->init();
    deviceStateCapability_->setSupportedFeatures(featuresConfig_.getSupportedFeatures());
    deviceStateCapability_->setUnsupportedFeatures(featuresConfig_.getUnsupportedFeatures());
    deviceStateCapability_->setExperiments(featuresConfig_.getExperiments());

    deviceController_->getEndpointStorage()->getLocalEndpoint()->addCapability(deviceStateCapability_);

    deviceStateCapability_->setAppStateChanged([this](const proto::AppState& appState) {
        deviceState_->setAppState(appState);
        commandSpotterCapability_->onStateUpdated();
        broadcastAppState(appState);
    });

    playbackCapability_ = std::make_shared<PlaybackControlCapability>(
        asyncQueue_, device_, getDirectiveProcessor(),
        *deviceState_, toGlagold_, toInterfaced_, deviceController_->getRemotingMessageRouter());
    playbackCapability_->init();
    deviceStateCapability_->addListener(playbackCapability_);
    Y_VERIFY(getDirectiveProcessor()->addDirectiveHandler(playbackCapability_));

    tandemCapability_ = std::make_shared<YandexIO::TandemCapability>(
        device_, ipcFactory_, authProvider_, asyncQueue_,
        getDirectiveProcessor(), deviceState_, deviceStateCapability_,
        playbackCapability_, deviceController_->getEndpointStorage());
    Y_VERIFY(getDirectiveProcessor()->addDirectiveHandler(tandemCapability_));
    tandemCapability_->init();

    filePlayerCapability_ = std::make_shared<FilePlayerCapability>(
        asyncQueue_, device_, activityTracker_, getDirectiveProcessor(),
        ipcFactory_, deviceController_->getRemotingMessageRouter());
    filePlayerCapability_->init();
    Y_VERIFY(getDirectiveProcessor()->addDirectiveHandler(filePlayerCapability_));

    aliceCapability_ = std::make_shared<YandexIO::AliceCapability>(
        aliceConfig_, *deviceState_, activityTracker_, getDirectiveProcessor(),
        deviceContext_, device_->telemetry(), voiceDialog_, server_, toInterfaced_, voiceStats_,
        deviceController_->getRemotingMessageRouter(), stereoPairProvider_, filePlayerCapability_);
    aliceCapability_->init();
    deviceController_->setAliceCapability(aliceCapability_);
    Y_VERIFY(getDirectiveProcessor()->addDirectiveHandler(aliceCapability_));

    if (tandemCapability_) {
        aliceCapability_->addListener(tandemCapability_);
    }

    audioPlayerCapability_ = std::make_shared<AudioPlayerCapability>(
        activityTracker_, getDirectiveProcessor(),
        toAudioClientd_, deviceContext_, device_, filePlayerCapability_, deviceStateCapability_);
    Y_VERIFY(getDirectiveProcessor()->addDirectiveHandler(audioPlayerCapability_));
    aliceCapability_->addListener(audioPlayerCapability_);

    alarmCapability_ = std::make_shared<AlarmCapability>(
        asyncQueue_, device_, ipcFactory_, deviceController_->getRemotingMessageRouter(),
        getDirectiveProcessor(), aliceCapability_, filePlayerCapability_, playbackCapability_, deviceStateCapability_);
    alarmCapability_->init();
    Y_VERIFY(getDirectiveProcessor()->addDirectiveHandler(alarmCapability_));
    getDirectiveProcessor()->addListener(alarmCapability_);

    legacyPlayerCapability_ = std::make_shared<YandexIO::LegacyPlayerCapability>(
        getDirectiveProcessor(), *deviceState_, device_, deviceContext_, toMediad_, toInterfaced_, deviceStateCapability_);
    Y_VERIFY(getDirectiveProcessor()->addDirectiveHandler(legacyPlayerCapability_));

    auto externalCommandCapability = std::make_shared<ExternalCommandCapability>(
        device_, toBugReport_, toInterfaced_, toCalld_, toNotificationd_);
    Y_VERIFY(getDirectiveProcessor()->addDirectiveHandler(externalCommandCapability));

    deviceController_->onIotCapabilityEnabledChanged();

    if (device_->configuration()->hasServiceConfig("interfaced")) {
        screenCapability_ = std::make_shared<YandexIO::ScreenCapability>(
            deviceStateCapability_, asyncQueue_, getDirectiveProcessor(), *deviceState_, toInterfaced_);
        Y_VERIFY(getDirectiveProcessor()->addDirectiveHandler(screenCapability_));
    }

    bluetoothCapability_ = std::make_shared<YandexIO::BluetoothCapability>(asyncQueue_, deviceStateCapability_, ipcFactory_, activityTracker_, getDirectiveProcessor(), toInterfaced_, toMediad_);
    Y_VERIFY(getDirectiveProcessor()->addDirectiveHandler(bluetoothCapability_));

    if (multiroomProvider_) {
        Y_VERIFY(glagolClusterProvider_);
        mrForwarderCapability_ = std::make_shared<MRForwarderCapability>(device_, glagolClusterProvider_, multiroomProvider_);
        Y_VERIFY(getDirectiveProcessor()->addDirectiveHandler(mrForwarderCapability_));
        multiroomCapability_ = std::make_shared<MultiroomCapability>(
            asyncQueue_,
            device_,
            ipcFactory_,
            sdk_,
            activityTracker_,
            getDirectiveProcessor(),
            toAudioClientd_,
            clockTowerProvider_,
            glagolClusterProvider_,
            multiroomProvider_,
            stereoPairProvider_,
            userConfigProvider_);
        Y_VERIFY(getDirectiveProcessor()->addDirectiveHandler(multiroomCapability_));
    }

    if (stereoPairProvider_) {
        auto stereoPairCapability = std::make_shared<StereoPairCapability>(stereoPairProvider_);
        Y_VERIFY(getDirectiveProcessor()->addDirectiveHandler(stereoPairCapability));
    }

    naviOldSpotterCapability_ = std::make_shared<NaviOldSpotterCapability>(
        device_, audioSource_, getDirectiveProcessor(), aliceConfig_, *deviceState_, asyncQueue_,
        deviceController_->getRemotingMessageRouter());
    naviOldSpotterCapability_->init();
    aliceCapability_->addListener(naviOldSpotterCapability_);
    deviceStateCapability_->addListener(naviOldSpotterCapability_);
    naviOldSpotterCapability_->onCapabilityStateChanged(deviceStateCapability_, deviceStateCapability_->getState());

    commandSpotterCapability_ = std::make_shared<CommandSpotterCapability>(
        asyncQueue_, aliceConfig_, *deviceState_, getDirectiveProcessor(),
        device_->telemetry(), aliceCapability_, playbackCapability_, voiceDialog_,
        deviceController_->getRemotingMessageRouter());
    commandSpotterCapability_->init();
    aliceCapability_->addListener(commandSpotterCapability_);
    alarmCapability_->addListener(commandSpotterCapability_);
    // note: it's important to add commandSpotterCapability_ to deviceStateCapability after
    // adding aliceDeviceState to deviceStateCapability. Otherwise alcieDeviceState will not have
    // latest state
    deviceStateCapability_->addListener(commandSpotterCapability_);
    commandSpotterCapability_->onStateUpdated();

    activationSpotterCapability_ = std::make_shared<ActivationSpotterCapability>(
        asyncQueue_, aliceConfig_, voiceDialog_, *deviceState_, deviceController_->getRemotingMessageRouter());
    activationSpotterCapability_->init();

    initPreprocessors();
}

void SpeechkitEndpoint::initPreprocessors()
{
    YIO_LOG_INFO("initPreprocessors");

    if (multiroomProvider_ != nullptr) {
        getDirectiveProcessor()->addDirectivePreprocessor(std::make_shared<MultiroomPreprocessor>(device_->deviceId(), multiroomProvider_, deviceState_));
    }

    if (stereoPairProvider_ != nullptr) {
        getDirectiveProcessor()->addDirectivePreprocessor(std::make_shared<StereoPairPreprocessor>(stereoPairProvider_, userConfigProvider_));
    }

    getDirectiveProcessor()->addDirectivePreprocessor(alarmCapability_);
    getDirectiveProcessor()->addDirectivePreprocessor(std::make_shared<AudioPlayerPreprocessor>());
    if (screenCapability_ != nullptr) {
        getDirectiveProcessor()->addDirectivePreprocessor(screenCapability_);
    }

    if (tandemCapability_ != nullptr) {
        getDirectiveProcessor()->addDirectivePreprocessor(tandemCapability_);
    }

    getDirectiveProcessor()->addDirectivePreprocessor(std::make_shared<AliceCapabilityPreprocessor>(bluetoothCapability_ != nullptr, *deviceState_));
    if (bluetoothCapability_ != nullptr) {
        getDirectiveProcessor()->addDirectivePreprocessor(std::make_shared<YandexIO::BluetoothCapabilityPreprocessor>(*deviceState_));
    }
    if (tryGetBool(device_->configuration()->getServiceConfig("testpoint"), "use_testpoint_preprocessor", false)) {
        auto testpointPeer = std::make_shared<TestpointPeer>(*ipcFactory_);
        getDirectiveProcessor()->addDirectivePreprocessor(std::make_shared<YandexIO::TestpointDirectivePreprocessor>(testpointPeer));

        playbackCapability_->setTestpointPeer(testpointPeer);
    }
    updateLocalVinsPreprocessor();
}

void SpeechkitEndpoint::reportNavigationRequest(const proto::ControlRequest& controlRequest) {
    const std::string& vinsRequestId = controlRequest.vins_request_id();
    if (vinsRequestId.empty() || !controlRequest.has_navigation_request()) {
        return;
    }

    Json::Value etePayload;
    appendETEHeader(vinsRequestId, etePayload);
    if (controlRequest.navigation_request().has_go_down()) {
        etePayload["command_name"] = Directives::GO_DOWN;
    } else if (controlRequest.navigation_request().has_go_up()) {
        etePayload["command_name"] = Directives::GO_UP;
    } else if (controlRequest.navigation_request().has_go_left()) {
        etePayload["command_name"] = Directives::GO_BACKWARD;
    } else if (controlRequest.navigation_request().has_go_right()) {
        etePayload["command_name"] = Directives::GO_FORWARD;
    } else {
        return;
    }

    device_->telemetry()->reportEvent("external_command_received", jsonToString(etePayload));
}

const std::shared_ptr<YandexIO::DirectiveProcessor>& SpeechkitEndpoint::getDirectiveProcessor() const {
    return deviceController_->getDirectiveProcessor();
}
