#include "alice_audio_source.h"

#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/logging/logging.h>

using namespace quasar;
using namespace SpeechKit;

namespace {
    const SpeechKit::SoundInfo SOUND_INFO = SpeechKit::SoundInfo(SoundFormat::PCM, 1, 16000, 2);

    SpeechKit::SoundInfo::ChannelType toSpeechkitChannelType(const YandexIO::ChannelData::Type type) {
        switch (type) {
            case YandexIO::ChannelData::Type::RAW:
                return SpeechKit::SoundInfo::ChannelType::RAW_MIC;
            case YandexIO::ChannelData::Type::VQE:
                return SpeechKit::SoundInfo::ChannelType::VQE_MAIN;
            case YandexIO::ChannelData::Type::FEEDBACK:
                return SpeechKit::SoundInfo::ChannelType::FEEDBACK;
            case YandexIO::ChannelData::Type::BEAMFORMING:
                return SpeechKit::SoundInfo::ChannelType::VQE_BEAMFORMING;
            case YandexIO::ChannelData::Type::BACKGROUND_NOISE_REDUCER:
                return SpeechKit::SoundInfo::ChannelType::VQE_BNR;
            case YandexIO::ChannelData::Type::MAIN_MIC_SYNC:
                return SpeechKit::SoundInfo::ChannelType::MAIN_MIC_SYNC;
            case YandexIO::ChannelData::Type::AUXILIARY_MIC_SYNC:
                return SpeechKit::SoundInfo::ChannelType::AUXILIARY_MIC_SYNC;
            case YandexIO::ChannelData::Type::FEEDBACK_SYNC:
                return SpeechKit::SoundInfo::ChannelType::FEEDBACK_SYNC;
            default:
                return SpeechKit::SoundInfo::ChannelType::UNKNOWN;
        }
    }

    SpeechKit::SoundInfo makeSoundInfo(const YandexIO::ChannelData& channel) {
        SpeechKit::SoundInfo info{SoundFormat::PCM, 1, channel.sampleRate, channel.sampleSize};
        info.setChannelType(toSpeechkitChannelType(channel.type));
        return info;
    }

    CompositeSoundBuffer::SharedPtr convert(
        const YandexIO::ChannelsData& allChannels)
    {
        auto makeSoundBuffer = [](const auto& channel) -> std::shared_ptr<SoundBuffer> {
            if (channel.data.empty()) {
                return nullptr;
            }
            std::vector<uint8_t> audioData;
            audioData.resize(channel.data.size() * channel.sampleSize);
            memcpy(audioData.data(), channel.data.data(), audioData.size());
            return std::make_shared<SoundBuffer>(makeSoundInfo(channel), std::move(audioData));
        };

        /* main buffer initialization */
        const auto mainChannel = std::find_if(allChannels.cbegin(), allChannels.cend(), [](const auto& channel) {
            return channel.isForRecognition;
        });
        if (mainChannel == allChannels.cend()) {
            return nullptr; // should not happen
        }
        auto mainBuffer = makeSoundBuffer(*mainChannel);
        if (mainBuffer == nullptr) {
            /* main channel should not be empty */
            return nullptr;
        }

        CompositeSoundBuffer::ChannelNameToBuffer buffers{{mainChannel->name, mainBuffer}};
        for (auto channel = allChannels.cbegin(); channel != allChannels.cend(); ++channel) {
            if (channel == mainChannel) {
                // skip mainChannel, since buffers already has it
                continue;
            }

            if (auto soundBuffer = makeSoundBuffer(*channel)) {
                buffers.emplace(channel->name, std::move(soundBuffer));
            } else {
                YIO_LOG_WARN("Empty buffer with channel name [" << channel->name << "]");
            }
        }

        return std::make_shared<CompositeSoundBuffer>(mainChannel->name, std::move(buffers));
    }

    std::optional<YandexIO::ChannelData::VqeInfo> getVqeInfo(const YandexIO::ChannelsData& allChannels) {
        const auto mainChannel = std::find_if(allChannels.cbegin(), allChannels.cend(), [](const auto& channel) {
            return channel.isForRecognition;
        });
        if (mainChannel == allChannels.cend()) {
            return std::nullopt;
        }
        if (mainChannel->type != YandexIO::ChannelData::Type::VQE) {
            /* do not get vqeInfo if mainChannel is not "vqe" type */
            return std::nullopt;
        }

        return mainChannel->meta.vqeInfo.value_or(YandexIO::ChannelData::VqeInfo());
    }
} // namespace

AliceAudioSource::AliceAudioSource(OnVqeInfoChanged onVqeInfoChanged, std::shared_ptr<VoiceStats> voiceStats)
    : onVqeInfoChanged_(std::move(onVqeInfoChanged))
    , voiceStats_(std::move(voiceStats))
{
}

std::shared_ptr<AliceAudioSource> AliceAudioSource::create(OnVqeInfoChanged onVqeInfoChanged, std::shared_ptr<VoiceStats> voiceStats)
{
    return std::shared_ptr<AliceAudioSource>(new AliceAudioSource(std::move(onVqeInfoChanged), std::move(voiceStats)));
}

int AliceAudioSource::getBufferCaptureTimeout() const {
    return 100; // Milliseconds
}

const SoundInfo& AliceAudioSource::getSoundInfo() const {
    return SOUND_INFO;
}

void AliceAudioSource::subscribe(std::weak_ptr<AudioSource::AudioSourceListener> listener)
{
    std::lock_guard<std::mutex> lock(listenersMutex_);
    if (listeners_.end() == std::find_if(listeners_.begin(), listeners_.end(),
                                         [listener](const AudioSource::AudioSourceListener::WeakPtr& wp) {
                                             return !(wp.owner_before(listener) || listener.owner_before(wp));
                                         }))
    {
        listeners_.push_back(listener);
        if (auto slistener = listener.lock()) {
            slistener->onAudioSourceStarted(shared_from_this());
        }
    }
}

void AliceAudioSource::unsubscribe(std::weak_ptr<AudioSource::AudioSourceListener> listener)
{
    std::lock_guard<std::mutex> lock(listenersMutex_);

    listeners_.remove_if([listener](const AudioSource::AudioSourceListener::WeakPtr& wp) {
        return !(wp.owner_before(listener) || listener.owner_before(wp));
    });
}

SpeechKit::CompositeSoundBuffer::SharedPtr AliceAudioSource::createSpeechkitBuffer(
    const google::protobuf::RepeatedPtrField<quasar::proto::AudioInput>& audioInputs) {
    SpeechKit::CompositeSoundBuffer::ChannelNameToBuffer buffers;
    for (const auto& audioInput : audioInputs) {
        if (!audioInput.has_channel_name()) {
            YIO_LOG_WARN("Skipping audio data without channel name");
            continue;
        }

        std::vector<uint8_t> data(audioInput.audio_data().begin(), audioInput.audio_data().end());
        buffers.emplace(audioInput.channel_name(), std::make_shared<SpeechKit::SoundBuffer>(SOUND_INFO, std::move(data)));
    }

    if (buffers.empty()) {
        return nullptr;
    }

    return std::make_shared<SpeechKit::CompositeSoundBuffer>(
        SpeechKit::CompositeSoundBuffer::DefaultMainChannelName, std::move(buffers));
}

void AliceAudioSource::onAudioData(const YandexIO::ChannelsData& data)
{
    if (const auto optVqeInfo = getVqeInfo(data); optVqeInfo.has_value() && optVqeInfo != vqeInfo_) {
        vqeInfo_ = optVqeInfo.value();
        /* Vqe Info Changed: notify SpeechkitEndpoint */
        onVqeInfoChanged_(vqeInfo_);
    }

    auto speechkitBuffer = convert(data);
    if (speechkitBuffer == nullptr) {
        return;
    }

    std::vector<AudioSourceListener::SharedPtr> lockedListeners;
    {
        std::lock_guard<std::mutex> lock(listenersMutex_);
        lockedListeners.reserve(listeners_.size());
        for (auto& listener : listeners_) {
            if (auto listenerPtr = listener.lock()) {
                lockedListeners.emplace_back(listenerPtr);
            };
        }
    }
    auto wthis = shared_from_this();
    for (auto& listener : lockedListeners) {
        listener->onAudioSourceData(wthis, speechkitBuffer);
    }

    voiceStats_->pushAudioChannels(data);
}
