#include "broken_mic_info_logger.h"

#include "alice_audio_source.h"

#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>

using namespace quasar;

BrokenMicInfoLogger::BrokenMicInfoLogger(std::shared_ptr<YandexIO::ITelemetry> telemetry)
    : telemetry_(std::move(telemetry))
{
}

void BrokenMicInfoLogger::onSuccess(const std::string& id, const std::vector<std::string>& messageIds) {
    auto it = idToEntry_.find(id);
    if (it == idToEntry_.end()) {
        return;
    }

    std::unordered_map<std::string, std::string> args;
    fillCommonArgs(args, it->second);

    args["sendSuccess"] = std::to_string(true);
    for (const auto& messageId : messageIds) {
        args["soundLoggerMessageId"].append(messageId);
    }

    telemetry_->reportKeyValues("brokenMicDetected", args);
    idToEntry_.erase(it);
}

void BrokenMicInfoLogger::onFail(const std::string& id, const std::string& error) {
    auto it = idToEntry_.find(id);
    if (it == idToEntry_.end()) {
        return;
    }

    YIO_LOG_WARN("Fail to send broken mic detector audio data");

    std::unordered_map<std::string, std::string> args;
    fillCommonArgs(args, it->second);

    args["sendSuccess"] = std::to_string(false);
    args["error"] = error;

    telemetry_->reportKeyValues("brokenMicDetected", args);
    idToEntry_.erase(it);
}

void BrokenMicInfoLogger::log(const proto::BrokenMicInfo& message) {
    if (auto buffer = AliceAudioSource::createSpeechkitBuffer(message.audio_inputs())) {
        SpeechKit::SoundLogger::Entry entry;
        entry.sound.push_back(std::move(buffer));

        Json::Value jsonOptions;
        jsonOptions["extra"]["brokenMicDetector"] = true;
        entry.jsonOptions = jsonToString(jsonOptions);

        std::vector<double> rmsRatios;
        rmsRatios.reserve(message.rms_ratios_size());
        for (const auto& rmsRatio : message.rms_ratios()) {
            rmsRatios.push_back(rmsRatio);
        }
        idToEntry_[entry.id] = {rmsRatios, message.first_setup()};

        SpeechKit::SoundLogger::getInstance()->sendLog(entry, shared_from_this());
    } else {
        std::unordered_map<std::string, std::string> args;
        args["detected"] = std::to_string(message.detected());
        args["firstSetup"] = std::to_string(message.first_setup());
        telemetry_->reportKeyValues("brokenMicDetected", args);
    }
}

void BrokenMicInfoLogger::fillCommonArgs(std::unordered_map<std::string, std::string>& args, const SendEntry& entry) {
    args["detected"] = std::to_string(true);

    std::stringstream ss;
    for (const auto& value : entry.rmsRatios_) {
        ss << value << ",";
    }
    args["rmsRatios"] = ss.str();
    args["firstSetup"] = std::to_string(entry.firstSetup_);
}
