#include "jingle_audio_client_player.h"

#include <yandex_io/libs/base/utils.h>
#include <yandex_io/protos/quasar_proto.pb.h>

using namespace quasar;

namespace {

    quasar::proto::AudioPlayerDescriptor createDescriptor() {
        quasar::proto::AudioPlayerDescriptor result;
        result.set_type(quasar::proto::AudioPlayerDescriptor::STREAM);
        result.set_player_id(quasar::makeUUID());
        return result;
    }

} /* namespace */

JingleAudioClientPlayer::JingleAudioClientPlayer(std::shared_ptr<ipc::IConnector> connector)
    : descriptor_(createDescriptor())
    , connector_(std::move(connector))
{
    Y_VERIFY(connector_ != nullptr);
}

JingleAudioClientPlayer::~JingleAudioClientPlayer() {
    worker_.destroy();
}

void JingleAudioClientPlayer::play() {
}

void JingleAudioClientPlayer::pause() {
}

void JingleAudioClientPlayer::cancel() {
    // todo: implement
}

void JingleAudioClientPlayer::playData(SpeechKit::SoundBuffer::SharedPtr buffer) {
    worker_.add([this, buffer{std::move(buffer)}]() {
        handlePlayingStart(buffer->getInfo().getSampleRate());

        sendStreamData(reinterpret_cast<const char*>(buffer->getData().data()), buffer->getData().size());

        auto sthis = shared_from_this();
        notifyListeners([&sthis, &buffer](auto& listener) {
            listener.onPlayingData(sthis, buffer);
        });
    });
}

void JingleAudioClientPlayer::setDataEnd() {
    worker_.add([this]() {
        playing_ = false;

        proto::QuasarMessage msg;
        auto& mediaRequest = *msg.mutable_media_request();
        *mediaRequest.mutable_player_descriptor() = descriptor_;
        mediaRequest.mutable_stream_data_end();
        connector_->sendMessage(std::move(msg));
    });
}

void JingleAudioClientPlayer::sendStreamData(const char* data, size_t size) {
    proto::QuasarMessage msg;
    auto& mediaRequest = *msg.mutable_media_request();
    *mediaRequest.mutable_player_descriptor() = descriptor_;
    mediaRequest.set_stream_data(data, size);
    connector_->sendMessage(std::move(msg));
}

void JingleAudioClientPlayer::setVolume(float /* gain */) {
}

float JingleAudioClientPlayer::getVolume() const {
    return 1.0;
}

void JingleAudioClientPlayer::subscribe(AudioPlayerListener::WeakPtr listener) {
    worker_.add([this, listener{std::move(listener)}]() {
        removeExpired();

        const auto it = std::find_if(listeners_.begin(), listeners_.end(), [listener](const auto& wp) {
            return !(wp.owner_before(listener) || listener.owner_before(wp));
        });
        if (it == listeners_.end()) {
            listeners_.push_back(listener);
        }
    });
}

void JingleAudioClientPlayer::unsubscribe(AudioPlayerListener::WeakPtr listener) {
    worker_.add([this, listener{std::move(listener)}]() {
        removeExpired();

        listeners_.remove_if([listener](const auto& wp) {
            return !(wp.owner_before(listener) || listener.owner_before(wp));
        });
    });
}

void JingleAudioClientPlayer::handlePlayingStart(int rate) {
    if (!std::exchange(playing_, true)) {
        rate_ = rate;

        // create player
        proto::QuasarMessage msg;
        auto& mediaRequest = *msg.mutable_media_request();
        *mediaRequest.mutable_player_descriptor() = descriptor_;
        auto& playAudio = *mediaRequest.mutable_play_audio();
        playAudio.set_input_stream_rate(rate);
        playAudio.set_format(proto::Audio::PCM_STREAM);
        /* Jingle should play with full volume */
        playAudio.set_should_apply_audio_focus(false);
        /* do not report metrics for Jingle */
        playAudio.set_report_metrics(false);
        connector_->sendMessage(std::move(msg));
    }
}

void JingleAudioClientPlayer::removeExpired() {
    listeners_.remove_if([](const auto& listener) {
        return listener.expired();
    });
}

void JingleAudioClientPlayer::notifyListeners(std::function<void(AudioPlayerListener& wp)> func) {
    std::for_each(listeners_.begin(), listeners_.end(), [&func](auto& wp) {
        if (auto listener = wp.lock()) {
            func(*listener);
        }
    });
    removeExpired();
}
void JingleAudioClientPlayer::onAudioClientEvent(const proto::AudioClientEvent& event) {
    worker_.add([this, event]() {
        handleAudioClientEvent(event);
    });
}

void JingleAudioClientPlayer::handleAudioClientEvent(const proto::AudioClientEvent& event) {
    if (event.has_player_descriptor() && event.player_descriptor().player_id() == descriptor_.player_id()) {
        if (!event.has_state()) {
            return;
        }
        auto sthis = shared_from_this();
        switch (event.state()) {
            case proto::AudioClientState::PLAYING:
                notifyListeners([&sthis](auto& listener) {
                    listener.onPlayingBegin(sthis);
                });
                break;
            case proto::AudioClientState::PAUSED:
                notifyListeners([&sthis](auto& listener) {
                    listener.onPlayingPaused(sthis);
                });
                break;
            case proto::AudioClientState::FINISHED:
                notifyListeners([&sthis](auto& listener) {
                    listener.onPlayingDone(sthis);
                });
                break;
            case proto::AudioClientState::FAILED: {
                SpeechKit::Error error(SpeechKit::Error::ErrorAudioPlayer, "Gstreamer audioclient player error: " + event.error_text());
                notifyListeners([&sthis, &error](auto& listener) {
                    listener.onPlayingError(sthis, error);
                });
                break;
            }
            default:
                // ignore all other player states
                break;
        }
    }
}
