#include "quasar_voice_dialog_linux_impl.h"

#include "sk_platform_info.h"

#include <yandex_io/libs/device/defines.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/speechkit_logger/speechkit_logger.h>
#include <yandex_io/libs/telemetry/telemetry.h>

#include <speechkit/OggOpusReader.h>
#include <speechkit/SoundLogger.h>
#include <speechkit/SpeechKit.h>

#include <chrono>
#include <memory>
#include <thread>

YIO_DEFINE_LOG_MODULE("alice");

using namespace quasar;

MetricaEventLogger::MetricaEventLogger(std::shared_ptr<YandexIO::IDevice> device)
    : device_(std::move(device))
    /* Events that should be skipped in reportEvent method */
    , eventsToSkip_({"ysk_core_partial_results"})
{
}

void MetricaEventLogger::reportEvent(const std::string& eventName, const EventArgs& args)
{
    if (eventsToSkip_.count(eventName)) {
        /* Skip unnecessary/spamming events */
        return;
    }
    std::unordered_map<std::string, std::string> keyValues(args.begin(), args.end());
    device_->telemetry()->reportKeyValues(eventName, keyValues);
}

SpeechKitLoggerGuard::SpeechKitLoggerGuard(const std::string& name, std::shared_ptr<SpeechKit::Logger> newLogger)
    : name_(name)
{
    YIO_LOG_INFO("Installing SpeechKit logger " << name_);

    // Call to getInstance ensures default SpeechKit::Logger is set
    SpeechKit::SpeechKit::getInstance();

    SpeechKit::Logger::setInstance(std::move(newLogger));
}

SpeechKitLoggerGuard::~SpeechKitLoggerGuard() {
    YIO_LOG_INFO("Uninstalling SpeechKit logger " << name_);
    SpeechKit::Logger::setInstance(nullptr);
}

QuasarVoiceDialogLinuxImpl::QuasarVoiceDialogLinuxImpl(std::shared_ptr<YandexIO::IDevice> device,
                                                       std::shared_ptr<SpeechKit::VoiceService::VoiceServiceListener> listener,
                                                       std::shared_ptr<AliceAudioSource> audioSource,
                                                       SpeechKit::AudioPlayer::SharedPtr audioPlayer)
    : device_(std::move(device))
    , loggerGuard_("QuasarDebugLogger", std::make_shared<YandexIO::SpeechkitLogger>())
    , platformInfo_(std::make_shared<SkPlatformInfo>(device_))
    , audioSource_(std::move(audioSource))
{
    YIO_LOG_INFO("create QuasarVoiceDialogLinuxImpl");

    eventLogger_ = std::make_shared<MetricaEventLogger>(device_);

    const auto spConfig = device_->configuration()->getServiceConfig("aliced");

    auto persistentStoragePath = tryGetString(spConfig, "sk_persistent_storage", "");
    YIO_LOG_DEBUG("SkPersistentStorage path: '" << persistentStoragePath << "'");
    if (!persistentStoragePath.empty()) {
        persistentStorage_ = std::make_shared<SkPersistentStorage>(std::move(persistentStoragePath));
    }

    const auto spotterStartSound = tryGetString(spConfig, "spotterStartSound", DEFAULT_SPOTTER_START_SOUND);
    const auto spotterCancelSound = tryGetString(spConfig, "spotterCancelSound", DEFAULT_SPOTTER_CANCEL_SOUND);

    earcons_.startVoiceInputEarcon = ::SpeechKit::OggOpusReader::readOpusFile(spotterStartSound);
    // cancelEarcon and recognitionErrorEarcon have the same sound (user shouldn't differentiate these events), it's not a typo
    earcons_.cancelEarcon = ::SpeechKit::OggOpusReader::readOpusFile(spotterCancelSound);
    earcons_.recognitionErrorEarcon = ::SpeechKit::OggOpusReader::readOpusFile(spotterCancelSound);

    initSpeechKit();

    Y_VERIFY(listener != nullptr);
    voiceDialog_ = ::SpeechKit::VoiceService::create({}, listener, audioSource_, std::move(audioPlayer));
}

void QuasarVoiceDialogLinuxImpl::initSpeechKit()
{
    auto sk = ::SpeechKit::SpeechKit::getInstance();
    initLogging();
    sk->setDeviceId(device_->deviceId());
    sk->setEventLogger(eventLogger_);
    sk->setPersistentStorage(persistentStorage_);
    sk->setPlatformInfo(platformInfo_);
}

void QuasarVoiceDialogLinuxImpl::startVoiceInput(const SpeechKit::UniProxy::Header& header, const std::string& voiceInput, bool jingle)
{
    if (jingle) {
        voiceDialog_->startVoiceInput(header, voiceInput, earcons_);
    } else {
        voiceDialog_->startVoiceInput(header, voiceInput, ::SpeechKit::VoiceDialog::Earcons());
    }
}

void QuasarVoiceDialogLinuxImpl::stopInput() {
    voiceDialog_->stopRecognition();
}

void QuasarVoiceDialogLinuxImpl::startMusicInput(const SpeechKit::UniProxy::Header& header, const std::string& musicInput, bool jingle)
{
    if (jingle) {
        voiceDialog_->startMusicInput(header, musicInput, earcons_);
    } else {
        voiceDialog_->startMusicInput(header, musicInput, ::SpeechKit::VoiceDialog::Earcons());
    }
}

void QuasarVoiceDialogLinuxImpl::startPhraseSpotter()
{
    voiceDialog_->startPhraseSpotter();
}

void QuasarVoiceDialogLinuxImpl::startCommandSpotter(SpeechKit::PhraseSpotterSettings settings) {
    voiceDialog_->startCommandSpotter(settings);
}

void QuasarVoiceDialogLinuxImpl::stopCommandSpotter() {
    voiceDialog_->stopCommandSpotter();
}

void QuasarVoiceDialogLinuxImpl::cancel(bool silent)
{
    voiceDialog_->cancel(silent);
}

void QuasarVoiceDialogLinuxImpl::playCancelSound()
{
    voiceDialog_->playCancelSound();
}

void QuasarVoiceDialogLinuxImpl::prepare(const std::string& uuid, const std::string& yandexUid,
                                         const std::string& timezone, const std::string& group, const std::string& subgroup)
{
    auto sk = ::SpeechKit::SpeechKit::getInstance();
    const bool stopConnection = sk->getUuid() != uuid || sk->getYandexUid() != yandexUid || platformInfo_->getOlsonDbTimezoneName() != timezone || platformInfo_->getQuasmodromGroup() != group || platformInfo_->getQuasmodromSubgroup() != subgroup;
    sk->setUuid(uuid);
    sk->setYandexUid(yandexUid);
    platformInfo_->setTimezone(timezone);
    platformInfo_->setQuasmodromGroups(group, subgroup);

    if (stopConnection) {
        YIO_LOG_INFO("Restart dialog connection.");
        voiceDialog_->stopConnection();
    }
    voiceDialog_->startConnection();
}

void QuasarVoiceDialogLinuxImpl::sendEvent(const SpeechKit::UniProxy::Header& header, const std::string& payload)
{
    voiceDialog_->sendEvent(header, payload);
}

void QuasarVoiceDialogLinuxImpl::startTextInput(const SpeechKit::UniProxy::Header& header, const std::string& payload)
{
    voiceDialog_->startTextInput(header, payload);
}

void QuasarVoiceDialogLinuxImpl::update(const AliceConfig& config)
{
    setApiKey(config.getApiKey());
    setUseDnsCache(config.getDnsCacheEnabled());
    setLoggingSettings(config.getSubThresholdSendRate(), config.getSoundLogMaxParallelSendings());
    setLogLevel(config.getSpeechkitLogLevel());
}

void QuasarVoiceDialogLinuxImpl::setSettings(::SpeechKit::VoiceServiceSettings settings)
{
    voiceDialog_->setSettings(settings);
}

void QuasarVoiceDialogLinuxImpl::setPlayer(::SpeechKit::AudioPlayer::SharedPtr player)
{
    if (player == nullptr) {
        YIO_LOG_ERROR_EVENT("VoiceDialog.PlayerIsNull", "Can't set SpeechKit::VoiceService nullptr player");
        return;
    }

    voiceDialog_->setAudioPlayer(std::move(player));
}

void QuasarVoiceDialogLinuxImpl::initLogging() {
    const auto yiodConfig = device_->configuration()->getServiceConfig("yiod");
    defaultLogLevel_ = getString(getJson(yiodConfig, "Logging"), "level");
    YandexIO::setSpeechkitLogLevel(defaultLogLevel_);
}

void QuasarVoiceDialogLinuxImpl::setApiKey(const std::string& key) {
    auto sk = ::SpeechKit::SpeechKit::getInstance();
    if (sk->getApiKey() != key) {
        sk->setApiKey(key);
        YIO_LOG_INFO("Stop dialog connection due to api key changing.");
        voiceDialog_->stopConnection();
    }
}

void QuasarVoiceDialogLinuxImpl::setUseDnsCache(bool useDnsCache) {
    ::SpeechKit::SpeechKit::setDnsCacheEnabled(useDnsCache);
}

void QuasarVoiceDialogLinuxImpl::setLoggingSettings(std::chrono::milliseconds subthresholdRate, unsigned parallelSendings) {
    auto sk = ::SpeechKit::SoundLogger::getInstance();
    sk->setSubThresholdSendRate(subthresholdRate);
    sk->setMaximumParallelSendings(parallelSendings);
}

void QuasarVoiceDialogLinuxImpl::setLogLevel(std::optional<std::string> optLevel) {
    auto levelToSet = optLevel.value_or(defaultLogLevel_);
    YandexIO::setSpeechkitLogLevel(levelToSet);
}

void QuasarVoiceDialogLinuxImpl::setSynchronizeStatePayload(const std::string& synchronizeStatePayload) {
    voiceDialog_->setSynchronizeStatePayload(synchronizeStatePayload);
}

void QuasarVoiceDialogLinuxImpl::startInterruptionSpotter()
{
    voiceDialog_->startInterruptionPhraseSpotter();
}

void QuasarVoiceDialogLinuxImpl::stopInterruptionSpotter()
{
    voiceDialog_->stopInterruptionPhraseSpotter();
}

void QuasarVoiceDialogLinuxImpl::updatePrevReqId(const std::string& requestId)
{
    if (persistentStorage_) {
        YIO_LOG_INFO("Set prevReqId to " << requestId);
        persistentStorage_->setString("prev_req_id", requestId);
    }
}

void QuasarVoiceDialogLinuxImpl::onTtsStarted(const std::string& messageId) {
    voiceDialog_->onTTSPlayerStart(messageId);
}

void QuasarVoiceDialogLinuxImpl::onTtsCompleted(const std::string& messageId) {
    voiceDialog_->onTTSPlayerEnd(messageId);
}

void QuasarVoiceDialogLinuxImpl::onTtsError(const SpeechKit::Error& error, const std::string& messageId) {
    voiceDialog_->onTTSPlayerError(error, messageId);
}
