#include "random_sound_logger.h"

#include <yandex_io/libs/base/crc32.h>
#include <yandex_io/libs/base/named_callback_queue.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>

YIO_DEFINE_LOG_MODULE("random_sound_logger");

using namespace quasar;

namespace {

    constexpr int DEFAULT_PERIOD_SEC = 60 * 60;
    constexpr int DEFAULT_DURATION_MS = 10 * 1000;
    constexpr int WAIT_RESULT_TIMEOUT_MS = 5 * 1000;

} // namespace

RandomSoundLogger::RandomSoundLogger(
    std::shared_ptr<ICallbackQueue> callbackQueue,
    const YandexIO::DeviceID& deviceId,
    const AliceConfig& aliceConfig,
    const AliceDeviceState& aliceDeviceState)
    : aliceConfig_(aliceConfig)
    , aliceDeviceState_(aliceDeviceState)
    , deviceId_{deviceId}
    , asyncQueue_{std::move(callbackQueue)}
    , uniqueCallback_{asyncQueue_, UniqueCallback::ReplaceType::INSERT_BACK}
{
}

RandomSoundLogger::~RandomSoundLogger() {
    uniqueCallback_.reset();
    asyncQueue_->add([this] {
        if (state_ == State::STREAMING) {
            stopStreaming();
        }
    }, lifetime_);
    asyncQueue_->wait();
    lifetime_.die();
    asyncQueue_.reset();
}

void RandomSoundLogger::onAudioSourceStopped(SpeechKit::AudioSource::SharedPtr /*audioSource*/) {
    asyncQueue_->add([this] {
        if (state_ == State::STREAMING) {
            stopStreaming();
        }
    });
}

void RandomSoundLogger::onAudioSourceError(SpeechKit::AudioSource::SharedPtr audioSource, const SpeechKit::Error& error) {
    YIO_LOG_WARN(error.getString());

    onAudioSourceStopped(std::move(audioSource));
}

void RandomSoundLogger::onAudioSourceData(SpeechKit::AudioSource::SharedPtr /*audioSource*/,
                                          SpeechKit::CompositeSoundBuffer::SharedPtr soundBuffer)
{
    asyncQueue_->add([this, soundBuffer = std::move(soundBuffer)]() mutable {
        if (!checkActive()) {
            return;
        }

        if (state_ == State::WAITING_FOR_NEXT_POINT) {
            YIO_LOG_INFO("Log point reached. Starting streaming.");

            soundLogStream_ = SpeechKit::SoundLogger::getInstance()->createStream(shared_from_this(), {});
            Json::Value extraPayload{aliceDeviceState_.buildSoundLogExtra(aliceConfig_)};
            extraPayload["context"] = "random";
            extraPayload["durationMs"] = std::to_string(config_.duration.count());
            soundLogStream_->setExtraPayload(jsonToString(extraPayload));

            if (config_.channelNames.count("*")) {
                std::set<std::string> bufferChannelNames;
                for (const auto& [channelName, _] : soundBuffer->getChannelToBuffers()) {
                    bufferChannelNames.emplace_hint(bufferChannelNames.end(), channelName);
                }
                soundLogStream_->setChannelNames(bufferChannelNames);
            } else if (config_.channelNames.empty()) {
                soundLogStream_->setChannelNames({soundBuffer->getMainChannelName()});
            } else {
                soundLogStream_->setChannelNames(config_.channelNames);
            }

            state_ = State::STREAMING;
        }

        appendSound(std::move(soundBuffer));

        if (isAllSoundSent()) {
            YIO_LOG_INFO("All sound sent. Stop streaming.")
            stopStreaming();
        }
    });
}

void RandomSoundLogger::onSuccess(const std::string& /*id*/, const std::vector<std::string>& messageIds) {
    std::stringstream logStr;
    logStr << "Success logging. Message IDs:";
    for (const auto& messageId : messageIds) {
        logStr << " " << messageId;
    }
    YIO_LOG_INFO(logStr.str());

    asyncQueue_->add([this] {
        onStreamingEnded();
    });
}

void RandomSoundLogger::onFail(const std::string& /*id*/, const std::string& error) {
    YIO_LOG_ERROR_EVENT("RandomSoundLogger.OnFail", "Fail logging. Error: " << error);

    asyncQueue_->add([this] {
        onStreamingEnded();
    });
}

void RandomSoundLogger::applyConfig(const Json::Value& jsonConfig) {
    asyncQueue_->add([this, config{Config::fromJson(jsonConfig)}] {
        if (config_ != config) {
            YIO_LOG_INFO("Received new config: " << config.toString());

            if (config_.period != config.period) {
                reinitJitter(config.period);
            }
            config_ = config;

            switch (state_) {
                case State::WAITING_FOR_RESULT:
                    // After the end of the waiting, the next log point will be scheduled automatically.
                    // So do nothing
                    return;
                case State::STREAMING:
                    if (!config_.enabled) {
                        stopStreaming();
                    }
                    break;
                case State::WAITING_FOR_NEXT_POINT:
                default:
                    scheduleNextLogPoint();
            }
        }
    });
}

void RandomSoundLogger::onCapabilityStateChanged(const std::shared_ptr<YandexIO::ICapability>& /*capability*/, const NAlice::TCapabilityHolder& state) {
    if (!state.HasDeviceStateCapability() || !state.GetDeviceStateCapability().GetState().HasDeviceState()) {
        return;
    }
    const auto& deviceState = state.GetDeviceStateCapability().GetState().GetDeviceState();
    if (!deviceState.HasMicsMuted()) {
        return;
    }
    setMicsMuted(deviceState.GetMicsMuted());
}

void RandomSoundLogger::onCapabilityEvents(const std::shared_ptr<YandexIO::ICapability>& /*capability*/, const std::vector<NAlice::TCapabilityEvent>& /*events*/) {
}

void RandomSoundLogger::setMicsMuted(bool muted) {
    asyncQueue_->add([this, muted] {
        if (micsMuted_ == muted) {
            return;
        }

        micsMuted_ = muted;

        if (config_.enabled) {
            const auto now = std::chrono::steady_clock::now();

            if (micsMuted_) {
                switch (state_) {
                    case State::STREAMING:
                        stopStreaming();
                        break;
                    case State::WAITING_FOR_NEXT_POINT:
                        muteSubtractionPoint_ = now;
                        break;
                    default:
                        // do nothing
                        break;
                }
            } else {
                nextLogPoint_ += (now - muteSubtractionPoint_);
                YIO_LOG_INFO("Microphones is enabled. Time to next log: " << std::chrono::duration_cast<std::chrono::milliseconds>(nextLogPoint_ - now).count() << "ms");
            }
        }
    });
}

void RandomSoundLogger::reinitJitter(const std::chrono::seconds& period) {
    double milliseconds = std::chrono::duration_cast<std::chrono::milliseconds>(period).count();
    double prob = 1.0 / milliseconds;
    randomGenerator_.seed(getCrc32(deviceId_) + getNowTimestampMs());
    jitter_ = std::geometric_distribution<ssize_t>(prob);
}

bool RandomSoundLogger::checkActive() {
    if (!config_.enabled || state_ == State::WAITING_FOR_RESULT || micsMuted_) {
        return false;
    }

    auto now = std::chrono::steady_clock::now();

    return now > nextLogPoint_;
}

void RandomSoundLogger::appendSound(SpeechKit::CompositeSoundBuffer::SharedPtr soundBuffer) {
    const auto bufferDuration = std::chrono::milliseconds(soundBuffer->getMainBuffer()->calculateTimeMs());
    const auto remainingDuration = config_.duration - durationSent_;

    if (remainingDuration >= bufferDuration) {
        soundLogStream_->appendSound(std::move(soundBuffer));
        durationSent_ += bufferDuration;
        return;
    }

    SpeechKit::CompositeSoundBuffer::ChannelNameToBuffer channelToBuffer;
    for (const auto& [channelName, buffer] : soundBuffer->getChannelToBuffers()) {
        const auto newSize = SpeechKit::SoundBuffer::calculateBufferSize(buffer->getInfo(), remainingDuration);
        std::vector<uint8_t> newData{buffer->getData().begin(), buffer->getData().begin() + newSize};
        auto newBuffer = std::make_shared<SpeechKit::SoundBuffer>(buffer->getInfo(), std::move(newData));
        channelToBuffer.emplace(channelName, std::move(newBuffer));
    }
    soundLogStream_->appendSound(std::make_shared<SpeechKit::CompositeSoundBuffer>(
        soundBuffer->getMainChannelName(), std::move(channelToBuffer)));
    durationSent_ += remainingDuration;
}

bool RandomSoundLogger::isAllSoundSent() const {
    return durationSent_ >= config_.duration;
}

void RandomSoundLogger::scheduleNextLogPoint() {
    if (!config_.enabled) {
        return;
    }

    const auto now = std::chrono::steady_clock::now();

    if (micsMuted_) {
        muteSubtractionPoint_ = now;
    }

    const std::chrono::milliseconds timeToNextCall(jitter_(randomGenerator_));
    YIO_LOG_INFO("Time to next log: " << timeToNextCall.count() << "ms");
    nextLogPoint_ = now + timeToNextCall;
}

void RandomSoundLogger::stopStreaming() {
    soundLogStream_->stopStreaming();
    durationSent_ = std::chrono::milliseconds::zero();
    state_ = State::WAITING_FOR_RESULT;
    uniqueCallback_.executeDelayed([this] {
        YIO_LOG_ERROR_EVENT("RandomSoundLogger.StopStreamingTimeout", "Wait result timeout expired.");
        onStreamingEnded();
    }, std::chrono::milliseconds(WAIT_RESULT_TIMEOUT_MS), lifetime_);
}

void RandomSoundLogger::onStreamingEnded() {
    uniqueCallback_.reset();
    soundLogStream_.reset();
    scheduleNextLogPoint();
    state_ = State::WAITING_FOR_NEXT_POINT;
}

bool RandomSoundLogger::Config::operator!=(const RandomSoundLogger::Config& other) const {
    return enabled != other.enabled ||
           period != other.period ||
           duration != other.duration ||
           channelNames != other.channelNames;
}

std::string RandomSoundLogger::Config::toString() const {
    std::stringstream ss;
    ss << "enabled=" << std::to_string(enabled) << " "
       << "periodSec=" << std::to_string(period.count()) << " "
       << "durationMs=" << std::to_string(duration.count());
    return ss.str();
}

RandomSoundLogger::Config RandomSoundLogger::Config::fromJson(const Json::Value& jsonConfig) {
    RandomSoundLogger::Config config;
    config.enabled = tryGetBool(jsonConfig, "enabled", false);
    if (!config.enabled) {
        return config;
    }

    config.period = std::chrono::seconds(tryGetInt(jsonConfig, "periodSec", DEFAULT_PERIOD_SEC));
    config.duration = std::chrono::milliseconds(tryGetInt(jsonConfig, "durationMs", DEFAULT_DURATION_MS));
    config.channelNames = tryGetEmplaceableStringSet<std::set<std::string>>(jsonConfig, "channelNames", {});

    return config;
}
