#pragma once

#include <yandex_io/sdk/interfaces/i_capability.h>

#include <yandex_io/libs/device/i_device.h>
#include <yandex_io/libs/threading/i_callback_queue.h>
#include <yandex_io/libs/threading/unique_callback.h>
#include <yandex_io/services/aliced/alice_config/alice_config.h>
#include <yandex_io/services/aliced/device_state/alice_device_state.h>

#include <speechkit/AudioSource.h>
#include <speechkit/SoundLogger.h>

#include <json/json.h>

#include <chrono>
#include <random>
#include <set>
#include <string>

namespace quasar {

    class RandomSoundLogger
        : public SpeechKit::AudioSource::AudioSourceListener,
          public SpeechKit::SoundLogger::ResultSink,
          public YandexIO::ICapability::IListener,
          public std::enable_shared_from_this<RandomSoundLogger> {
    public:
        RandomSoundLogger(
            std::shared_ptr<ICallbackQueue> callbackQueue,
            const YandexIO::DeviceID& deviceId,
            const AliceConfig& aliceConfig,
            const AliceDeviceState& aliceDeviceState);
        ~RandomSoundLogger();

        // from SpeechKit::AudioSource::AudioSourceListener
        void onAudioSourceStarted(SpeechKit::AudioSource::SharedPtr /*audioSource*/) override{};
        void onAudioSourceStopped(SpeechKit::AudioSource::SharedPtr audioSource) override;
        void onAudioSourceError(SpeechKit::AudioSource::SharedPtr audioSource, const SpeechKit::Error& error) override;
        void onAudioSourceData(SpeechKit::AudioSource::SharedPtr audioSource,
                               SpeechKit::CompositeSoundBuffer::SharedPtr soundBuffer) override;

        // from SpeechKit::SoundLogger::ResultSink
        void onSuccess(const std::string& id, const std::vector<std::string>& messageIds) override;
        void onFail(const std::string& id, const std::string& error) override;

        // from ICapability::IListener
        void onCapabilityStateChanged(const std::shared_ptr<YandexIO::ICapability>& capability, const NAlice::TCapabilityHolder& state) override;
        void onCapabilityEvents(const std::shared_ptr<YandexIO::ICapability>& capability, const std::vector<NAlice::TCapabilityEvent>& events) override;

        void applyConfig(const Json::Value& jsonConfig);

    private:
        void reinitJitter(const std::chrono::seconds& period);
        bool checkActive();
        void appendSound(SpeechKit::CompositeSoundBuffer::SharedPtr soundBuffer);
        bool isAllSoundSent() const;
        void scheduleNextLogPoint();
        void stopStreaming();
        void onStreamingEnded();
        void setMicsMuted(bool muted);

    private:
        struct Config {
            bool enabled{false};
            std::chrono::seconds period{0};
            std::chrono::milliseconds duration{0};
            std::set<std::string> channelNames;

            bool operator!=(const Config& other) const;

            std::string toString() const;

            static Config fromJson(const Json::Value& jsonConfig);
        };

        enum class State {
            STREAMING,
            WAITING_FOR_RESULT,
            WAITING_FOR_NEXT_POINT
        };

        const AliceConfig& aliceConfig_;
        const AliceDeviceState& aliceDeviceState_;
        Config config_;
        const YandexIO::DeviceID deviceId_;
        std::shared_ptr<ICallbackQueue> asyncQueue_;
        Lifetime lifetime_;
        UniqueCallback uniqueCallback_;
        std::geometric_distribution<ssize_t> jitter_;
        std::mt19937 randomGenerator_;
        State state_{State::WAITING_FOR_NEXT_POINT};
        std::chrono::time_point<std::chrono::steady_clock> nextLogPoint_;
        SpeechKit::SoundLogger::StreamPtr soundLogStream_;
        std::chrono::milliseconds durationSent_{0};

        std::chrono::time_point<std::chrono::steady_clock> muteSubtractionPoint_;
        bool micsMuted_{false};
    };

} // namespace quasar
