#include "http_pinger.h"

#include <yandex_io/libs/logging/logging.h>

using namespace quasar;

HttpPinger::HttpPinger(Settings settings, std::shared_ptr<YandexIO::IDevice> device)
    : client_{std::move(settings.name), device}
{
    client_.allowRequestResponseLogging(false);
    client_.setCalcRetryDelayFunction(makeRetryCalculator(std::move(settings.retryCalculator)));
    client_.setRetriesCount(settings.retriesCount);
    client_.setTimeout(settings.timeout);
}

HttpPinger::PingResult HttpPinger::ping(const HttpPinger::PingRequest& request)
{
    using namespace std::chrono;

    result_.reset();

    auto begin = steady_clock::now();
    const auto postResult = client_.post("uniproxy-ping", request.url, request.data);
    auto end = steady_clock::now();

    if (!result_.has_value()) {
        result_ = PingResult{};
    }
    result_->responseCode = postResult.responseCode;
    result_->latency = duration_cast<milliseconds>(end - begin);

    return *result_;
}

const std::optional<HttpPinger::PingResult>& HttpPinger::getLastResult() const {
    return result_;
}

HttpClient::CalcRetryDelayFunction HttpPinger::makeRetryCalculator(RetryCalculator retryCalculator)
{
    return [this, retryCalculator = std::move(retryCalculator)](int retryNum) {
        YIO_LOG_DEBUG("retryNum: " << retryNum);

        if (!result_.has_value()) {
            result_ = PingResult{};
        }
        result_->retries = retryNum + 1; // retryNum starts from 0

        return retryCalculator(retryNum);
    };
}
