#include "uniproxy_pinger.h"

#include <yandex_io/libs/base/crc32.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>

#include <future>
#include <optional>
#include <random>
#include <string>
#include <vector>

using namespace quasar;

namespace {

    template <typename CharT, typename ValueT>
    std::basic_ostream<CharT>& operator<<(std::basic_ostream<CharT>& os, const std::vector<ValueT>& vector)
    {
        os << "[";
        for (const auto& el : vector) {
            os << el << ", ";
        }
        os << "]";
        return os;
    }

} // namespace

std::optional<UniProxyPinger::Settings> UniProxyPinger::parseSettings(const Json::Value& config)
{
    if (!config.isObject()) {
        return std::nullopt;
    }

    Settings settings;

    settings.randomUrls = tryGetVector<std::string>(config, "randomUrls");
    settings.url = tryGetString(config, "url");
    settings.timeout = tryGetMillis(config, "timeoutMs", settings.timeout);
    settings.interval = tryGetMillis(config, "intervalMs", settings.interval);
    settings.payloadBytes = tryGetUInt32(config, "payloadBytes", settings.payloadBytes);
    settings.retriesCount = tryGetUInt32(config, "retriesCount", settings.retriesCount);
    settings.parallelPings = tryGetUInt32(config, "parallelPings", settings.parallelPings);

    return settings;
}

UniProxyPinger::UniProxyPinger(Settings settings, std::shared_ptr<::YandexIO::IDevice> device)
    : device_{std::move(device)}
    , asyncQueue_{"UniProxyPinger"}
    , randomGenerator_(getCrc32(device_->deviceId()) + getNowTimestampMs())
    , settings_{std::move(settings)}
    , payload_{generatePayload(settings_.payloadBytes)}
{
    YIO_LOG_INFO("Create UniProxyPinger. url: " << settings_.url
                                                << ", settings_.randomUrls: " << settings_.randomUrls
                                                << ", parallelPings: " << settings_.parallelPings
                                                << ", intervalMs: " << settings_.interval.count()
                                                << ", timeoutMs: " << settings_.timeout.count()
                                                << ", retries: " << settings_.retriesCount);

    if (settings_.parallelPings > 0 && urlPresents()) {
        asyncQueue_.add([this] {
            ping();
        }, lifetime_);
    }
}

UniProxyPinger::~UniProxyPinger()
{
    YIO_LOG_INFO("Destroy UniProxyPinger");
    lifetime_.die();
}

const UniProxyPinger::Settings& UniProxyPinger::getSettings() const {
    return settings_;
}

void UniProxyPinger::ping()
{
    YIO_LOG_DEBUG("Start pings");

    std::vector<std::unique_ptr<HttpPinger>> pingers;
    std::vector<std::future<HttpPinger::PingResult>> results;
    {
        std::string name{"UniProxyHttpPinger" + std::to_string(0)};
        std::string url{!settings_.url.empty() ? settings_.url : getRandomUrl()};
        pingers.push_back(createPinger(std::move(name)));
        results.push_back(std::async(std::launch::deferred,
                                     &HttpPinger::ping, pingers.back().get(), createRequest(std::move(url))));
    }
    for (std::uint32_t i{1}; i != settings_.parallelPings; ++i) {
        std::string name{"UniProxyHttpPinger" + std::to_string(i)};
        pingers.push_back(createPinger(std::move(name)));
        results.push_back(std::async(std::launch::async,
                                     &HttpPinger::ping, pingers.back().get(), createRequest(getRandomUrl())));
    }

    std::optional<std::chrono::milliseconds> minLatency;
    std::uint32_t failedRequests{0};
    std::vector<std::uint32_t> retries(results.size());

    for (std::size_t i{0}; i != results.size(); ++i) {
        try {
            const auto pingResult = results[i].get();
            if (pingResult.responseCode != 200) {
                ++failedRequests;
            }
            if (!minLatency.has_value() || pingResult.latency < *minLatency) {
                minLatency = pingResult.latency;
            }
            retries[i] = pingResult.retries;
        } catch (const std::exception& e) {
            YIO_LOG_WARN("UniProxy ping failed: " << e.what());
            const auto failedResult{pingers[i]->getLastResult()};
            if (failedResult.has_value()) {
                retries[i] = failedResult->retries;
            }
            ++failedRequests;
        }
    }

    Json::Value eventArgs;
    if (minLatency.has_value()) {
        eventArgs["latency_ms"] = static_cast<std::uint32_t>(minLatency->count());
    }
    eventArgs["requests"] = results.size();
    eventArgs["failed_requests"] = failedRequests;
    eventArgs["retries"] = vectorToJson(retries);

    device_->telemetry()->reportEvent("uniProxyHttpPing", jsonToString(eventArgs));

    schedulePing();
}

std::unique_ptr<HttpPinger> UniProxyPinger::createPinger(std::string name) const {
    HttpPinger::RetryCalculator retryCalculator = [](int retryNum) {
        // The same policy for reconnection as in SpeechKit during requests.
        const std::chrono::seconds maxDelay{30};

        std::chrono::seconds delay{2 << retryNum};
        if (delay < maxDelay) {
            return delay;
        }

        return maxDelay;
    };

    const HttpPinger::Settings settings{
        .name = std::move(name),
        .timeout = settings_.timeout,
        .retriesCount = settings_.retriesCount,
        .retryCalculator = std::move(retryCalculator),
    };
    return std::make_unique<HttpPinger>(settings, device_);
}

HttpPinger::PingRequest UniProxyPinger::createRequest(std::string url) const {
    return HttpPinger::PingRequest{
        .url = std::move(url),
        .data = payload_,
    };
}

void UniProxyPinger::schedulePing()
{
    YIO_LOG_DEBUG("Schedule pings in " << settings_.interval.count() << " ms");

    asyncQueue_.addDelayed([this] {
        ping();
    }, settings_.interval, lifetime_);
}

bool UniProxyPinger::urlPresents() const {
    return !settings_.url.empty() || !settings_.randomUrls.empty();
}

std::string UniProxyPinger::getRandomUrl() const {
    if (settings_.randomUrls.empty()) {
        return settings_.url;
    }

    std::uniform_int_distribution<std::size_t> d{0, settings_.randomUrls.size() - 1};
    return settings_.randomUrls[d(randomGenerator_)];
}

std::string UniProxyPinger::generatePayload(std::uint32_t payloadBytes) const {
    std::string workload(payloadBytes, '\0');

    std::uniform_int_distribution<char> d{'a', 'z'};
    std::generate(workload.begin(), workload.end(), [this, &d] {
        return d(randomGenerator_);
    });

    return workload;
}
