#include "account_storage.h"

#include "auth_endpoint.h"
#include "util/system/yassert.h"
#include "yandex_io/protos/account_storage.pb.h"

#include <yandex_io/libs/base/crc32.h>
#include <yandex_io/libs/base/persistent_file.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/protobuf_utils/debug.h>

YIO_DEFINE_LOG_MODULE("auth");

using namespace quasar;

namespace {

    constexpr auto PROLONG_TOKEN_INTERVAL = std::chrono::days{31};

    constexpr auto MIN_AUTH_TOKEN_UPDATE_BACKOFF = std::chrono::minutes{1};
    constexpr auto MAX_AUTH_TOKEN_UPDATE_BACKOFF = std::chrono::days{31};

    constexpr auto MIN_REVOKE_ACCOUNT_BACKOFF = std::chrono::minutes{1};
    constexpr auto MAX_REVOKE_ACCOUNT_BACKOFF = std::chrono::days{31};

    constexpr auto REQUEST_TOKEN_RETRIES = 3;
    constexpr auto REQUEST_TOKEN_DELAY = std::chrono::seconds{1};

    AccountStorage::TimePoint timePointFromProto(int64_t ts) {
        return AccountStorage::TimePoint() + std::chrono::seconds{ts};
    }

    int64_t timePointToProto(AccountStorage::TimePoint ts) {
        return std::chrono::time_point_cast<std::chrono::seconds>(ts).time_since_epoch().count();
    }

} // namespace

AccountStorage::AccountStorage(std::shared_ptr<YandexIO::IDevice> device, std::shared_ptr<ICallbackQueue> callbackQueue)
    : device_(std::move(device))
    , httpClient_("passport", device_)
    , timeoutGenerator_(getCrc32(device_->deviceId()) + getNowTimestampMs())
    , callbackQueue_(std::move(callbackQueue))
    , updateAccountBackoffer_(0, 0)
    , revokeAccountBackoffer_(0, 0)
{
    auto config = device_->configuration()->getServiceConfig(AuthEndpoint::SERVICE_NAME);
    fileName_ = getString(config, "accountStorageFile");

    passportUrl_ = getString(config, "passportUrl");
    loginUrl_ = getString(config, "loginUrl");

    xTokenClientId_ = getString(config, "xTokenClientId");
    xTokenClientSecret_ = getString(config, "xTokenClientSecret");

    authTokenClientId_ = getString(config, "authTokenClientId");
    authTokenClientSecret_ = getString(config, "authTokenClientSecret");
    deviceName_ = getString(config, "deviceName");

    httpClient_.allowRequestResponseLogging(false); // Requests and responses contain confidential auth tokens
    httpClient_.setTimeout(std::chrono::milliseconds{5000});
    httpClient_.setRetriesCount(tryGetInt(config, "oauthRetryCount", 2));

    if (config.isMember("oauthMinRequestTimeMs")) {
        httpClient_.setMinRetriesTime(std::chrono::milliseconds{config["oauthMinRequestTimeMs"].asInt()});
    }

    loadAccounts();

    updateAccountBackoffer_.initCheckPeriod(
        MIN_AUTH_TOKEN_UPDATE_BACKOFF,
        MIN_AUTH_TOKEN_UPDATE_BACKOFF,
        MAX_AUTH_TOKEN_UPDATE_BACKOFF);

    revokeAccountBackoffer_.initCheckPeriod(
        MIN_REVOKE_ACCOUNT_BACKOFF,
        MIN_REVOKE_ACCOUNT_BACKOFF,
        MAX_REVOKE_ACCOUNT_BACKOFF);
}

AccountStorage::ErrorCode AccountStorage::requestAccount(const std::string& authCode, proto::AccountType accountType, bool withXToken, proto::AccountInfo& accountInfo)
{
    if (accountType == proto::AccountType::UNDEFINED) {
        throw std::runtime_error("Cannot request account with type UNDEFINED");
    }

    try {
        ErrorCode errorCode{ErrorCode::OK};

        if (withXToken) {
            errorCode = requestAccount(XCode{.code = authCode}, accountInfo);
        } else {
            errorCode = requestAccount(OAuthCode{.code = authCode}, accountInfo);
        }

        if (errorCode != ErrorCode::OK) {
            return errorCode;
        }

        accountInfo.set_type(accountType);
        addAccount(accountInfo);

        return errorCode;
    } catch (const std::exception& e) {
        YIO_LOG_ERROR_EVENT("AccountStorage.FailedGetAccountByCode", "Error: " << e.what());
        return ErrorCode::NO_INTERNET;
    }
}

AccountStorage::ErrorCode AccountStorage::requestAccount(const XCode& xcode, proto::AccountInfo& accountInfo)
{
    {
        const auto errorCode = requestXToken(xcode, accountInfo);
        if (errorCode != ErrorCode::OK) {
            return errorCode;
        }
    }

    {
        const auto errorCode = requestOAuthToken(XToken{.token = accountInfo.xtoken()}, accountInfo);
        if (errorCode != ErrorCode::OK) {
            return errorCode;
        }
    }

    return ErrorCode::OK;
}

AccountStorage::ErrorCode AccountStorage::requestAccount(const OAuthCode& oauthCode, proto::AccountInfo& accountInfo)
{
    {
        const auto errorCode = requestOAuthToken(oauthCode, accountInfo);
        if (errorCode != ErrorCode::OK) {
            return errorCode;
        }
    }

    {
        const auto errorCode = requestUid(OAuthToken{.token = accountInfo.auth_token()}, accountInfo);
        if (errorCode != ErrorCode::OK) {
            return errorCode;
        }
    }

    return ErrorCode::OK;
}

AccountStorage::ErrorCode AccountStorage::requestXToken(const XCode& xcode, proto::AccountInfo& accountInfo)
{
    std::stringstream requestData;
    requestData << "grant_type=authorization_code&code=" << xcode.code
                << "&client_id=" << xTokenClientId_ << "&client_secret=" << xTokenClientSecret_
                << "&device_id=" << urlEncode(device_->deviceId()) << "&device_name=" << urlEncode(deviceName_);

    const auto oauthUrl = passportUrl_ + "/token";
    YIO_LOG_INFO("Sending xtoken request to url '" << oauthUrl << "' for authCode " << maskToken(xcode.code));

    const auto tokenResponse = httpClient_.post("auth-xcode", oauthUrl, requestData.str());

    YIO_LOG_INFO("Response code: " << tokenResponse.responseCode);

    const auto body = tryParseJson(tokenResponse.body);
    if (!body.has_value()) {
        YIO_LOG_ERROR_EVENT("AccountStorage.BadJson.AccountByCode", "Can't parse xtoken response");
        return ErrorCode::CODE_EXPIRED;
    }

    const auto& jsonBody = *body;
    // TODO: more strict rules
    if (!isSuccessHttpCode(tokenResponse.responseCode)) {
        const auto error = tryGetString(jsonBody, "error", "");
        const auto description = tryGetString(jsonBody, "error_description", "");
        YIO_LOG_WARN("Can't get token. Error: " << error << " Description: " << description);
        return ErrorCode::CODE_EXPIRED;
    }

    accountInfo.set_xtoken(getString(jsonBody, "access_token"));
    accountInfo.set_refresh_token(getString(jsonBody, "refresh_token"));

    YIO_LOG_INFO("authCode=" << maskToken(xcode.code)
                             << " got xtoken=" << maskToken(accountInfo.xtoken())
                             << " and refresh_token=" << maskToken(accountInfo.refresh_token()));

    return ErrorCode::OK;
}

AccountStorage::ErrorCode AccountStorage::requestOAuthToken(const XToken& xtoken, proto::AccountInfo& accountInfo)
{
    for (auto i = 0; i != REQUEST_TOKEN_RETRIES; ++i) {
        const auto result = tryRequestOAuthToken(xtoken, accountInfo);
        switch (result) {
            case OAuthResponseResult::INVALID_GRANT: {
                /* Not very smart backoff */
                std::this_thread::sleep_for(REQUEST_TOKEN_DELAY);
                /* INVALID_GRANT should be retried */
                continue;
            }
            case OAuthResponseResult::OK: {
                return ErrorCode::OK;
            }
            case OAuthResponseResult::ERROR: {
                return ErrorCode::WRONG_TOKEN;
            }
        }
    }

    return ErrorCode::WRONG_TOKEN;
}

AccountStorage::ErrorCode AccountStorage::requestOAuthToken(const OAuthCode& oauthCode, proto::AccountInfo& accountInfo)
{
    std::stringstream requestData;
    requestData << "grant_type=authorization_code&code=" << oauthCode.code
                << "&client_id=" << authTokenClientId_ << "&client_secret=" << authTokenClientSecret_
                << "&device_id=" << urlEncode(device_->deviceId()) << "&device_name=" << urlEncode(deviceName_);

    const auto oauthUrl = passportUrl_ + "/token";
    YIO_LOG_INFO("Sending oauth token request to url '" << oauthUrl << "' for authCode " << maskToken(oauthCode.code));

    const auto tokenResponse = httpClient_.post("auth-code", oauthUrl, requestData.str());

    YIO_LOG_INFO("Response code: " << tokenResponse.responseCode);

    const auto body = tryParseJson(tokenResponse.body);
    if (!body.has_value()) {
        YIO_LOG_ERROR_EVENT("AccountStorage.BadJson.AccountByCode", "Can't parse oauth_token response");
        return ErrorCode::CODE_EXPIRED;
    }

    const auto& jsonBody = *body;
    // TODO: more strict rules
    if (!isSuccessHttpCode(tokenResponse.responseCode)) {
        const auto error = tryGetString(jsonBody, "error", "");
        const auto description = tryGetString(jsonBody, "error_description", "");
        YIO_LOG_WARN("Can't get token. Error: " << error << " Description: " << description);
        return ErrorCode::CODE_EXPIRED;
    }

    accountInfo.set_auth_token(getString(jsonBody, "access_token"));
    accountInfo.set_refresh_token(getString(jsonBody, "refresh_token"));

    YIO_LOG_INFO("authCode=" << maskToken(oauthCode.code)
                             << " got oauth_token=" << maskToken(accountInfo.auth_token())
                             << " and refresh_token=" << maskToken(accountInfo.refresh_token()));

    return ErrorCode::OK;
}

AccountStorage::ErrorCode AccountStorage::requestUid(const OAuthToken& oauthToken, proto::AccountInfo& accountInfo)
{
    const std::string requestData = "format=json";
    const HttpClient::Headers headers{
        {"Authorization", "OAuth " + std::string(oauthToken.token)}};

    const auto uidResponse = httpClient_.post("uid", loginUrl_ + "/info", requestData, headers);
    YIO_LOG_INFO("Response code: " << uidResponse.responseCode);

    const auto body = tryParseJson(uidResponse.body);
    if (!body.has_value()) {
        YIO_LOG_ERROR_EVENT("AccountStorage.BadJson.AccountByCode", "Can't parse uid response");
        return ErrorCode::CODE_EXPIRED;
    }

    const auto& jsonBody = *body;
    // TODO: more strict rules
    if (!isSuccessHttpCode(uidResponse.responseCode)) {
        const auto error = tryGetString(jsonBody, "error", "");
        const auto description = tryGetString(jsonBody, "error_description", "");
        YIO_LOG_WARN("Can't get token. Error: " << error << " Description: " << description);
        return ErrorCode::CODE_EXPIRED;
    }

    const auto id = getString(jsonBody, "id");
    accountInfo.set_id(std::stoull(id));

    YIO_LOG_INFO("authToken=" << maskToken(oauthToken.token) << " got uid=" << accountInfo.id());

    return ErrorCode::OK;
}

AccountStorage::OAuthResponseResult AccountStorage::tryRequestOAuthToken(const XToken& xtoken, proto::AccountInfo& accountInfo)
{
    std::stringstream requestData;
    requestData << "grant_type=x-token&access_token=" << xtoken.token
                << "&client_id=" << authTokenClientId_ << "&client_secret=" << authTokenClientSecret_
                << "&device_id=" << urlEncode(device_->deviceId()) << "&device_name=" << urlEncode(deviceName_);

    const auto oauthUrl = passportUrl_ + "/token";
    YIO_LOG_INFO("Sending oauth auth_token request to url '" << oauthUrl << "' for xtoken=" << maskToken(xtoken.token));

    HttpClient::HttpResponse authTokenResponse = httpClient_.post("x-token", oauthUrl, requestData.str());

    YIO_LOG_INFO("Received oauth auth_token response for xtoken=" << maskToken(xtoken.token)
                                                                  << ". Code: " << authTokenResponse.responseCode);

    Json::Value body;
    try {
        body = parseJson(authTokenResponse.body);
    } catch (const Json::Exception& e) {
        YIO_LOG_ERROR_EVENT("AccountStorage.BadJson.OAuthToken", "Can't parse oauth response");
        return OAuthResponseResult::ERROR;
    }
    // TODO: more strict rules
    if (!isSuccessHttpCode(authTokenResponse.responseCode)) {
        const auto error = tryGetString(body, "error", "");
        const auto description = tryGetString(body, "error_description", "");
        YIO_LOG_ERROR_EVENT("AccountStorage.FailedGetOAuthToken", "Can't get oauth token: Error: " << error << " Description: " << description);

        if (authTokenResponse.responseCode == 400 && error == "invalid_grant") {
            /* invalid grant can be retried */
            return OAuthResponseResult::INVALID_GRANT;
        }
        return OAuthResponseResult::ERROR;
    }

    accountInfo.set_auth_token(body["access_token"].asString());
    accountInfo.set_id(body["uid"].asUInt64());

    YIO_LOG_INFO("xtoken=" << maskToken(xtoken.token)
                           << " got oauth_token=" << maskToken(accountInfo.auth_token())
                           << " and uid=" << accountInfo.id());

    return OAuthResponseResult::OK;
}

AccountStorage::ErrorCode AccountStorage::refreshAccountXToken(proto::AccountInfo account)
{
    try {
        const auto result = refreshXToken(RefreshToken{.token = account.refresh_token()}, account);
        if (result != OAuthResponseResult::OK) {
            return ErrorCode::WRONG_TOKEN;
        }

        const auto updateResult = updateAccount(account);
        notifyAccountUpdated(updateResult, account);

        return ErrorCode::OK;
    } catch (const std::exception& e) {
        YIO_LOG_ERROR_EVENT("AccountStorage.FailedUpdateAccountByOAuth", "Error: " << e.what());
        return ErrorCode::NO_INTERNET;
    }
}

AccountStorage::OAuthResponseResult AccountStorage::refreshXToken(const RefreshToken& refreshToken, proto::AccountInfo& accountInfo)
{
    for (auto i = 0; i != REQUEST_TOKEN_RETRIES; ++i) {
        const auto result = tryRefreshXToken(refreshToken, accountInfo);
        /* INVALID_GRANT should be retried */
        if (result != OAuthResponseResult::INVALID_GRANT) {
            return result;
        }
        /* Not very smart backoff */
        std::this_thread::sleep_for(REQUEST_TOKEN_DELAY);
    }
    return OAuthResponseResult::ERROR;
}

AccountStorage::OAuthResponseResult AccountStorage::tryRefreshXToken(const RefreshToken& refreshToken, proto::AccountInfo& accountInfo)
{
    std::stringstream requestData;
    requestData << "grant_type=refresh_token&refresh_token=" << refreshToken.token
                << "&client_id=" << xTokenClientId_ << "&client_secret=" << xTokenClientSecret_;

    const auto oauthUrl = passportUrl_ + "/token";
    YIO_LOG_INFO("Sending xtoken request to url '" << oauthUrl << "' for refreshToken=" << maskToken(refreshToken.token));

    const auto response = httpClient_.post("refreshXToken", oauthUrl, requestData.str());

    YIO_LOG_INFO("Received xtoken response for refreshToken=" << maskToken(refreshToken.token)
                                                              << ". Code: " << response.responseCode);

    const auto body = tryParseJson(response.body);
    if (!body.has_value()) {
        YIO_LOG_ERROR_EVENT("AccountStorage.BadJson.RefreshXToken", "Can't parse token response");
        return OAuthResponseResult::ERROR;
    }

    const auto& jsonBody = *body;

    // TODO: more strict rules
    if (!isSuccessHttpCode(response.responseCode)) {
        const auto error = tryGetString(jsonBody, "error", "");
        const auto description = tryGetString(jsonBody, "error_description", "");
        YIO_LOG_ERROR_EVENT("AccountStorage.RefreshXToken", "Can't get xtoken: Error: " << error << " Description: " << description);

        if (response.responseCode == 400 && error == "invalid_grant") {
            /* invalid grant can be retried */
            return OAuthResponseResult::INVALID_GRANT;
        }
        return OAuthResponseResult::ERROR;
    }

    auto newXToken = jsonBody["access_token"].asString();
    auto newRefreshToken = jsonBody["refresh_token"].asString();
    YIO_LOG_INFO("refreshToken=" << maskToken(refreshToken.token)
                                 << " xtoken changed from " << maskToken(accountInfo.xtoken()) << " to " << maskToken(newXToken)
                                 << " refreshToken changed from " << maskToken(accountInfo.refresh_token()) << " to " << maskToken(newRefreshToken)
                                 << " exprires in " << jsonBody["expires_in"].asInt64() << "s");

    accountInfo.set_xtoken(std::move(newXToken));
    accountInfo.set_refresh_token(std::move(newRefreshToken));

    return OAuthResponseResult::OK;
}

AccountStorage::OAuthResponseResult AccountStorage::refreshOAuthToken(const RefreshToken& refreshToken, proto::AccountInfo& accountInfo)
{
    for (auto i = 0; i != REQUEST_TOKEN_RETRIES; ++i) {
        const auto result = tryRefreshOAuthToken(refreshToken, accountInfo);
        /* INVALID_GRANT should be retried */
        if (result != OAuthResponseResult::INVALID_GRANT) {
            return result;
        }
        /* Not very smart backoff */
        std::this_thread::sleep_for(REQUEST_TOKEN_DELAY);
    }
    return OAuthResponseResult::ERROR;
}

AccountStorage::OAuthResponseResult AccountStorage::tryRefreshOAuthToken(const RefreshToken& refreshToken, proto::AccountInfo& accountInfo)
{
    std::stringstream requestData;
    requestData << "grant_type=refresh_token&refresh_token=" << refreshToken.token
                << "&client_id=" << authTokenClientId_ << "&client_secret=" << authTokenClientSecret_;

    const auto oauthUrl = passportUrl_ + "/token";
    YIO_LOG_INFO("Sending oauthToken request to url '" << oauthUrl << "' for refreshToken=" << maskToken(refreshToken.token));

    const auto response = httpClient_.post("refreshOAuthToken", oauthUrl, requestData.str());

    YIO_LOG_INFO("Received oauthToken response for refreshToken=" << maskToken(refreshToken.token)
                                                                  << ". Code: " << response.responseCode);

    const auto body = tryParseJson(response.body);
    if (!body.has_value()) {
        YIO_LOG_ERROR_EVENT("AccountStorage.BadJson.RefreshOAuthToken", "Can't parse token response");
        return OAuthResponseResult::ERROR;
    }

    const auto& jsonBody = *body;

    // TODO: more strict rules
    if (!isSuccessHttpCode(response.responseCode)) {
        const auto error = tryGetString(jsonBody, "error", "");
        const auto description = tryGetString(jsonBody, "error_description", "");
        YIO_LOG_ERROR_EVENT("AccountStorage.RefreshOAuthToken", "Can't get oauth token: Error: " << error << " Description: " << description);

        if (response.responseCode == 400 && error == "invalid_grant") {
            /* invalid grant can be retried */
            return OAuthResponseResult::INVALID_GRANT;
        }
        return OAuthResponseResult::ERROR;
    }

    auto newOAuthToken = jsonBody["access_token"].asString();
    auto newRefreshToken = jsonBody["refresh_token"].asString();
    YIO_LOG_INFO("refreshToken=" << maskToken(refreshToken.token)
                                 << " oauthToken changed from " << maskToken(accountInfo.auth_token()) << " to " << maskToken(newOAuthToken)
                                 << " refreshToken changed from " << maskToken(accountInfo.refresh_token()) << " to " << maskToken(newRefreshToken)
                                 << " exprires in " << jsonBody["expires_in"].asInt64() << "s");

    accountInfo.set_auth_token(std::move(newOAuthToken));
    accountInfo.set_refresh_token(std::move(newRefreshToken));

    return OAuthResponseResult::OK;
}

AccountStorage::AccountId AccountStorage::addAccount(proto::AccountInfo& account, TimePoint additionTime)
{
    const auto updateResult = updateAccount(account, additionTime);
    notifyAccountUpdated(updateResult, account);

    return account.id();
}

AccountStorage::ErrorCode AccountStorage::deleteAccount(AccountId id)
{
    std::lock_guard guard(mutex_);

    if (currentAccount_.has_value() && id == currentAccount_->id()) {
        YIO_LOG_ERROR_EVENT("AccountStorage.FailedDeleteAccount", "Cannot delete current account. Id: " << id);
        return ErrorCode::WRONG_USER;
    }

    const auto accountIt = accounts_.find(id);
    if (accountIt == accounts_.end()) {
        YIO_LOG_ERROR_EVENT("AccountStorage.FailedDeleteAccount", "Unknown id: " << id);
        return ErrorCode::WRONG_USER;
    }

    const auto accountInfo = accountIt->second;

    accounts_.erase(accountIt);
    saveAccounts();

    if (accountInfo.has_auth_token()) {
        scheduleRevokeAuthToken(accountInfo.auth_token(), std::chrono::milliseconds::zero());
    }
    if (accountInfo.has_xtoken()) {
        scheduleRevokeXToken(accountInfo.xtoken(), std::chrono::milliseconds::zero());
    }

    if (onAccountDeleted) {
        onAccountDeleted(accountInfo);
    }

    return ErrorCode::OK;
}

void AccountStorage::scheduleRevokeAuthToken(std::string authToken, std::chrono::milliseconds timeout)
{
    YIO_LOG_INFO("Schedule revoke auth token " << maskToken(authToken) << " in " << timeout.count() / 1000 << "s");

    callbackQueue_->addDelayed([this, token = std::move(authToken)]() mutable {
        const auto result = revokeToken("revoke auth token", token, authTokenClientId_, authTokenClientSecret_);

        if (result != ErrorCode::OK) {
            const auto timeout = revokeAccountBackoffer_.getDelayBetweenCalls();
            scheduleRevokeAuthToken(std::move(token), timeout);
        }
    }, timeout);
}

void AccountStorage::scheduleRevokeXToken(std::string xToken, std::chrono::milliseconds timeout)
{
    YIO_LOG_INFO("Schedule revoke xtoken " << maskToken(xToken) << " in " << timeout.count() / 1000 << "s");

    callbackQueue_->addDelayed([this, token = std::move(xToken)]() mutable {
        const auto result = revokeToken("revoke xtoken", token, xTokenClientId_, xTokenClientSecret_);

        if (result != ErrorCode::OK) {
            const auto timeout = revokeAccountBackoffer_.getDelayBetweenCalls();
            scheduleRevokeXToken(std::move(token), timeout);
        }
    }, timeout);
}

AccountStorage::ErrorCode AccountStorage::revokeToken(std::string_view tag, std::string_view token,
                                                      std::string_view clientId, std::string_view clientSecret)
{
    try {
        std::stringstream requestData;
        requestData << "client_id=" << clientId << "&client_secret=" << clientSecret << "&access_token=" << token;

        const auto revokeTokenUrl = passportUrl_ + "/revoke_token";
        YIO_LOG_INFO("Sending revoke_token request to url '" << revokeTokenUrl << "' for token=" << maskToken(token));

        HttpClient::HttpResponse revokeTokenResponse = httpClient_.post(tag, revokeTokenUrl, requestData.str());

        YIO_LOG_INFO("Received revoke_token response for token=" << maskToken(token)
                                                                 << ". Code: " << revokeTokenResponse.responseCode);

        Json::Value body;
        try {
            body = parseJson(revokeTokenResponse.body);
        } catch (const Json::Exception& e) {
            YIO_LOG_ERROR_EVENT("AccountStorage.BadJson.RevokeToken", "Can't parse revoke token response");
            return ErrorCode::WRONG_TOKEN;
        }
        // TODO: more strict rules
        if (!isSuccessHttpCode(revokeTokenResponse.responseCode)) {
            const auto error = tryGetString(body, "error", "");
            const auto description = tryGetString(body, "error_description", "");
            YIO_LOG_ERROR_EVENT("AccountStorage.FailedRevokeToken", "Can't revoke " << tag << " token: Error: " << error << " Description: " << description);

            return ErrorCode::WRONG_TOKEN;
        }

        return ErrorCode::OK;
    } catch (const std::exception& e) {
        YIO_LOG_ERROR_EVENT("AccountStorage.FailedRevokeToken", "Error: " << e.what());
        return ErrorCode::NO_INTERNET;
    }
}

AccountStorage::ErrorCode AccountStorage::refreshAccountOAuthToken(proto::AccountInfo account)
{
    try {
        const auto result = refreshOAuthToken(RefreshToken{.token = account.refresh_token()}, account);
        if (result != OAuthResponseResult::OK) {
            return ErrorCode::WRONG_TOKEN;
        }

        const auto updateResult = updateAccount(account);
        notifyAccountUpdated(updateResult, account);

        return ErrorCode::OK;
    } catch (const std::exception& e) {
        YIO_LOG_ERROR_EVENT("AccountStorage.FailedUpdateAccountByOAuth", "Error: " << e.what());
        return ErrorCode::NO_INTERNET;
    }
}

AccountStorage::ErrorCode AccountStorage::updateOAuthToken(const std::string& authToken)
{
    auto account = findAccountByOAuth(authToken);
    if (!account.has_value()) {
        return ErrorCode::WRONG_TOKEN;
    }

    const AccountId accountId = account->id();

    if (account->type() == proto::AccountType::GUEST) {
        YIO_LOG_INFO("OAuthToken update is not supported for guest accounts. uid: " << accountId);
        return ErrorCode::WRONG_TOKEN;
    }

    if (!account->has_xtoken()) {
        YIO_LOG_INFO("Can't update oauthToken without xtoken. uid: " << accountId);
        return ErrorCode::WRONG_TOKEN;
    }

    {
        std::scoped_lock lock{scheduledOAuthTokenUpdateRequestsMutex_};
        if (scheduledOAuthTokenUpdateRequests_.find(accountId) != scheduledOAuthTokenUpdateRequests_.end()) {
            return ErrorCode::DELAYED;
        }
    }

    const auto result = updateAccountOAuthToken(std::move(*account));
    if (result == ErrorCode::WRONG_TOKEN) {
        scheduleOAuthTokenUpdate(accountId);
    }

    return result;
}

AccountStorage::ErrorCode AccountStorage::updateAccountOAuthToken(proto::AccountInfo account)
{
    const auto accountId = account.id();

    try {
        const auto errorCode = requestOAuthToken(XToken{.token = account.xtoken()}, account);
        if (errorCode != ErrorCode::OK) {
            return errorCode;
        }

        const auto updateResult = updateAccount(account);
        notifyAccountUpdated(updateResult, account);

        std::scoped_lock lock{scheduledOAuthTokenUpdateRequestsMutex_};
        scheduledOAuthTokenUpdateRequests_.erase(accountId);

        return ErrorCode::OK;
    } catch (const std::exception& e) {
        scheduleOAuthTokenUpdate(accountId);
        return ErrorCode::NO_INTERNET;
    }
}

std::optional<proto::AccountInfo> AccountStorage::findAccount(AccountId id) const {
    std::scoped_lock lock{mutex_};
    if (auto foundAccountIt = accounts_.find(id); foundAccountIt != accounts_.end()) {
        return foundAccountIt->second;
    }

    return std::nullopt;
}

std::optional<proto::AccountInfo> AccountStorage::findAccountByOAuth(const std::string& authToken) const {
    std::scoped_lock lock{mutex_};
    for (const auto& account : accounts_) {
        if (account.second.auth_token() == authToken) {
            return account.second;
        }
    }

    return std::nullopt;
}

void AccountStorage::changeAccount(AccountId accountId, bool deleteOldAccount)
{
    AccountStorage::UpdateAccountResult changeResult{UpdateAccountResult::NOT_CHANGED};
    std::vector<proto::AccountInfo> deletedAccounts;
    proto::AccountInfo account;

    {
        std::scoped_lock lock{mutex_};

        account = getAccount(accountId);
        if (account.type() != proto::AccountType::OWNER) {
            throw std::runtime_error("Cannot change current account to non-admin account");
        }

        if (!currentAccount_.has_value() || currentAccount_->id() != account.id()) {
            changeResult = UpdateAccountResult::ACCOUNT_CHANGED;
        } else if (currentAccount_->xtoken() != account.xtoken() || currentAccount_->auth_token() != account.auth_token()) {
            changeResult = UpdateAccountResult::TOKEN_CHANGED;
        } else if (currentAccount_->last_token_refresh_timestamp() != account.last_token_refresh_timestamp()) {
            changeResult = UpdateAccountResult::ACCOUNT_REFRESHED;
        }

        currentAccount_ = account;

        if (deleteOldAccount) {
            deletedAccounts = deleteInactiveAccounts();
        }

        saveAccounts();
    }

    YIO_LOG_INFO("New account:"
                 << " uid=" << account.id()
                 << " xtoken=" << maskToken(account.xtoken())
                 << " authToken=" << maskToken(account.auth_token())
                 << " refreshToken=" << maskToken(account.refresh_token()));

    notifyAccountUpdated(changeResult, account);

    if (onAccountDeleted) {
        for (const auto& account : deletedAccounts) {
            onAccountDeleted(account);
        }
    }
}

std::optional<proto::AccountInfo> AccountStorage::getCurrentAccount() const {
    std::lock_guard guard(mutex_);
    return currentAccount_;
}

std::vector<proto::AccountInfo> AccountStorage::getAllAccounts() const {
    std::scoped_lock lock(mutex_);

    std::vector<proto::AccountInfo> result;
    result.reserve(accounts_.size());
    for (const auto& [_, account] : accounts_) {
        result.push_back(account);
    }

    return result;
}

// should be called under mutex
proto::AccountInfo AccountStorage::getAccount(AccountId accountId) const {
    const auto account = tryGetAccount(accountId);
    if (!account) {
        throw std::runtime_error("Cannot get account with id " + std::to_string(accountId));
    }

    return *account;
}

// should be called under mutex
std::optional<proto::AccountInfo> AccountStorage::tryGetAccount(AccountId accountId) const {
    const auto accountIt = accounts_.find(accountId);
    if (accounts_.end() == accountIt) {
        return std::nullopt;
    }
    return accountIt->second;
}

AccountStorage::UpdateAccountResult AccountStorage::updateAccount(proto::AccountInfo& account, TimePoint additionTime) {
    std::scoped_lock lock(mutex_);

    UpdateAccountResult result;

    account.set_last_token_refresh_timestamp(timePointToProto(additionTime));

    if (const auto it = accounts_.find(account.id()); it == accounts_.end()) {
        result = UpdateAccountResult::ACCOUNT_ADDED;
    } else {
        const auto& foundAccount = it->second;
        if (foundAccount.xtoken() != account.xtoken() || foundAccount.auth_token() != account.auth_token()) {
            result = UpdateAccountResult::TOKEN_CHANGED;
        } else {
            result = UpdateAccountResult::ACCOUNT_REFRESHED;
        }
    }

    YIO_LOG_INFO("Update account: " << shortUtf8DebugString(account))
    accounts_[account.id()] = account;

    if (currentAccount_.has_value() && currentAccount_->id() == account.id()) {
        currentAccount_ = account;
    }

    saveAccounts();

    scheduleProlongAccount(account);

    return result;
}

void AccountStorage::notifyAccountUpdated(UpdateAccountResult updateResult, const proto::AccountInfo& account) const {
    switch (updateResult) {
        case UpdateAccountResult::NOT_CHANGED:
            return;
        case UpdateAccountResult::ACCOUNT_ADDED:
            if (onAccountAdded) {
                onAccountAdded(account);
            }
            break;
        case UpdateAccountResult::ACCOUNT_CHANGED:
            if (onAccountChange) {
                onAccountChange(account);
            }
            break;
        case UpdateAccountResult::TOKEN_CHANGED:
            if (onTokenChange) {
                onTokenChange(account);
            }
            break;
        case UpdateAccountResult::ACCOUNT_REFRESHED:
            if (onAccountRefresh) {
                onAccountRefresh(account);
            }
            break;
    }
}

// should be called under mutex
std::vector<proto::AccountInfo> AccountStorage::deleteInactiveAccounts()
{
    if (currentAccount_.has_value()) {
        accounts_.erase(currentAccount_->id());
    }

    std::vector<proto::AccountInfo> deletedAccounts;
    for (const auto& [id, account] : accounts_) {
        YIO_LOG_INFO("Delete account: " << shortUtf8DebugString(account));
        deletedAccounts.push_back(account);
    }

    accounts_.clear();

    if (currentAccount_.has_value()) {
        accounts_.emplace(currentAccount_->id(), *currentAccount_);
    }

    return deletedAccounts;
}

void AccountStorage::loadAccounts()
{
    if (!fileExists(fileName_)) {
        // Storage is not created yet
        return;
    }

    TString serialized = getFileContent(fileName_);
    proto::AccountStorage storage;
    if (!storage.ParseFromString(serialized)) {
        YIO_LOG_ERROR_EVENT("AccountStorage.BadProto.AccountStorage", "Cannot parse account storage data from " << fileName_);
        return; // Leave accounts_ empty
    }

    for (const auto& accountInfo : storage.accounts()) {
        YIO_LOG_INFO("Load account: " << shortUtf8DebugString(accountInfo));
        accounts_[accountInfo.id()] = accountInfo;
    }

    if (accounts_.empty()) {
        return;
    }

    const AccountId currentAccountId = storage.has_current_account_id() ? storage.current_account_id() : accounts_.begin()->first;
    YIO_LOG_INFO("Current account: " << currentAccountId);

    // Explicitly set Admin account type to current account to update an old account from previous versions.
    accounts_[currentAccountId].set_type(proto::AccountType::OWNER);
    currentAccount_ = accounts_[currentAccountId];

    // Clear accounts with invalid type (no type or UNDEFINED)
    std::erase_if(accounts_, [](const auto& item) {
        const auto& [key, value] = item;
        return !value.has_type() || value.type() == proto::AccountType::UNDEFINED;
    });
}

// should be called under mutex
void AccountStorage::saveAccounts()
{
    PersistentFile file(fileName_, PersistentFile::Mode::TRUNCATE);

    proto::AccountStorage storage_;

    for (const auto& [id, account] : accounts_) {
        YIO_LOG_INFO("Save account: " << shortUtf8DebugString(account))
        *storage_.add_accounts() = account;
    }

    if (currentAccount_.has_value()) {
        storage_.set_current_account_id(currentAccount_->id());
    }

    if (!file.write(storage_.SerializeAsString())) {
        throw std::runtime_error("Cannot write to file " + fileName_);
    }
}

void AccountStorage::scheduleProlongAccounts()
{
    std::scoped_lock lock{mutex_};
    for (auto& [id, account] : accounts_) {
        scheduleProlongAccount(account);
    }
}

void AccountStorage::scheduleProlongAccount(const proto::AccountInfo& account)
{
    const auto lastTokenRefreshTime = timePointFromProto(account.last_token_refresh_timestamp());
    const auto timeout = getProlongTokenTimeout(lastTokenRefreshTime);

    const auto& tokenType = account.has_xtoken() ? "xtoken" : "oauthToken";

    YIO_LOG_INFO("Schedule " << tokenType << " refresh callback for account uid: "
                             << account.id() << " in " << timeout.count() << "s");
    auto callback = getProlongTokenCallback(account.id(), lastTokenRefreshTime);
    callbackQueue_->addDelayed(std::move(callback), timeout);
}

std::function<void()> AccountStorage::getProlongTokenCallback(AccountId id, TimePoint lastTokenRefreshTime)
{
    return [this, id, lastTokenRefreshTime] {
        auto account = std::invoke([this, id] {
            std::scoped_lock lock{mutex_};
            return tryGetAccount(id);
        });

        if (!account) {
            return;
        }

        const auto accountLastRefreshTime = timePointFromProto(account->last_token_refresh_timestamp());
        if (lastTokenRefreshTime != accountLastRefreshTime) {
            // last_token_refresh_timestamp was changed between this callback creation and now.
            // So we need to do nothing here.
            return;
        }

        // We might not have xtoken for guest users
        if (account->has_xtoken()) {
            prolongXToken(std::move(*account), lastTokenRefreshTime);
            return;
        }

        prolongOAuthToken(std::move(*account), lastTokenRefreshTime);
    };
}

void AccountStorage::prolongXToken(proto::AccountInfo account, TimePoint lastTokenRefreshTime)
{
    const auto id = account.id();
    const auto errorCode = refreshAccountXToken(std::move(account));
    switch (errorCode) {
        case ErrorCode::OK: {
            YIO_LOG_INFO("xtoken successfully updated.");
            device_->telemetry()->reportEvent("xTokenProlongated");
            return;
        }
        case ErrorCode::NO_INTERNET: {
            const auto timeout = std::chrono::seconds{300};
            YIO_LOG_ERROR_EVENT("AccountStorage.FailedProlongXToken", "No internet during x-token prolongation. Reschedule callback in: " << timeout.count() << "s");
            callbackQueue_->addDelayed(getProlongTokenCallback(id, lastTokenRefreshTime), timeout);
            return;
        }
        default: {
            YIO_LOG_ERROR_EVENT("AccountStorage.FailedProlongXToken", "Can't prolong x-token lifetime. Error code: " << static_cast<int>(errorCode));
            return;
        }
    }
}

void AccountStorage::prolongOAuthToken(proto::AccountInfo account, TimePoint lastTokenRefreshTime)
{
    const auto id = account.id();
    const auto errorCode = refreshAccountOAuthToken(std::move(account));
    switch (errorCode) {
        case ErrorCode::OK: {
            YIO_LOG_INFO("oauthToken successfully updated.");
            device_->telemetry()->reportEvent("oauthTokenProlongated");
            return;
        }
        case ErrorCode::NO_INTERNET: {
            const auto timeout = std::chrono::seconds{300};
            YIO_LOG_ERROR_EVENT("AccountStorage.FailedProlongOAuthToken", "No internet during oauthToken prolongation. Reschedule callback in: " << timeout.count() << "s");
            callbackQueue_->addDelayed(getProlongTokenCallback(id, lastTokenRefreshTime), timeout);
            return;
        }
        default: {
            YIO_LOG_ERROR_EVENT("AccountStorage.FailedProlongOAuthToken", "Can't prolong oauthToken lifetime. Error code: " << static_cast<int>(errorCode));
            return;
        }
    }
}

std::chrono::seconds AccountStorage::getProlongTokenTimeout(TimePoint lastTokenRefreshTime)
{
    using namespace std::chrono;

    std::uniform_int_distribution<int64_t> d(0, 86400);
    const auto delay = std::chrono::seconds{d(timeoutGenerator_)};
    const auto prolongTokenTime = lastTokenRefreshTime + PROLONG_TOKEN_INTERVAL + delay;
    const auto now = system_clock::now();

    if (prolongTokenTime >= now) {
        return duration_cast<seconds>(prolongTokenTime - now);
    }

    return delay;
}

void AccountStorage::scheduleOAuthTokenUpdate(AccountId accountId)
{
    std::scoped_lock lock{scheduledOAuthTokenUpdateRequestsMutex_};

    const int failedUpdates = scheduledOAuthTokenUpdateRequests_[accountId]++;
    updateAccountBackoffer_.resetDelayBetweenCallsToDefault();
    for (int i{0}; i != failedUpdates; ++i) {
        updateAccountBackoffer_.increaseDelayBetweenCalls();
    }

    const auto timeout = updateAccountBackoffer_.getDelayBetweenCalls();
    YIO_LOG_INFO("Schedule oauthToken update for account uid: " << accountId << " in " << timeout.count() / 1000 << "s");

    callbackQueue_->addDelayed([this, accountId] {
        auto account = findAccount(accountId);
        if (!account.has_value()) {
            std::scoped_lock lock{scheduledOAuthTokenUpdateRequestsMutex_};
            scheduledOAuthTokenUpdateRequests_.erase(accountId);
            return;
        }

        if (const auto result = updateAccountOAuthToken(std::move(*account)); result == ErrorCode::WRONG_TOKEN) {
            scheduleOAuthTokenUpdate(accountId);
        }
    }, timeout);
}
