#pragma once

#include <yandex_io/libs/delay_timings_policy/delay_timings_policy.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/http_client/http_client.h>
#include <yandex_io/libs/threading/i_callback_queue.h>
#include <yandex_io/protos/account_storage.pb.h>

#include <chrono>
#include <functional>
#include <map>
#include <mutex>
#include <optional>
#include <random>
#include <string>
#include <string_view>
#include <vector>

namespace quasar {

    /*
     * Хранилище всех аккаунтов, когда-либо добавленных на колонку.
     * Добавленные аккаунты сохраняются в файл на диске, путь к которому задается полем конфига authd/accountStorageFile.
     * В каждый момент времени активен только один аккаунт. Идентификатор активного аккаунта хранится в том же файле.
     */
    class AccountStorage {
    public:
        AccountStorage(std::shared_ptr<YandexIO::IDevice> device, std::shared_ptr<ICallbackQueue> callbackQueue);

        enum class ErrorCode {
            OK = 0,
            CODE_EXPIRED,
            NO_INTERNET,
            WRONG_TOKEN,
            DELAYED,
            WRONG_USER,
            UNDEFINED_TYPE
        };

        using Duration = std::chrono::seconds;
        using TimePoint = std::chrono::time_point<std::chrono::system_clock, Duration>;

        using AccountId = std::uint64_t;

    public:
        using AccountAddedHandler = std::function<void(const proto::AccountInfo&)>;
        AccountAddedHandler onAccountAdded;

        using AccountChangeHandler = std::function<void(const proto::AccountInfo&)>;
        AccountChangeHandler onAccountChange;

        using AccountDeletedHandler = std::function<void(const proto::AccountInfo&)>;
        AccountDeletedHandler onAccountDeleted;

        using TokenChangeHandler = std::function<void(const proto::AccountInfo&)>;
        TokenChangeHandler onTokenChange;

        using AccountRefreshHandler = std::function<void(const proto::AccountInfo&)>;
        AccountRefreshHandler onAccountRefresh;

    public:
        // Request an account by authCode. On success calls onAccountAdded if it's an unknown account, calls onTokenChange if it's a known account
        // and authToken or xToken has changed, calls onAccountRefresh if it's a known account and tokens haven't changed.
        ErrorCode requestAccount(const std::string& authCode, proto::AccountType accountType, bool withXToken, proto::AccountInfo& accountInfo);
        // Update account's authToken. On succcess calls either onTokenChange if token was changed or onAccountRefresh if token didn't change
        ErrorCode updateOAuthToken(const std::string& authToken);
        // Delete account. On success calls onAccountDeleted
        ErrorCode deleteAccount(AccountId id);
        // Changes current account. On success calls onAccountChange. Also calls onAccountDeleted for every deleted account
        void changeAccount(AccountId accountId, bool deleteOldAccount = true);
        // Returns current OWNER accounts, if it exists
        std::optional<proto::AccountInfo> getCurrentAccount() const;
        // Returns all accounts
        std::vector<proto::AccountInfo> getAllAccounts() const;
        // Set periodic callback for every account to
        void scheduleProlongAccounts();

        // Used for tests only
        AccountId addAccount(proto::AccountInfo& accountInfo, TimePoint additionTime = std::chrono::time_point_cast<Duration>(std::chrono::system_clock::now()));

    private:
        void loadAccounts();
        void saveAccounts();

        // Returns deleted accounts
        std::vector<proto::AccountInfo> deleteInactiveAccounts();

        void scheduleRevokeAuthToken(std::string authToken, std::chrono::milliseconds timeout);
        void scheduleRevokeXToken(std::string xToken, std::chrono::milliseconds timeout);
        ErrorCode revokeToken(std::string_view tag, std::string_view token, std::string_view clientId, std::string_view clientSecret);

        proto::AccountInfo getAccount(AccountId accountId) const;
        std::optional<proto::AccountInfo> tryGetAccount(AccountId accountId) const;

    private:
        enum class UpdateAccountResult {
            NOT_CHANGED = 0,
            ACCOUNT_ADDED,
            ACCOUNT_CHANGED,
            TOKEN_CHANGED,
            ACCOUNT_REFRESHED
        };

        UpdateAccountResult updateAccount(proto::AccountInfo& account, TimePoint additionTime = std::chrono::time_point_cast<Duration>(std::chrono::system_clock::now()));
        void notifyAccountUpdated(UpdateAccountResult updateResult, const proto::AccountInfo& account) const;

        std::optional<proto::AccountInfo> findAccount(AccountId id) const;
        std::optional<proto::AccountInfo> findAccountByOAuth(const std::string& authToken) const;

        void scheduleProlongAccount(const proto::AccountInfo& account);
        std::function<void()> getProlongTokenCallback(AccountId id, TimePoint lastTokenRefreshTime);
        std::chrono::seconds getProlongTokenTimeout(TimePoint lastTokenRefreshTime);

        void prolongXToken(proto::AccountInfo account, TimePoint lastTokenRefreshTime);
        void prolongOAuthToken(proto::AccountInfo account, TimePoint lastTokenRefreshTime);

        ErrorCode refreshAccountXToken(proto::AccountInfo account);
        ErrorCode refreshAccountOAuthToken(proto::AccountInfo account);

        void scheduleOAuthTokenUpdate(AccountId accountId);
        ErrorCode updateAccountOAuthToken(proto::AccountInfo account);

        enum class OAuthResponseResult {
            OK,            /* Successfully got oauth token */
            INVALID_GRANT, /* auth return "invalid_grant" error */
            ERROR          /* any other error */
        };

        struct UidResponse {
            enum class Status {
                OK,
                ERROR
            };
            Status status{Status::ERROR};
            AccountId uid{0};
        };

        struct XToken {
            std::string_view token;
        };

        struct RefreshToken {
            std::string_view token;
        };

        struct OAuthToken {
            std::string_view token;
        };

        struct XCode {
            std::string_view code;
        };

        struct OAuthCode {
            std::string_view code;
        };

    private:
        /**
         * @brief Request OAuth token and Uid from passport.
         * @note: Can retry "invalid_grant" error only
         */
        OAuthResponseResult tryRequestOAuthToken(const XToken& xtoken, proto::AccountInfo& accountInfo);

        ErrorCode requestXToken(const XCode& xcode, proto::AccountInfo& accountInfo);
        ErrorCode requestOAuthToken(const XToken& xtoken, proto::AccountInfo& accountInfo);
        ErrorCode requestOAuthToken(const OAuthCode& oauthCode, proto::AccountInfo& accountInfo);
        ErrorCode requestUid(const OAuthToken& oauthToken, proto::AccountInfo& accountInfo);

        ErrorCode requestAccount(const XCode& xcode, proto::AccountInfo& accountInfo);
        ErrorCode requestAccount(const OAuthCode& oauthCode, proto::AccountInfo& accountInfo);

        OAuthResponseResult refreshXToken(const RefreshToken& refreshToken, proto::AccountInfo& accountInfo);
        OAuthResponseResult tryRefreshXToken(const RefreshToken& refreshToken, proto::AccountInfo& accountInfo);

        OAuthResponseResult refreshOAuthToken(const RefreshToken& refreshToken, proto::AccountInfo& accountInfo);
        OAuthResponseResult tryRefreshOAuthToken(const RefreshToken& refreshToken, proto::AccountInfo& accountInfo);

    private:
        std::shared_ptr<YandexIO::IDevice> device_;

        mutable std::mutex mutex_;

        std::map<AccountId, proto::AccountInfo> accounts_;
        std::optional<proto::AccountInfo> currentAccount_;

        std::string fileName_;

        HttpClient httpClient_;

        std::string passportUrl_;
        std::string loginUrl_;

        std::string xTokenClientId_;
        std::string xTokenClientSecret_;

        std::string authTokenClientId_;
        std::string authTokenClientSecret_;
        std::string deviceName_;

        std::default_random_engine timeoutGenerator_;
        std::shared_ptr<ICallbackQueue> callbackQueue_;

        std::map<AccountId, int> scheduledOAuthTokenUpdateRequests_;
        mutable std::mutex scheduledOAuthTokenUpdateRequestsMutex_;
        BackoffRetriesWithRandomPolicy updateAccountBackoffer_;
        BackoffRetriesWithRandomPolicy revokeAccountBackoffer_;

        AccountStorage(const AccountStorage&) = delete;
        AccountStorage& operator=(const AccountStorage&) = delete;
    };

} // namespace quasar
