#include "auth_endpoint.h"

#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/ipc/message.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/protobuf_utils/debug.h>

#include "yandex_io/protos/account_storage.pb.h"
#include <yandex_io/protos/quasar_proto.pb.h>

#include <json/value.h>

#include <chrono>

YIO_DEFINE_LOG_MODULE("auth");

using namespace quasar;

namespace {

    void accountInfoToAuthEvent(const proto::AccountInfo& accountInfo, proto::AuthEvent* authEvent) {
        if (accountInfo.has_xtoken()) {
            authEvent->set_xtoken(accountInfo.xtoken());
        }
        if (accountInfo.has_auth_token()) {
            authEvent->set_auth_token(accountInfo.auth_token());
        }
        if (accountInfo.has_id()) {
            authEvent->set_passport_uid(std::to_string(accountInfo.id()));
        }
        if (accountInfo.has_type()) {
            authEvent->set_type(accountInfo.type());
        }
        if (accountInfo.has_last_token_refresh_timestamp()) {
            authEvent->set_tag(accountInfo.last_token_refresh_timestamp());
        }
    }

} // namespace

AuthEndpoint::AuthEndpoint(std::shared_ptr<YandexIO::IDevice> device, std::shared_ptr<ipc::IIpcFactory> ipcFactory)
    : device_(std::move(device))
    , callbackQueue_(std::make_shared<NamedCallbackQueue>("AuthEndpoint"))
    , accountStorage_(device_, callbackQueue_)
    , server_(ipcFactory->createIpcServer(SERVICE_NAME))
    , ntpdConnector_(ipcFactory->createIpcConnector("ntpd"))
{
    server_->setMessageHandler(std::bind(&AuthEndpoint::processQuasarMessage, this, std::placeholders::_1, std::placeholders::_2));

    server_->setClientConnectedHandler([this](auto& connection) {
        auto startupMessage = ipc::buildMessage([this](auto& msg) {
            auto* ownerStartupInfo = msg.mutable_owner_startup_info();
            if (const auto optAccountInfo = accountStorage_.getCurrentAccount(); optAccountInfo.has_value()) {
                ownerStartupInfo->set_auth_token(optAccountInfo->auth_token());
                ownerStartupInfo->set_xtoken(optAccountInfo->xtoken());
                ownerStartupInfo->set_passport_uid(std::to_string(optAccountInfo->id()));
                ownerStartupInfo->set_tag(makeTag());
            }

            auto* allStartupInfo = msg.mutable_all_startup_info();
            for (const auto& accountInfo : accountStorage_.getAllAccounts()) {
                auto* authEvent = allStartupInfo->add_accounts();
                accountInfoToAuthEvent(accountInfo, authEvent);
                authEvent->set_tag(makeTag());
            }
        });

        connection.send(startupMessage);
    });

    server_->listenService();

    accountStorage_.onAccountAdded = [this](const proto::AccountInfo& accountInfo) {
        auto message = ipc::buildMessage([&accountInfo](auto& msg) {
            accountInfoToAuthEvent(accountInfo, msg.mutable_add_user_event());
        });
        server_->sendToAll(message);
    };

    accountStorage_.onAccountChange = [this](const proto::AccountInfo& accountInfo) {
        auto message = ipc::buildMessage([&accountInfo](auto& msg) {
            accountInfoToAuthEvent(accountInfo, msg.mutable_change_user_event());
        });
        server_->sendToAll(message);
    };

    accountStorage_.onAccountDeleted = [this](const proto::AccountInfo& accountInfo) {
        auto message = ipc::buildMessage([&accountInfo](auto& msg) {
            accountInfoToAuthEvent(accountInfo, msg.mutable_delete_user_event());
        });
        server_->sendToAll(message);
    };

    accountStorage_.onTokenChange = [=](const proto::AccountInfo& accountInfo) {
        auto message = ipc::buildMessage([&accountInfo](auto& msg) {
            accountInfoToAuthEvent(accountInfo, msg.mutable_change_user_event());
        });
        server_->sendToAll(message);
    };

    accountStorage_.onAccountRefresh = [=](const proto::AccountInfo& accountInfo) {
        auto message = ipc::buildMessage([&accountInfo](auto& msg) {
            accountInfoToAuthEvent(accountInfo, msg.mutable_refresh_user_event());
        });
        server_->sendToAll(message);
    };

    ntpdConnector_->setMessageHandler([this](const auto& message) {
        processNtpdMessage(*message);
    });
    ntpdConnector_->connectToService();
}

void AuthEndpoint::waitConnectionsAtLeast(size_t count) {
    server_->waitConnectionsAtLeast(count);
}

void AuthEndpoint::processNtpdMessage(const proto::QuasarMessage& message)
{
    if (message.has_ntp_sync_event() && message.ntp_sync_event().has_is_ntp_sync_successful()) {
        const bool lastNtpSynchronized = std::exchange(ntpSynchronized_, message.ntp_sync_event().is_ntp_sync_successful());
        if (ntpSynchronized_ && !lastNtpSynchronized) {
            YIO_LOG_INFO("Time synchronized. Schedule tokens prolongation.");
            accountStorage_.scheduleProlongAccounts();
        }
    }
}

void AuthEndpoint::processQuasarMessage(const ipc::SharedMessage& message, ipc::IServer::IClientConnection& connection)
{
    if (message->has_add_user_request())
    {
        tryProcessAddUserRequest(*message, connection);
    }
    if (message->has_delete_user_request())
    {
        tryProcessDeleteUserRequest(*message, connection);
    }
    if (message->has_change_user_request())
    {
        tryProcessChangeUserRequest(*message, connection);
    }
    if (message->has_auth_token_update_request())
    {
        tryProcessAuthTokenUpdateRequest(*message, connection);
    }
}

void AuthEndpoint::tryProcessAddUserRequest(const quasar::proto::QuasarMessage& message, ipc::IServer::IClientConnection& connection)
{
    try {
        processAddUserRequest(message, connection);
    } catch (const std::exception& e) {
        YIO_LOG_ERROR_EVENT("AuthEndpoint.FailedProcessAddUserRequest", "Failed to add user with auth code " << maskToken(message.add_user_request().auth_code()) << ". Error: " << e.what());
        throw;
    }
}

void AuthEndpoint::tryProcessDeleteUserRequest(const quasar::proto::QuasarMessage& message, ipc::IServer::IClientConnection& connection)
{
    try {
        processDeleteUserRequest(message, connection);
    } catch (const std::exception& e) {
        YIO_LOG_ERROR_EVENT("AuthEndpoint.FailedProcessDeleteUserRequest", "Failed to delete user with id " << message.delete_user_request().id() << ". Error: " << e.what());
        throw;
    }
}

void AuthEndpoint::processAddUserRequest(const proto::QuasarMessage& message, ipc::IServer::IClientConnection& connection)
{
    YIO_LOG_INFO("Add user request");

    const auto& authCode = message.add_user_request().auth_code();
    const auto& accountType = message.add_user_request().account_type();
    const bool withXToken = message.add_user_request().with_xtoken();

    proto::AccountInfo accountInfo;
    const auto result = accountStorage_.requestAccount(authCode, accountType, withXToken, accountInfo);

    auto response = ipc::buildMessage([&message, &result, &accountInfo](auto& msg) {
        msg.set_request_id(message.request_id());
        switch (result)
        {
            case AccountStorage::ErrorCode::OK:
                YIO_LOG_INFO("requestAccount success");
                msg.mutable_add_user_response()->set_status(proto::AddUserResponse::OK);
                msg.mutable_add_user_response()->set_auth_token(accountInfo.auth_token());
                msg.mutable_add_user_response()->set_xtoken(accountInfo.xtoken());
                msg.mutable_add_user_response()->set_id(accountInfo.id());
                msg.mutable_add_user_response()->set_tag(makeTag());

                break;
            case AccountStorage::ErrorCode::NO_INTERNET:
                YIO_LOG_ERROR_EVENT("AuthEndpoint.FailedAddAccountByCode.NoInternet", "addAccountByCode failed: NO_INTERNET");
                msg.mutable_add_user_response()->set_status(proto::AddUserResponse::NO_INTERNET);
                break;
            case AccountStorage::ErrorCode::CODE_EXPIRED:
                YIO_LOG_ERROR_EVENT("AuthEndpoint.FailedAddAccountByCode.CodeExpired", "addAccountByCode failed: CODE_EXPIRED");
                msg.mutable_add_user_response()->set_status(proto::AddUserResponse::CODE_EXPIRED);
                break;
            case AccountStorage::ErrorCode::WRONG_TOKEN:
                YIO_LOG_ERROR_EVENT("AuthEndpoint.FailedAddAccountByCode.WrongToken", "addAccountByCode failed: WRONG_TOKEN");
                msg.mutable_add_user_response()->set_status(proto::AddUserResponse::INVALID_TOKEN);
                break;
            default:
                throw std::runtime_error("Unknown error code of account storage: " + std::to_string(int(result)));
        }
    });
    connection.send(response);

    YIO_LOG_INFO("processAddUserRequest finish");
}

void AuthEndpoint::processDeleteUserRequest(const quasar::proto::QuasarMessage& message, ipc::IServer::IClientConnection& connection)
{
    YIO_LOG_INFO("Delete user request with uid=" << message.delete_user_request().id());

    auto result = accountStorage_.deleteAccount(message.delete_user_request().id());

    auto response = ipc::buildMessage([&message, &result](auto& msg) {
        msg.set_request_id(message.request_id());
        switch (result)
        {
            case AccountStorage::ErrorCode::OK:
                YIO_LOG_INFO("deleteAccount success");
                msg.mutable_delete_user_response()->set_status(proto::DeleteUserResponse::OK);
                break;
            case AccountStorage::ErrorCode::WRONG_USER:
                YIO_LOG_ERROR_EVENT("AuthEndpoint.FailedDeleteAccount.WrongUser", "deleteAccount failed: WRONG_USER");
                msg.mutable_delete_user_response()->set_status(proto::DeleteUserResponse::WRONG_USER);
                break;
            default:
                throw std::runtime_error("Unknown error code of account storage: " + std::to_string(int(result)));
        }
    });
    connection.send(response);

    YIO_LOG_INFO("processDeleteUserRequest finish");
}

void AuthEndpoint::tryProcessChangeUserRequest(const proto::QuasarMessage& message, ipc::IServer::IClientConnection& connection)
{
    try {
        processChangeUserRequest(message, connection);
    } catch (const std::exception& e) {
        YIO_LOG_ERROR_EVENT("AuthEndpoint.FailedProcessChangeUserRequest", "Failed to change user to " << message.change_user_request().id() << ". Error: " << e.what());
        throw;
    }
}

void AuthEndpoint::processChangeUserRequest(const proto::QuasarMessage& message, ipc::IServer::IClientConnection& connection)
{
    YIO_LOG_INFO("Change user request with uid=" << message.change_user_request().id());
    accountStorage_.changeAccount(message.change_user_request().id());
    YIO_LOG_INFO("User changed to uid=" << message.change_user_request().id() << " successfully");

    auto response = ipc::buildMessage([&message](auto& msg) {
        msg.set_request_id(message.request_id());
    });
    connection.send(response);
    YIO_LOG_INFO("processChangeUserRequest finish");
}

void AuthEndpoint::tryProcessAuthTokenUpdateRequest(const proto::QuasarMessage& message, ipc::IServer::IClientConnection& connection)
{
    try {
        processAuthTokenUpdateRequest(message, connection);
    } catch (const std::exception& e) {
        YIO_LOG_ERROR_EVENT("AuthEndpoint.FailedProcessAuthTokenUpdateRequest",
                            "Failed to process invalid OAuth token " << maskToken(message.auth_token_update_request().auth_token())
                                                                     << ". Error: " << e.what());
        throw;
    }
}

void AuthEndpoint::processAuthTokenUpdateRequest(const proto::QuasarMessage& message, ipc::IServer::IClientConnection& connection)
{
    const std::string token = message.auth_token_update_request().auth_token();
    const auto& source = message.auth_token_update_request().source();
    YIO_LOG_INFO("Auto token update request from " << source << ", token= " << maskToken(token));

    const auto result = accountStorage_.updateOAuthToken(token);
    switch (result)
    {
        case AccountStorage::ErrorCode::OK:
            YIO_LOG_INFO("updateAccountByOAuth success");
            device_->telemetry()->reportEvent("accountUpdated");
            break;
        case AccountStorage::ErrorCode::NO_INTERNET:
            YIO_LOG_ERROR_EVENT("AuthEndpoint.FailedUpdateAccountByOAuth.NoInternet", "updateAccountByOAuth failed: NO_INTERNET");
            break;
        case AccountStorage::ErrorCode::WRONG_TOKEN:
            YIO_LOG_ERROR_EVENT("AuthEndpoint.FailedUpdateAccountByOAuth.WrongToken", "updateAccountByOAuth failed: WRONG_TOKEN");
            break;
        case AccountStorage::ErrorCode::DELAYED:
            YIO_LOG_INFO("updateAccountByOAuth delayed");
            break;
        default:
            throw std::runtime_error("Unknown error code of account storage: " + std::to_string(int(result)));
    }

    auto response = ipc::buildMessage([&message](auto& msg) {
        msg.set_request_id(message.request_id());
    });
    connection.send(response);
    YIO_LOG_INFO("processAuthTokenUpdateRequest finish");
}

std::int64_t AuthEndpoint::makeTag() {
    return getNowTimestampMs();
}

int AuthEndpoint::port() const {
    return server_->port();
}

const std::string AuthEndpoint::SERVICE_NAME = "authd";
