#pragma once

#include "account_storage.h"

#include <yandex_io/libs/base/named_callback_queue.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/ipc/i_ipc_factory.h>

#include <memory>
#include <mutex>
#include <string>

namespace quasar {

    class AuthEndpoint {
    public:
        AuthEndpoint(std::shared_ptr<YandexIO::IDevice> device, std::shared_ptr<ipc::IIpcFactory> ipcFactory);

        void waitConnectionsAtLeast(size_t count);

        int port() const;

        static const std::string SERVICE_NAME;

    private:
        void processNtpdMessage(const quasar::proto::QuasarMessage& message);
        void processQuasarMessage(const ipc::SharedMessage& message, ipc::IServer::IClientConnection& connection);

        /**
         * Получает oauth-токен пользователя и добавляет его аккаунт в список известных аккаунтов, но пока не переключается
         * на него. Чтобы переключиться, нужно вызвать processChangeUserRequest.
         *
         * Так сделано, потому что прежде чем переключиться на другого пользователя, мы должны с ним зарегистрироваться
         * в бэкэнде. А чтобы зарегистрироваться в бэкэнде нам нужно знать oauth-токен этого пользователя, который в этом
         * методе как раз и получается.
         *
         * Таким образом, последовательность действий при логине/перелогине получается следующая:
         *
         * 1. processAddUserRequest - получает oauth-токен, добавляет аккаунт в список известных аккаунтов на диске. С этого
         *      момента и до завершения пункта 3 в этом списке будет один активный аккаунт и один неактивный (или несколько,
         *      в случае неудачных прошлых попыток) неактивный.
         * 2. FirstRunEndpoint - с полученным oauth-токеном регистрируется в бэкэнде
         * 3. processChangeUserRequest - окончательно переключается на этого пользователя и удаляет из списка все остальные
         */
        void processAddUserRequest(const quasar::proto::QuasarMessage& message, ipc::IServer::IClientConnection& connection);
        void tryProcessAddUserRequest(const quasar::proto::QuasarMessage& message, ipc::IServer::IClientConnection& connection);

        /**
         * Удаляет указанного пользователя. Нельзя удалить текущего пользователя.
         */
        void processDeleteUserRequest(const quasar::proto::QuasarMessage& message, ipc::IServer::IClientConnection& connection);
        void tryProcessDeleteUserRequest(const quasar::proto::QuasarMessage& message, ipc::IServer::IClientConnection& connection);

        /** Ищет аккаунт по идентификатору в списке известных аккаунтов и делает его активным.
         * Все последущие запросы будут
         * Только Admin аккаунты могут быть активными.
         */
        void processChangeUserRequest(const quasar::proto::QuasarMessage& message, ipc::IServer::IClientConnection& connection);
        void tryProcessChangeUserRequest(const quasar::proto::QuasarMessage& message, ipc::IServer::IClientConnection& connection);

        /* Пробует заново получить OAuth токен. */
        void processAuthTokenUpdateRequest(const proto::QuasarMessage& message, ipc::IServer::IClientConnection& connection);
        void tryProcessAuthTokenUpdateRequest(const proto::QuasarMessage& message, ipc::IServer::IClientConnection& connection);

        static std::int64_t makeTag();

        std::shared_ptr<YandexIO::IDevice> device_;
        std::shared_ptr<NamedCallbackQueue> callbackQueue_;

        AccountStorage accountStorage_;
        std::shared_ptr<ipc::IServer> server_;

        std::shared_ptr<ipc::IConnector> ntpdConnector_;
        bool ntpSynchronized_ = false;
    };

} // namespace quasar
