#include "ble_endpoint.h"

#include "quasar_gatt_characteristic.h"

#include <yandex_io/services/setupd/setup_service.h>

#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/logging/logging.h>

#include <util/system/yassert.h>

YIO_DEFINE_LOG_MODULE("ble_init");

using namespace quasar;
using namespace quasar::ble_configuration;
using namespace gatt_permission;

namespace {
    std::vector<uint8_t> convertFromString(const std::string& value) {
        return {(uint8_t*)value.data(), (uint8_t*)value.data() + value.size()};
    }

    std::string convertToString(const std::vector<uint8_t>& value) {
        return {(char*)value.data(), (char*)value.data() + value.size()};
    }

    std::string getStatusString(proto::SetupStatusMessage::SetupStatus status) {
        Json::Value value;
        value["status"] = status;
        return jsonToString(value);
    }

    QuasarGattProtocol getCharacteristicProtocol(const Json::Value& characteristic) {
        const auto protocol = tryGetString(characteristic, "protocol", "chunked");

        if (protocol == "plain") {
            return QuasarGattProtocol::PLAIN;
        } else if (protocol == "chunked") {
            return QuasarGattProtocol::CHUNKED;
        }

        throw std::runtime_error("Unknown protocol type");
    }
} // namespace

BleEndpoint::BleEndpoint(
    std::shared_ptr<YandexIO::IDevice> device,
    std::shared_ptr<ipc::IIpcFactory> ipcFactory,
    std::shared_ptr<IDeviceStateProvider> deviceStateProvider,
    std::shared_ptr<BluetoothLE> ble)
    : device_(std::move(device))
    , bleImpl_(std::move(ble))
    , setupdConnector_(ipcFactory->createIpcConnector("setupd"))
    , wifidConnector_(ipcFactory->createIpcConnector("wifid"))
    , server_(ipcFactory->createIpcServer(SERVICE_NAME))

{
    auto serviceConfig = device_->configuration()->getServiceConfig(SERVICE_NAME);
    enabled_ = tryGetBool(serviceConfig, "enabled", false);

    auto advertisingParams = device_->hal()->getHALInfo();
    Y_VERIFY(advertisingParams.has_value());

    uint16_t manufacturerId = advertisingParams->manufacturerId;
    uint8_t deviceColor = advertisingParams->deviceColorChar;
    uint8_t deviceType = (uint8_t)(advertisingParams->platformType);
    bleImpl_->setManufacturerData({manufacturerId, {deviceType, deviceColor}});

    Json::Value gattServices = getJson(serviceConfig, "services");
    Json::Value configurationServiceData = getJson(gattServices, "configuration");
    const auto configurationServicePtr = std::make_shared<IGattService>("configuration", getString(configurationServiceData, "uuid"));

    initConfigurationService(configurationServicePtr, getJson(configurationServiceData, "characteristics"));

    setupdConnector_->connectToService();

    wifidConnector_->setMessageHandler([this](const auto& message) {
        if (message->has_wifi_list()) {
            const auto hotspots = message->wifi_list().hotspots();
            if (hotspots.empty()) {
                YIO_LOG_WARN("Zero hotspots");
            }

            Json::Value wifiList{Json::arrayValue};
            for (const auto& hotspot : hotspots) {
                Json::Value value;
                value["ssid"] = hotspot.ssid();
                value["signal"] = hotspot.rssi();
                value["secure"] = hotspot.secure();
                wifiList.append(value);
            }
            // After initConfigurationService() this std::shared_ptr must be already set
            characteristics_["wifiList"]->setValue(convertFromString(jsonToString(wifiList)));
        }
    });
    wifidConnector_->connectToService();

    deviceStateProvider->configurationChangedSignal().connect(
        [this](const auto& deviceState) {
            if (deviceState->configuration == DeviceState::Configuration::CONFIGURING) {
                switchBleAdvertising(enabled_, true);
            }
        }, lifetime_);

    server_->setMessageHandler(std::bind(&BleEndpoint::processQuasarMessage, this, std::placeholders::_1));
    server_->listenService();
}

void BleEndpoint::initConfigurationService(const std::shared_ptr<IGattService>& service,
                                           const Json::Value& characteristicsConfig)
{
    bleImpl_->addGattService(service);

    for (auto it = characteristicsConfig.begin(); it != characteristicsConfig.end(); it++) {
        auto characteristic = std::make_shared<QuasarGattCharacteristic>(
            it.name(),
            getString(*it, "uuid"),
            getCharacteristicPermissions(*it),
            getCharacteristicProtocol(*it));

        bleImpl_->addGattCharacteristic("configuration", characteristic);

        characteristics_.emplace(it.name(), std::move(characteristic));
    }

    characteristics_["deviceId"]->setValue(convertFromString(device_->deviceId()));
    characteristics_["status"]->setValue(convertFromString(getStatusString(proto::SetupStatusMessage::NONE)));

    characteristics_["connect"]->setWriteCallback([this](std::vector<uint8_t> data) {
        onConnectMessageReceived(std::move(data));
    });

    YIO_LOG_INFO("Configuration service inited");
}

std::set<GattPermission> BleEndpoint::getCharacteristicPermissions(const Json::Value& characteristic)
{
    std::set<GattPermission> result;
    const auto perms = tryGetArray(characteristic, "permissions");
    for (auto& perm : perms) {
        try {
            result.insert(stringToPermission(perm.asString()));
        } catch (const std::runtime_error& e) {
            YIO_LOG_ERROR_EVENT("BleEndpoint.FailedAddPermission", "Adding permission error: " << e.what());
        }
    }
    return result;
}

BleEndpoint::~BleEndpoint()
{
    lifetime_.die();
    wifidConnector_->shutdown();
    setupdConnector_->shutdown();
    server_->shutdown();
}

void BleEndpoint::processQuasarMessage(const ipc::SharedMessage& message)
{
    if (message->has_setup_status_message()) {
        sendStatusToClient(message->setup_status_message().status());

        /*   If critical update is required after configuration, then we shouldn't stop BLE server,
         * as we need to send UPDATE_DOWNLOAD_FINISH notification afterwards. Moreover, in this
         * case we needn't stop BLE server at all, because our device will be rebooted automatically
         * after successful OTA download.
         *   So, when we get UPDATE_IS_REQUIRED, we remember it as `updateInProgress`, and when
         * we get CONFIGURATION_FINISHED later, we don't stop BLE server in case `updateInProgress` is true. */
        if (message->setup_status_message().status() == proto::SetupStatusMessage::UPDATE_IS_REQUIRED) {
            updateInProgress_ = true;
        } else if (message->setup_status_message().status() == proto::SetupStatusMessage::CONFIGURATION_FINISHED) {
            /* We don't use ConfigurationState::CONFIGURED here because somehow it arrives
             * long before us finding out whether we need to download update or not. */
            switchBleAdvertising(enabled_, updateInProgress_);
        }
    }
}

void BleEndpoint::sendStatusToClient(proto::SetupStatusMessage::SetupStatus status)
{
    if (status == proto::SetupStatusMessage::CONFIGURATION_FINISHED) {
        // Internal status
        return;
    }

    const auto value = getStatusString(status);

    YIO_LOG_INFO("Sending setup status " << value);

    characteristics_["status"]->setValue(convertFromString(value));
}

void BleEndpoint::switchBleAdvertising(bool enabled, bool setupInProgress)
{
    YIO_LOG_INFO("switchBleAdvertising");
    bool wasStarted = enabled_ && setupInProgress_;
    bool shouldStart = enabled && setupInProgress;

    if (wasStarted != shouldStart) {
        if (shouldStart) {
            YIO_LOG_INFO("Starting BLE server");
            bleImpl_->startAdvertising();
        } else {
            YIO_LOG_INFO("Stopping BLE server");
            bleImpl_->stopAdvertising();
        }
    }

    enabled_ = enabled;
    setupInProgress_ = setupInProgress;
}

void BleEndpoint::onConnectMessageReceived(std::vector<uint8_t> data) {
    auto setupValueMaybe = tryParseJson(convertToString(data));
    if (setupValueMaybe.has_value()) {
        auto setupValue = setupValueMaybe.value();
        proto::QuasarMessage message;
        if (setupValue["ciphertext"].isString()) {
            if (!setupValue["C"].isString() || !setupValue["M"].isString()) {
                YIO_LOG_ERROR_EVENT("BleEndpoint.IncompleteEncryptedSetupCredentials", "Incomplete encrypted setup credentials value");
                return;
            }

            YIO_LOG_INFO("Got encrypted setup credentials");

            auto setupMessage = message.mutable_encrypted_setup_credentials_message();
            setupMessage->set_c(base64Decode(getString(setupValue, "C")));
            setupMessage->set_m(base64Decode(getString(setupValue, "M")));
            setupMessage->set_ciphertext(base64Decode(getString(setupValue, "ciphertext")));
            setupMessage->set_source(proto::SetupSource::BLE);
        } else if (setupValue["plaintext"].isString()) {
            YIO_LOG_INFO("Got plaintext setup credentials");

            auto setupMessage = message.mutable_setup_credentials_message();
            setupMessage->set_setup_credentials(base64Decode(getString(setupValue, "plaintext")));
            setupMessage->set_source(proto::SetupSource::BLE);
        } else {
            YIO_LOG_ERROR_EVENT("BleEndpoint.InvalidSetupCredentialsFormat", "Unknown setup credentials value");
            return;
        }
        setupdConnector_->sendMessage(std::move(message));
    } else {
        YIO_LOG_ERROR_EVENT("BleEndpoint.BadJson.SetupCredentials", "Can't parse received value");
        return;
    }
}

int BleEndpoint::port() const {
    return server_->port();
}

const std::string BleEndpoint::SERVICE_NAME = "ble_initd";
