#include "gatt_value_format.h"

#include <util/system/byteorder.h>

using namespace quasar;

ble_configuration::QuasarGattMessageHeader ble_configuration::deserialize(const std::vector<uint8_t>& data, size_t& from) {
    QuasarGattMessageHeader res;
    if (data.size() - from < sizeof(res.version)) {
        throw std::runtime_error("Data size is too small for header");
    }

    memcpy(&res.version, data.data() + from, sizeof(res.version));
    from += sizeof(res.version);

    switch (res.version) {
        case QuasarGattProtocol::PLAIN: {
            res.size = data.size() - from;
            break;
        }
        case QuasarGattProtocol::CHUNKED: {
            if (data.size() - from < sizeof(res.size)) {
                throw std::runtime_error("Data size is too small for header");
            }

            memcpy(&res.size, data.data() + from, sizeof(res.size));
            from += sizeof(res.size);
            res.size = InetToHost<uint32_t>(res.size);
            break;
        }
        default:
            throw std::runtime_error("Unsupported protocol type");
    }

    return res;
}

std::vector<uint8_t> ble_configuration::serialize(const QuasarGattMessageHeader& header) {
    std::vector<uint8_t> res;
    res.insert(res.end(), static_cast<uint8_t>(header.version));
    switch (header.version) {
        case QuasarGattProtocol::PLAIN: {
            if (header.size + sizeof(header.version) > MAX_CHUNK_SIZE_BYTES) {
                throw std::runtime_error("Data is too large for default protocol");
            }
            break;
        }
        case QuasarGattProtocol::CHUNKED: {
            uint32_t sizeBigEndian = HostToInet<uint32_t>(header.size);
            res.insert(res.end(), (uint8_t*)&sizeBigEndian, (uint8_t*)&sizeBigEndian + sizeof(sizeBigEndian));
            break;
        }
        default:
            throw std::runtime_error("Unsupported protocol type");
    }
    return res;
}
