#pragma once

#include <cstdint>
#include <vector>

namespace quasar {

    namespace ble_configuration {

        // GATT characteristic value is limited to 512 bytes
        static constexpr size_t MAX_CHUNK_SIZE_BYTES = 512;

        // We set maximum data size to evade very big memory allocations
        static constexpr int MAX_DATA_SIZE_BYTES = 1024 * 1024;

        // All GATT values start with a protocol version (1 byte)
        enum class QuasarGattProtocol: uint8_t {
            // Version is immediately followed by payload.
            // Payload size is limited to 511 bytes.
            PLAIN = 0,
            // Version is followed by payload size (4 bytes big-endian)
            // Payload is contained in one or more GATT characteristic values (up to 512 bytes each)
            CHUNKED = 1
        };

        struct QuasarGattMessageHeader {
            QuasarGattProtocol version{QuasarGattProtocol::PLAIN};
            uint32_t size{0};
        };

        /**
         * Deserializes header from byte vector
         * @param[in] data serialized header (with probably some data after header)
         * @param[in,out] from index from where to start reading. If header reading is successful, this index moves forward on header size
         * @return QuasarGattMessageHeader
         * @throws std::runtime_error if data size is not big enough for reading header
         */
        QuasarGattMessageHeader deserialize(const std::vector<uint8_t>& data, size_t& from);
        std::vector<uint8_t> serialize(const QuasarGattMessageHeader& header);

    } // namespace ble_configuration

} // namespace quasar
