#include "gatt_value_reader.h"

using namespace quasar;
using namespace quasar::ble_configuration;

GattValueReader::GattValueReader(
    const std::vector<uint8_t>& value,
    QuasarGattProtocol protocol)
    : chunkOffset_(0)
{
    QuasarGattMessageHeader header{
        protocol,
        static_cast<uint32_t>(value.size()),
    };

    data_ = serialize(header);
    data_.insert(data_.end(), value.begin(), value.end());
}

GattValueReader::Result GattValueReader::read(uint16_t mtu, uint16_t offset) {
    if (isFinished()) {
        return {ResultCode::ALL_READ, {}};
    }

    if (offset > MAX_CHUNK_SIZE_BYTES || offset > data_.size() - chunkOffset_) {
        return {ResultCode::INVALID_OFFSET, {}};
    }

    size_t len = std::min<size_t>({
        data_.size() - chunkOffset_ - offset,
        MAX_CHUNK_SIZE_BYTES - offset,
        mtu - 1u,
    });

    std::vector<uint8_t> result{
        data_.begin() + chunkOffset_ + offset,
        data_.begin() + chunkOffset_ + offset + len,
    };

    // Last read of each chunk should be shorter than (MTU - 1)
    if (len < mtu - 1) {
        chunkOffset_ += MAX_CHUNK_SIZE_BYTES;
    }

    return {ResultCode::OK, std::move(result)};
}

bool GattValueReader::isFinished() const {
    return chunkOffset_ >= data_.size();
}

GattValueReader::Result::Result(GattValueReader::ResultCode code, std::vector<uint8_t> data)
    : code(code)
    , data(std::move(data))
{
}
