#include "gatt_value_writer.h"

#include <yandex_io/libs/logging/logging.h>

YIO_DEFINE_LOG_MODULE("ble_init");

using namespace quasar;
using namespace quasar::ble_configuration;

GattValueWriter::GattValueWriter(std::weak_ptr<QuasarGattCharacteristic> ownerWeak)
    : ownerWeak_(std::move(ownerWeak))
{
}

GattValueWriter::ResultCode GattValueWriter::write(const std::vector<uint8_t>& value) {
    if (isAlreadyFull_) {
        YIO_LOG_ERROR_EVENT("GattValueWriter.FailedWrite.Overflow", "Gatt value is already full, skipping data");
        return ResultCode::ALREADY_FULL;
    }
    if (isError_) {
        YIO_LOG_ERROR_EVENT("GattValueWriter.FailedWrite.BrokenState", "There was error when writing data, writer may be invalid");
        return ResultCode::ALREADY_MISTAKEN;
    }
    size_t beginDataIdx = 0;
    if (header_.size == 0) {
        try {
            header_ = deserialize(value, beginDataIdx);
        } catch (const std::runtime_error& e) {
            isError_ = true;
            return ResultCode::HEADER_ERROR;
        }
        if (header_.size > MAX_DATA_SIZE_BYTES) {
            isError_ = true;
            return ResultCode::TOO_LARGE_DATA;
        }
        data_.reserve(header_.size);
    }
    data_.insert(data_.end(), value.begin() + beginDataIdx, value.end());
    if (data_.size() >= header_.size) {
        YIO_LOG_INFO("Received full GATT value");
        data_.resize(header_.size);
        isAlreadyFull_ = true;
    }

    // We received the whole data
    if (isAlreadyFull_) {
        if (auto owner = ownerWeak_.lock()) {
            owner->onWriteSessionDone(data_);
        }
    }
    return ResultCode::OK;
}

bool GattValueWriter::isFinished() const {
    return isAlreadyFull_;
}
