#include "quasar_gatt_characteristic.h"

#include <yandex_io/libs/logging/logging.h>

using namespace quasar;
using namespace quasar::ble_configuration;

using namespace gatt_permission;

QuasarGattCharacteristic::QuasarGattCharacteristic(
    std::string name,
    std::string uuid,
    std::set<GattPermission> permissions,
    QuasarGattProtocol protocol)
    : IGattCharacteristic(std::move(name), std::move(uuid), std::move(permissions))
    , protocol_(protocol)
{
}

void QuasarGattCharacteristic::setWriteCallback(OnWriteSessionDone onWriteSessionDone) {
    std::lock_guard guard(mutex_);
    onWriteSessionDone_ = onWriteSessionDone;
}

void QuasarGattCharacteristic::setValue(std::vector<uint8_t> value) {
    std::lock_guard guard(mutex_);
    value_ = std::move(value);
    notify();
}

void QuasarGattCharacteristic::onWriteSessionDone(std::vector<uint8_t> value) {
    if (onWriteSessionDone_) {
        onWriteSessionDone_(std::move(value));
    }
}

std::vector<uint8_t> QuasarGattCharacteristic::readValueImpl(const IGattCharacteristic::Mac& mac, uint16_t mtu, uint16_t offset) {
    std::lock_guard guard(mutex_);

    auto it = readSessions_.find(mac);
    if (it == readSessions_.end()) {
        it = readSessions_.emplace(mac, std::make_shared<GattValueReader>(value_, protocol_)).first;
    }

    auto res = it->second->read(mtu, offset);

    if (res.code != GattValueReader::ResultCode::OK) {
        readSessions_.erase(it);
        throw std::runtime_error("Non-OK reading");
    }

    if (it->second->isFinished()) {
        readSessions_.erase(it);
    }

    return res.data;
}

void QuasarGattCharacteristic::writeValueImpl(const IGattCharacteristic::Mac& mac, const std::vector<uint8_t>& value) {
    std::lock_guard guard(mutex_);

    auto it = writeSessions_.find(mac);
    if (it == writeSessions_.end()) {
        it = writeSessions_.emplace(mac, std::make_shared<GattValueWriter>(weak_from_this())).first;
    }

    if (it->second->write(value) != GattValueWriter::ResultCode::OK) {
        writeSessions_.erase(it);
        throw std::runtime_error("Non-OK writing");
    }

    if (it->second->isFinished()) {
        writeSessions_.erase(it);
    }
}

void QuasarGattCharacteristic::dropWriteSession(const IGattCharacteristic::Mac& mac) {
    std::lock_guard guard(mutex_);
    writeSessions_.erase(mac);
}

void QuasarGattCharacteristic::dropReadSession(const IGattCharacteristic::Mac& mac) {
    std::lock_guard guard(mutex_);
    readSessions_.erase(mac);
}

std::vector<uint8_t> QuasarGattCharacteristic::getValue() const {
    std::vector<uint8_t> value = serialize({
        protocol_,
        static_cast<uint32_t>(value_.size()),
    });

    value.insert(value.end(), value_.begin(), value_.end());

    return value;
}
