#include "brain_service.h"

#include <google/protobuf/util/message_differencer.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>

#include <algorithm>
#include <chrono>
#include <functional>
#include <memory>
#include <cstdint>

using namespace quasar;

BrainService::BrainService(std::shared_ptr<YandexIO::IDevice> device, std::shared_ptr<ipc::IIpcFactory> ipcFactory)
    : device_(std::move(device))
{
    auto onMessage = [this](const ipc::SharedMessage& message) { handleQuasarMessage(*message); };
    auto createConnector = [&](const std::string& serviceName) {
        auto connector = ipcFactory->createIpcConnector(serviceName);
        connector->setMessageHandler(onMessage);
        return connector;
    };

    alicedConnector_ = createConnector("aliced");
    alarmdConnector_ = createConnector("alarmd");
    firstrundConnector_ = createConnector("firstrund");
    updatesdConnector_ = createConnector("updatesd");
    wifidConnector_ = createConnector("wifid");
    notificationdConnector_ = createConnector("notificationd");
    calldConnector_ = createConnector("calld");
    dndConnector_ = createConnector("do_not_disturb");
    ntpdConnector_ = createConnector("ntpd");
    iotConnector_ = createConnector("iot");
    multiroomConnector_ = createConnector("multiroomd");

    deviceContext_ = std::make_unique<YandexIO::DeviceContext>(ipcFactory,
                                                               [this]() {
                                                                   std::lock_guard<std::mutex> guard(mutex_);
                                                                   broadcastSDKState();
                                                               }, false);
}

void BrainService::start()
{
    alicedConnector_->connectToService();
    alarmdConnector_->connectToService();
    firstrundConnector_->connectToService();
    updatesdConnector_->connectToService();
    wifidConnector_->connectToService();
    notificationdConnector_->connectToService();
    dndConnector_->connectToService();
    calldConnector_->tryConnectToService();
    ntpdConnector_->tryConnectToService();
    iotConnector_->tryConnectToService();
    multiroomConnector_->tryConnectToService();

    deviceContext_->connectToSDK();
}

BrainService::~BrainService()
{
    alicedConnector_->shutdown();
    alarmdConnector_->shutdown();
    firstrundConnector_->shutdown();
    updatesdConnector_->shutdown();
    wifidConnector_->shutdown();
    notificationdConnector_->shutdown();
    dndConnector_->shutdown();
    calldConnector_->shutdown();
    ntpdConnector_->shutdown();
    iotConnector_->shutdown();
    multiroomConnector_->shutdown();

    deviceContext_.reset(nullptr);
}

uint64_t getEndTs(const proto::Alarm& alarm) {
    uint64_t endTs = alarm.start_timestamp_ms() + (alarm.duration_seconds() + alarm.paused_seconds()) * 1000;
    uint64_t now = std::chrono::system_clock::now().time_since_epoch().count();

    if (alarm.has_pause_timestamp_sec()) {
        endTs += (now / 1000 - alarm.pause_timestamp_sec()) * 1000;
    }
    return endTs;
}

void BrainService::handleQuasarMessage(const proto::QuasarMessage& quasarMessage) {
    std::lock_guard<std::mutex> guard(mutex_);

    bool broadcastState = false;
    if (quasarMessage.has_alice_state()) {
        const auto& aliceState = quasarMessage.alice_state();
        if (aliceState.has_state()) {
            aliceState_.CopyFrom(aliceState);
            broadcastState = true;
        }
    }

    if (quasarMessage.has_do_not_disturb_event()) {
        if (quasarMessage.do_not_disturb_event().has_is_dnd_enabled()) {
            isDoNotDisturbMode_ = quasarMessage.do_not_disturb_event().is_dnd_enabled();
        }
    }

    if (quasarMessage.has_alarm_event()) {
        if (quasarMessage.alarm_event().has_alarm_started()) {
            if (quasarMessage.alarm_event().alarm_started().alarm_type() == proto::Alarm_AlarmType_TIMER) {
                isTimerPlaying_ = true;
            } else {
                isAlarmPlaying_ = true;
            }
            broadcastState = true;
        } else if (quasarMessage.alarm_event().has_alarm_stopped()) {
            isAlarmPlaying_ = false;
            isTimerPlaying_ = false;
            broadcastState = true;
        }
    }

    // init with -1, so if message doesn't have alarms or timers info, we don't update saved value
    int32_t setAlarmsTimerCount = -1;

    if (quasarMessage.has_timers_state()) {
        auto now = std::chrono::system_clock::now().time_since_epoch();
        auto nowMs = std::chrono::duration_cast<std::chrono::milliseconds>(now).count();

        std::vector<uint64_t> timersMs;
        timers_.clear();
        for (const auto& timer : quasarMessage.timers_state().timers()) {
            if (timer.has_alarm_type() && timer.alarm_type() == proto::Alarm_AlarmType_TIMER &&
                timer.start_timestamp_ms() <= nowMs && !timer.has_pause_timestamp_sec()) // a valid non-paused timer
            {
                timers_.push_back({timer.id(), static_cast<std::uint64_t>(timer.start_timestamp_ms()), getEndTs(timer)});
            }
        }

        std::sort(timers_.begin(), timers_.end(), [](const auto& a, const auto& b) {
            // sort by end time.
            if (a.endTs != b.endTs) {
                return a.endTs < b.endTs;
            }
            return a.startTs < b.startTs;
        });
        broadcastState = true;

        if (setAlarmsTimerCount == -1) {
            setAlarmsTimerCount = 0;
        }
        setAlarmsTimerCount += timers_.size();
    }

    if (quasarMessage.has_set_alarms_state()) {
        if (setAlarmsTimerCount == -1) {
            setAlarmsTimerCount = 0;
        }
        setAlarmsTimerCount += quasarMessage.set_alarms_state().timers().size();
    }

    if (quasarMessage.has_alarms_state()) {
        const auto& alarms_state = quasarMessage.alarms_state();
        if (alarms_state.has_icalendar_state()) {
            icalendar_state_ = alarms_state.icalendar_state();
            broadcastState = true;
        }
    }

    if (setAlarmsTimerCount != -1) {
        setAlarmsTimersCount_ = setAlarmsTimerCount;
        broadcastState = true;
    }

    if (quasarMessage.has_reminder_message()) {
        if (quasarMessage.reminder_message().has_reminder_started()) {
            isReminderPlaying_ = true;
            broadcastState = true;
        } else if (quasarMessage.reminder_message().has_reminder_stopped()) {
            isReminderPlaying_ = false;
            broadcastState = true;
        }
    }

    if (quasarMessage.has_configuration_state()) {
        if (quasarMessage.configuration_state() != configurationState_) {
            configurationState_ = quasarMessage.configuration_state();
            broadcastState = true;
        }
    }

    if (quasarMessage.has_update_state()) {
        if (updateState_.state() != quasarMessage.update_state().state() ||
            updateState_.download_progress() != quasarMessage.update_state().download_progress() ||
            updateState_.is_critical() != quasarMessage.update_state().is_critical()) {
            updateState_.CopyFrom(quasarMessage.update_state());
            broadcastState = true;
        }
    }

    if (quasarMessage.has_app_state()) {
        if (appState_.music_state().is_paused() != quasarMessage.app_state().music_state().is_paused() || appState_.music_state().current_track_id() != quasarMessage.app_state().music_state().current_track_id() || appState_.radio_state().is_paused() != quasarMessage.app_state().radio_state().is_paused() || appState_.screen_state().is_screensaver_on() != quasarMessage.app_state().screen_state().is_screensaver_on() || appState_.screen_state().screen_type() != quasarMessage.app_state().screen_state().screen_type() || appState_.audio_player_state().state() != quasarMessage.app_state().audio_player_state().state() || appState_.video_state().is_paused() != quasarMessage.app_state().video_state().is_paused()) {
            appState_.CopyFrom(quasarMessage.app_state());
            broadcastState = true;
        }
    }

    if (quasarMessage.has_wifi_status()) {
        if (quasarMessage.wifi_status().status() != wifiStatus_.status() ||
            quasarMessage.wifi_status().internet_reachable() != wifiStatus_.internet_reachable()) {
            wifiStatus_.CopyFrom(quasarMessage.wifi_status());
            broadcastState = true;
        }
    }

    if (quasarMessage.has_wifi_list()) {
        const auto& wifiList = quasarMessage.wifi_list().hotspots();
        if (!std::equal(
                wifiList.begin(), wifiList.end(),
                wifiList_.hotspots().begin(), wifiList_.hotspots().end(),
                google::protobuf::util::MessageDifferencer::Equals)) {
            wifiList_.CopyFrom(quasarMessage.wifi_list());
            broadcastState = true;
        }
    }

    if (quasarMessage.has_notification_update_event() && quasarMessage.notification_update_event().has_notification_mode()) {
        switch (quasarMessage.notification_update_event().notification_mode()) {
            case proto::NotificationUpdateEvent_NotificationMode_ACTIVE:
                notificationState_ = proto::IOEvent_SDKState_NotificationState_AVAILABLE;
                break;
            case proto::NotificationUpdateEvent_NotificationMode_PASSIVE:
                notificationState_ = proto::IOEvent_SDKState_NotificationState_PASSIVE;
                break;
            case proto::NotificationUpdateEvent_NotificationMode_NONE:
            default:
                notificationState_ = proto::IOEvent_SDKState_NotificationState_NONE;
        }

        broadcastState = true;
    }

    if (quasarMessage.has_calld_state_changed()) {
        calldState_.CopyFrom(quasarMessage.calld_state_changed());
        broadcastState = true;
    }

    if (quasarMessage.has_ntp_sync_event() && quasarMessage.ntp_sync_event().has_is_ntp_sync_successful()) {
        clockSynchronized_ = quasarMessage.ntp_sync_event().is_ntp_sync_successful();
        broadcastState = true;
    }

    if (quasarMessage.has_iot_state()) {
        iotState_.CopyFrom(quasarMessage.iot_state());
        broadcastState = true;
    }

    if (quasarMessage.has_multiroom_state()) {
        multiroomState_.CopyFrom(quasarMessage.multiroom_state());
        broadcastState = true;
    }

    if (quasarMessage.has_all_startup_settings()) {
        allStartupSettings_.CopyFrom(quasarMessage.all_startup_settings());
        broadcastState = true;
    }

    if (broadcastState) {
        broadcastSDKState();
    }
}

void BrainService::broadcastSDKState()
{
    proto::IOEvent::SDKState state;
    state.mutable_alice_state()->CopyFrom(aliceState_);
    state.mutable_alarms_state()->set_alarm_playing(isAlarmPlaying_);
    state.mutable_alarms_state()->set_timer_playing(isTimerPlaying_);
    state.mutable_alarms_state()->set_set_alarms_timers_count(setAlarmsTimersCount_);
    state.set_icalendar_state(TString(icalendar_state_));
    state.mutable_reminder_state()->set_reminder_playing(isReminderPlaying_);
    state.mutable_do_not_disturb_state()->set_is_dnd_mode(isDoNotDisturbMode_);
    state.set_configuration_state(configurationState_);
    state.mutable_update_state()->CopyFrom(updateState_);
    state.mutable_app_state()->CopyFrom(appState_);
    state.mutable_wifi_status()->CopyFrom(wifiStatus_);
    state.mutable_wifi_list()->CopyFrom(wifiList_);
    state.set_notification_state(notificationState_);
    state.mutable_calld_state()->CopyFrom(calldState_);
    state.mutable_iot_state()->CopyFrom(iotState_);
    state.mutable_multiroom_state()->CopyFrom(multiroomState_);
    state.mutable_all_startup_settings()->CopyFrom(allStartupSettings_);
    auto timersTimings = state.mutable_timers_timings();
    for (const auto& [id, start, end] : timers_) {
        auto newElem = timersTimings->Add();
        newElem->set_start_timer_ms(start);
        newElem->set_end_timer_ms(end);
        newElem->set_id(TString(id));
    }
    state.mutable_ntpd_state();
    if (clockSynchronized_.has_value()) {
        state.mutable_ntpd_state()->set_clock_synchronized(clockSynchronized_.value());
    }

    deviceContext_->fireSDKState(state);
}
