#include "brick_endpoint.h"

#include <yandex_io/libs/base/persistent_file.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>

#include <fstream>
#include <stdio.h>

YIO_DEFINE_LOG_MODULE("brick");

using namespace quasar;

BrickEndpoint::BrickEndpoint(std::shared_ptr<YandexIO::IDevice> device, std::shared_ptr<ipc::IIpcFactory> ipcFactory)
    : device_(std::move(device))
    , server_(ipcFactory->createIpcServer(SERVICE_NAME))
    , syncdConnector_(ipcFactory->createIpcConnector("syncd"))
    , deviceContext_(ipcFactory, [this]() {
        std::scoped_lock guard(brickMutex_);
        deviceContext_.fireBrickStatusChanged(createBrickInfo());
    }, false)
{
    auto serviceConfig = device_->configuration()->getServiceConfig(SERVICE_NAME);
    firstBrickCheckDelay_ = std::chrono::milliseconds(
        tryGetUInt64(serviceConfig, "firstBrickCheckDelayMS", 10 * 60 * 1000 /* 10 minutes */));
    regularBrickCheckDelay_ = std::chrono::milliseconds(
        tryGetUInt64(serviceConfig, "regularBrickCheckDelayMS", 1 * 60 * 1000 /* 1 minute */));
    subscriptionModeFilename_ = tryGetString(serviceConfig, "subscriptionModeFilename", "");

    subscriptionMode_ = tryGetBool(serviceConfig, "subscriptionModeByDefault", false);
    useSteadyClock_ = tryGetBool(serviceConfig, "useSteadyClock", false);
    loadSubscriptionMode();
    if (!subscriptionMode_) {
        brickStatus_ = proto::BrickStatus::NOT_BRICK;
    } else {
        brickStatus_ = proto::BrickStatus::UNKNOWN_BRICK_STATUS;
    }
    device_->telemetry()->putAppEnvironmentValue("brickStatus", brickStatusToString(brickStatus_));

    auto periodicCheckBrickStatus = [this](PeriodicExecutor* executor) {
        checkBrickStatus();
        executor->setPeriodTime(regularBrickCheckDelay_);
    };
    brickExecutor_ = std::make_unique<PeriodicExecutor>(
        PeriodicExecutor::PECallback(periodicCheckBrickStatus),
        firstBrickCheckDelay_,
        PeriodicExecutor::PeriodicType::SLEEP_FIRST);

    server_->setClientConnectedHandler([this](auto& connection) {
        std::lock_guard<std::mutex> lockGuard(brickMutex_);
        proto::QuasarMessage brickMessage;
        brickMessage.set_brick_status(brickStatus_);
        YIO_LOG_DEBUG("Sending brick status onClientConnected " << brickStatusToString(brickStatus_));
        connection.send(std::move(brickMessage));
    });

    server_->listenService();

    syncdConnector_->setMessageHandler(std::bind(&BrickEndpoint::processQuasarMessage, this, std::placeholders::_1));
    syncdConnector_->connectToService();
    deviceContext_.connectToSDK();
}

BrickEndpoint::~BrickEndpoint()
{
    brickExecutor_.reset(nullptr);

    syncdConnector_->shutdown();
    server_->shutdown();
}

int BrickEndpoint::port() const {
    return server_->port();
}

void BrickEndpoint::processQuasarMessage(const ipc::SharedMessage& message)
{
    if (message->has_auth_failed()) {
        std::lock_guard<std::mutex> lockGuard(brickMutex_);
        updateStatus(proto::BrickStatus::UNKNOWN_BRICK_STATUS, brickStatusUrl_, subscriptionMode_);
    }
    if (message->has_subscription_state()) {
        std::lock_guard<std::mutex> lockGuard(brickMutex_);
        const auto& state = message->subscription_state();
        try {
            backendUpdateReceived_ = !state.is_saved_state();
            lastUpdateTimeSec_ = useSteadyClock_ ? getSteadyNowTimestampSec() : state.last_update_time();
            const auto info = parseJson(state.subscription_info());

            const std::string mode = tryGetString(info, "mode", "");
            if (mode.empty()) {
                YIO_LOG_ERROR_EVENT("BrickEndpoint.MissingSubscriptionStateField.Mode", "Subscription info missed required field mode");
            } else {
                const bool newSubscriptionMode = mode == "subscription";
                if (newSubscriptionMode) {
                    bool enabled = false;
                    try {
                        enabled = getBool(info, "enabled");
                    } catch (const std::runtime_error& e) {
                        YIO_LOG_ERROR_EVENT("BrickEndpoint.MissingSubscriptionStateField.Enabled", "Subscription info missed required field enabled for subscription mode");
                    }
                    try {
                        ttlSec_ = getInt64(info, "ttl");
                    } catch (const std::runtime_error& e) {
                        YIO_LOG_ERROR_EVENT("BrickEndpoint.MissingSubscriptionStateField.Ttl", "Subscription info missed required field ttl for subscription mode");
                    }
                    std::string newBrickUrl = tryGetString(info, "statusUrl", std::string());

                    if (enabled && !isBrickRequired()) {
                        updateStatus(proto::BrickStatus::NOT_BRICK, std::string(), newSubscriptionMode);
                    } else {
                        updateStatus(backendUpdateReceived_ ? proto::BrickStatus::BRICK : proto::BrickStatus::BRICK_BY_TTL, newBrickUrl, newSubscriptionMode);
                    }
                } else {
                    updateStatus(proto::BrickStatus::NOT_BRICK, std::string(), newSubscriptionMode);
                }
            }
        } catch (const Json::Exception& exc) {
            YIO_LOG_ERROR_EVENT("BrickEndpoint.BadJson.SubscriptionState", "Can't parse subscription info, " << exc.what());
        }
    }

    if (onQuasarMessageReceivedCallback) {
        onQuasarMessageReceivedCallback(*message); // Testing purposes only!
    }
}

/* must be called under brickMutex_ */
void BrickEndpoint::updateStatus(proto::BrickStatus newBrickStatus, const std::string& newBrickUrl, bool newSubscriptionMode)
{
    YIO_LOG_INFO(
        "Updating brickStatus: " << brickStatusToString(newBrickStatus) << ", brickStatusUrl: " << newBrickUrl << ", subscriptionMode: " << newSubscriptionMode);

    if (newBrickStatus != brickStatus_ || newBrickUrl != brickStatusUrl_ || newSubscriptionMode != subscriptionMode_) {
        YIO_LOG_INFO("Brick status changed. Sending to all");

        subscriptionMode_ = newSubscriptionMode;
        saveSubscriptionMode();
        brickStatus_ = newBrickStatus;
        brickStatusUrl_ = newBrickUrl;

        device_->telemetry()->putAppEnvironmentValue("brickStatus", brickStatusToString(brickStatus_));
        Json::Value status;
        status["status"] = brickStatusToString(brickStatus_);
        status["statusUrl"] = brickStatusUrl_;
        status["subscriptionMode"] = subscriptionMode_;
        device_->telemetry()->reportEvent("brickStatusChanged", jsonToString(status));

        deviceContext_.fireBrickStatusChanged(createBrickInfo());
        proto::QuasarMessage brickMessage;
        brickMessage.set_brick_status(brickStatus_);
        server_->sendToAll(std::move(brickMessage));
    }
}

/* must be called under brickMutex_ */
bool BrickEndpoint::isBrickRequired() const {
    if (!subscriptionMode_) {
        return false;
    }

    if (useSteadyClock_) {
        int64_t nowSec = getSteadyNowTimestampSec();
        // If backend brick update wasn't received wait till first brick check
        int64_t ttl = backendUpdateReceived_ ? ttlSec_ : duration_cast<std::chrono::seconds>(firstBrickCheckDelay_).count();
        return lastUpdateTimeSec_ + ttl < nowSec;
    } else {
        int64_t nowSec = getSystemNowTimestampSec();
        return lastUpdateTimeSec_ + ttlSec_ < nowSec;
    }
}

void BrickEndpoint::checkBrickStatus() {
    std::lock_guard<std::mutex> lockGuard(brickMutex_);
    if (brickStatus_ == proto::BrickStatus::BRICK || brickStatus_ == proto::BrickStatus::BRICK_BY_TTL) {
        return;
    }

    if (isBrickRequired()) {
        updateStatus(backendUpdateReceived_ ? proto::BrickStatus::BRICK : proto::BrickStatus::BRICK_BY_TTL, brickStatusUrl_, subscriptionMode_);
    }
}

std::string BrickEndpoint::brickStatusToString(proto::BrickStatus newBrickStatus)
{
    static_assert(proto::BrickStatus_ARRAYSIZE == 5, "New types were added!");
    switch (newBrickStatus) {
        case proto::BrickStatus::BRICK:
            return "BRICK";
        case proto::BrickStatus::BRICK_BY_TTL:
            return "BRICK_BY_TTL";
        case proto::BrickStatus::NOT_BRICK:
            return "NOT_BRICK";
        case proto::BrickStatus::UNKNOWN_BRICK_STATUS:
            return "UNKNOWN_BRICK_STATUS";
        default:
            throw std::runtime_error("Unknown type");
    }
}

/* must be called under brickMutex_ */
void BrickEndpoint::saveSubscriptionMode() const {
    if (subscriptionModeFilename_.empty()) {
        return;
    }

    PersistentFile file(subscriptionModeFilename_, PersistentFile::Mode::TRUNCATE);

    file.write(subscriptionMode_ ? "1" : "0");
}

void BrickEndpoint::loadSubscriptionMode()
{
    if (subscriptionModeFilename_.empty() || !fileExists(subscriptionModeFilename_)) {
        return;
    }

    std::ifstream file(subscriptionModeFilename_);
    if (file.good()) {
        std::string line;
        std::getline(file, line);
        try {
            int subscriptionMode = std::stoi(line);
            // Here convert int value, stored in file to bool
            subscriptionMode_ = subscriptionMode;
            YIO_LOG_INFO("Subscription mode " << subscriptionMode_ << " restored from file");
        } catch (const std::logic_error& e) {
            YIO_LOG_ERROR_EVENT("BrickEndpoint.FailedLoadSubscriptionMode", "Error when reading number from file " << subscriptionModeFilename_);
            std::remove(subscriptionModeFilename_.c_str());
        }
    }
}

proto::BrickInfo BrickEndpoint::createBrickInfo() const {
    proto::BrickInfo brickInfo;
    brickInfo.set_brick_status(brickStatus_);
    brickInfo.set_status_url(TString(brickStatusUrl_));
    brickInfo.set_subscription_mode(subscriptionMode_);
    return brickInfo;
}

int64_t BrickEndpoint::getSteadyNowTimestampSec()
{
    const auto now = std::chrono::steady_clock::now();
    return static_cast<int64_t>(std::chrono::duration_cast<std::chrono::seconds>(now.time_since_epoch()).count());
}

int64_t BrickEndpoint::getSystemNowTimestampSec()
{
    const auto now = std::chrono::system_clock::now();
    return static_cast<int64_t>(std::chrono::duration_cast<std::chrono::seconds>(now.time_since_epoch()).count());
}

const std::string BrickEndpoint::SERVICE_NAME = "brickd";
