#include "bug_report_endpoint.h"

#include <yandex_io/services/pushd/xiva_operations.h>

#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/subprocess/subprocess.h>
#include <yandex_io/protos/quasar_proto.pb.h>

#include <boost/algorithm/string/replace.hpp>
#include <boost/algorithm/string/trim.hpp>

using namespace quasar;

namespace {

    // '////some/program/path', ['first_arg', 'second_arg'] -> 'some_program_path_first_arg_second_arg.log
    std::string prepareNameForBugReport(const std::string& command, const std::vector<std::string>& args) {
        auto rval = fmt::format("{}_{}", command, join(args, "_"));
        std::replace(rval.begin(), rval.end(), '/', '_');
        boost::trim_if(rval, [](char c) { return c == '_'; });
        return rval + ".log";
    }

} // namespace

const std::string BugReportEndpoint::SERVICE_NAME = "bug_report";

BugReportEndpoint::BugReportEndpoint(
    std::shared_ptr<YandexIO::IDevice> device,
    std::shared_ptr<ipc::IIpcFactory> ipcFactory,
    std::shared_ptr<IUserConfigProvider> userConfigProvider)
    : device_(std::move(device))
    , server_(ipcFactory->createIpcServer(SERVICE_NAME))
    , pushdConnector_(ipcFactory->createIpcConnector("pushd"))
    , userConfigProvider_(std::move(userConfigProvider))
    , customCommandLogsConfig_(Json::objectValue)
{
    server_->setMessageHandler([this](const auto& message, ipc::IServer::IClientConnection& connection) {
        handleQuasarMessage(*message, connection);
    });
    server_->listenService();

    pushdConnector_->setMessageHandler([this](const auto& message) {
        handleQuasarMessage(*message);
    });
    pushdConnector_->connectToService();

    userConfigProvider_->jsonChangedSignal(IUserConfigProvider::ConfigScope::SYSTEM, "/bug_report/customCommandLogs").connect([this](const auto& jsonSystemConfig) {
        std::scoped_lock guard(customCommandLogsMutex_);
        customCommandLogsConfig_ = *jsonSystemConfig;
    }, lifetime_);
}

BugReportEndpoint::~BugReportEndpoint()
{
    lifetime_.die();
    server_->shutdown();
    pushdConnector_->shutdown();
}

void BugReportEndpoint::handleQuasarMessage(const proto::QuasarMessage& message, ipc::IServer::IClientConnection& /*connection*/) {
    if (message.has_bug_report_id()) {
        doBugReport(message.bug_report_id());
    }
}

void BugReportEndpoint::handleQuasarMessage(const proto::QuasarMessage& message) {
    if (message.has_push_notification()) {
        if (XivaOperations::SEND_BUGREPORT == message.push_notification().operation()) {
            YIO_LOG_INFO("Received send_bugreport push. Sending bugreport");
            doBugReport("PUSH-" + makeUUID());
        }
    }
}

void BugReportEndpoint::doBugReport(const std::string& bugReportId)
{
    YIO_LOG_INFO("Sending bugreport with id " << bugReportId);
    const auto& config = device_->configuration()->getServiceConfig(SERVICE_NAME);
    const auto& fileLogs = config["fileLogs"];

    for (const auto& log : fileLogs)
    {
        std::string filePath = log["fileName"].asString();
        if (!fileExists(filePath))
        {
            YIO_LOG_WARN("Cannot add file '" + filePath + " to bugreport. File not exists");
            continue;
        }
        std::unordered_map<std::string, std::string> bugReportData;
        bugReportData["id"] = bugReportId;
        bugReportData[getFileName(filePath)] = getFileTail(filePath, log["size"].asInt());
        device_->telemetry()->reportKeyValues("bugReport", bugReportData, true);
    }

    auto commandLogs = config["commandLogs"];
    const auto allowedCustomCommands = jsonToSet<std::string>(config["allowedCustomCommands"]);
    jsonMergeArrays(getFilteredCustomCommands(allowedCustomCommands), commandLogs);

    for (const auto& log : commandLogs) {
        const auto command = tryGetString(log, "command", "");
        const auto commandLineArgs = jsonToVec<std::string>(log["commandLineArgs"]);
        const auto keyName = prepareNameForBugReport(command, commandLineArgs);
        uint64_t tailSizeInBytes = tryGetUInt64(log, "size", 0);

        if (tailSizeInBytes == 0) {
            YIO_LOG_WARN("Invalid size in config for command: " << keyName);
            continue;
        }

        try {
            std::unordered_map<std::string, std::string> bugReportData;
            bugReportData["id"] = bugReportId;
            auto logOutput = getSystemCommandOutputTail(command, commandLineArgs, tailSizeInBytes);
            bugReportData[keyName] = std::move(logOutput.stdOut);
            if (!logOutput.stdErr.empty()) {
                bugReportData["stderr." + keyName] = std::move(logOutput.stdErr);
            }
            device_->telemetry()->reportKeyValues("bugReport", bugReportData, true);
        } catch (const std::runtime_error& e) {
            YIO_LOG_WARN("Can't send " << keyName << " output, error occurred: " << e.what());
        }
    }
}

Json::Value BugReportEndpoint::getFilteredCustomCommands(const std::set<std::string>& allowedCustomCommands) {
    std::scoped_lock guard(customCommandLogsMutex_);
    Json::Value result = Json::arrayValue;

    for (const auto& log : customCommandLogsConfig_) {
        const auto command = tryGetString(log, "command", "");
        if (allowedCustomCommands.contains(command)) {
            result.append(log);
            YIO_LOG_INFO("Custom command " << command << " will be run");
        } else {
            YIO_LOG_INFO(command << " is not allowed custom command");
        }
    }

    return result;
}
