#include "call_connector.h"

#include <yandex_io/protos/quasar_proto.pb.h>

using namespace quasar;

namespace {

    messenger::Status fromProtoStatus(const quasar::proto::CalldSessionState::Status& st) {
        switch (st) {
            case quasar::proto::CalldSessionState::NEW:
                return messenger::Status::NEW;

            case quasar::proto::CalldSessionState::DIALING:
                return messenger::Status::DIALING;

            case quasar::proto::CalldSessionState::RINGING:
                return messenger::Status::RINGING;

            case quasar::proto::CalldSessionState::ACCEPTING:
                return messenger::Status::ACCEPTING;

            case quasar::proto::CalldSessionState::CONNECTING:
                return messenger::Status::CONNECTING;

            case quasar::proto::CalldSessionState::CONNECTED:
                return messenger::Status::CONNECTED;

            case quasar::proto::CalldSessionState::ENDED:
                return messenger::Status::ENDED;

            case quasar::proto::CalldSessionState::NOCALL:
                return messenger::Status::NOCALL;

            default:
                // Never happens
                throw std::runtime_error("unexpected enum value");
        }
    }

    messenger::rtc::Direction fromProtoDirection(const quasar::proto::CalldSessionState::Direction& d) {
        switch (d) {
            case quasar::proto::CalldSessionState::OUTGOING:
                return messenger::rtc::Direction::OUTGOING;

            case quasar::proto::CalldSessionState::INCOMING:
                return messenger::rtc::Direction::INCOMING;

            default:
                // Never happens
                throw std::runtime_error("unexpected enum value");
        }
    }

    messenger::Session::State fromProtoState(const quasar::proto::CalldSessionState& st) {
        return {
            st.authorized(),
            st.connected(),
            fromProtoStatus(st.status()),
            fromProtoDirection(st.direction()),
            st.call_guid(),
            st.user_name(),
            st.user_avatar(),
            st.user_guid(),
            st.callerdeviceid(),
            st.caller_payload(),
            false, // TODO
            false,
        };
    }

} // namespace

CallConnector::CallConnector(std::shared_ptr<ipc::IIpcFactory> ipcFactory)
    : toCalld_(ipcFactory->createIpcConnector("calld"))
{
    toCalld_->setMessageHandler([this](const auto& message) {
        if (message->has_calld_state_changed()) {
            stateObserverList_.notifyObservers(fromProtoState(message->calld_state_changed()));
        }
    });

    toCalld_->connectToService();
    toCalld_->waitUntilConnected();
}

CallConnector::~CallConnector() {
}

messenger::Session::State CallConnector::getState() {
    proto::QuasarMessage request;

    request.mutable_calld_state_request();

    ipc::SharedMessage response;
    try {
        response = toCalld_->sendRequestSync(std::move(request), std::chrono::seconds(2));

    } catch (const std::runtime_error& e) {
        throw std::runtime_error("Can't get calld session state: " + std::string(e.what()));
    }

    if (!response->has_calld_state_response()) {
        throw std::runtime_error("Can't get calld session state: unexpected response " + response->DebugString());
    }

    return fromProtoState(response->calld_state_response());
}

void CallConnector::startCall(const std::string& userGuid, const std::string& callPayload) {
    proto::QuasarMessage message;
    message.mutable_call_action()->mutable_make_call();
    message.mutable_call_action()->set_user_guid(TString(userGuid));
    message.mutable_call_action()->set_payload(TString{callPayload});
    toCalld_->sendMessage(std::move(message));
}

void CallConnector::startCallToOwnDevice(const std::string& deviceId, const std::string& callPayload) {
    proto::QuasarMessage message;
    message.mutable_call_action()->mutable_make_call_to_own_device();
    message.mutable_call_action()->set_device_id(TString(deviceId));
    message.mutable_call_action()->set_payload(TString{callPayload});
    toCalld_->sendMessage(std::move(message));
}

void CallConnector::declineIncomingCall() {
    proto::QuasarMessage message;
    message.mutable_call_action()->mutable_decline_call();
    message.mutable_call_action()->set_call_guid("");
    toCalld_->sendMessage(std::move(message));
}

void CallConnector::acceptIncomingCall() {
    proto::QuasarMessage message;
    message.mutable_call_action()->mutable_accept_call();
    message.mutable_call_action()->set_call_guid("");
    toCalld_->sendMessage(std::move(message));
}

void CallConnector::hangupCall() {
    proto::QuasarMessage message;
    message.mutable_call_action()->mutable_hangup_call();
    message.mutable_call_action()->set_call_guid("");
    toCalld_->sendMessage(std::move(message));
}

void CallConnector::sendHeartbeat() {
    proto::QuasarMessage message;
    message.mutable_messenger_send_heartbeat();
    toCalld_->sendMessage(std::move(message));
}

void CallConnector::setAllowUsualCalls(bool allowUsualCalls) {
    proto::QuasarMessage message;
    message.mutable_calld_runtime_settings()->set_allow_usual_calls(allowUsualCalls);
    toCalld_->sendMessage(std::move(message));
}
