#include "call_endpoint.h"

#include "call_service.h"

#include <yandex_io/callkit/session/session.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>

using namespace quasar;

namespace {

    const auto CALL_HEARTBEAT_PERIOD = std::chrono::seconds(10);

    quasar::proto::CalldSessionState::Status toProtoStatus(messenger::Status st) {
        switch (st) {
            case messenger::Status::NEW:
                return quasar::proto::CalldSessionState::NEW;

            case messenger::Status::DIALING:
                return quasar::proto::CalldSessionState::DIALING;

            case messenger::Status::RINGING:
                return quasar::proto::CalldSessionState::RINGING;

            case messenger::Status::ACCEPTING:
                return quasar::proto::CalldSessionState::ACCEPTING;

            case messenger::Status::CONNECTING:
                return quasar::proto::CalldSessionState::CONNECTING;

            case messenger::Status::CONNECTED:
                return quasar::proto::CalldSessionState::CONNECTED;

            case messenger::Status::ENDED:
                return quasar::proto::CalldSessionState::ENDED;

            case messenger::Status::NOCALL:
                return quasar::proto::CalldSessionState::NOCALL;

            default:
                // Never happens
                throw std::runtime_error("unexpected enum value");
        }
    }

    quasar::proto::CalldSessionState::Direction toProtoDirection(messenger::rtc::Direction d) {
        switch (d) {
            case messenger::rtc::Direction::OUTGOING:
                return quasar::proto::CalldSessionState::OUTGOING;

            case messenger::rtc::Direction::INCOMING:
                return quasar::proto::CalldSessionState::INCOMING;

            default:
                // Never happens
                throw std::runtime_error("unexpected enum value");
        }
    }

    quasar::proto::CallError::Code toProtoError(messenger::CallTransport::ErrorCode ec) {
        switch (ec) {
            case messenger::CallTransport::ErrorCode::BAD_REQUEST:
                return quasar::proto::CallError::BAD_REQUEST;

            case messenger::CallTransport::ErrorCode::CONFLICT:
                return quasar::proto::CallError::CONFLICT;

            case messenger::CallTransport::ErrorCode::BLOCKED_BY_PRIVACY_SETTINGS:
                return quasar::proto::CallError::BLOCKED_BY_PRIVACY_SETTINGS;

            case messenger::CallTransport::ErrorCode::TIMEOUT:
                return quasar::proto::CallError::TIMEOUT;

            case messenger::CallTransport::ErrorCode::UNKNOWN:
                return quasar::proto::CallError::UNKNOWN;

            default:
                // Never happens
                throw std::runtime_error("unexpected enum value");
        }
    }

    void setError(messenger::CallTransport::ErrorCode code, const std::string& message, quasar::proto::CallError* errOut) {
        errOut->set_code(toProtoError(code));
        errOut->set_message(TString(message));
    }

    void setProtoState(const messenger::Session::State& st, quasar::proto::CalldSessionState* stOut) {
        stOut->set_authorized(st.authorized);
        stOut->set_connected(st.connected);
        stOut->set_status(toProtoStatus(st.status));
        stOut->set_direction(toProtoDirection(st.direction));
        stOut->set_call_guid(TString(st.callGuid));
        stOut->set_user_name(TString(st.userName));
        stOut->set_user_avatar(TString(st.userAvatar));
        stOut->set_user_guid(TString(st.userGuid));
    }

    Json::Value prepareCallReportValue(const messenger::Session::State& state) {
        Json::Value event;
        event["call_guid"] = state.callGuid;
        event["direction"] = state.direction == messenger::rtc::Direction::INCOMING ? "incoming" : "outgoing";
        event["is_self_call"] = state.isSelfCall;
        event["status"] = messenger::toString(state.status);
        return event;
    }

} // namespace

CallEndpoint::CallEndpoint(
    std::shared_ptr<YandexIO::IDevice> device,
    std::shared_ptr<ipc::IIpcFactory> ipcFactory,
    std::shared_ptr<IAuthProvider> authProvider,
    const std::string& passportUid,
    const std::string& authToken,
    bool allowUsualCallsInitially,
    const std::vector<std::string>& autoAcceptUsers,
    const std::vector<std::string>& xivaSubscriptions,
    std::shared_ptr<YandexIO::SDKInterface> sdk)
    : device_(std::move(device))
    , authProvider_(std::move(authProvider))
    , sessionProvider_(ipcFactory, device_, "CallService", "CallService", "0.1", passportUid, authToken, xivaSubscriptions)
    , session_(sessionProvider_.createQuasarPushdSession())
    , deviceContext_(std::make_shared<YandexIO::DeviceContext>(ipcFactory, nullptr, false))
    , sdk_(std::move(sdk))
{
    YIO_LOG_INFO("CallService creating");

    session_->setAllowUsualCalls(allowUsualCallsInitially);

    sessionStateSubscription_ = session_->subscribeStateChanged(
        [this, users = autoAcceptUsers](const messenger::Session::State& state) {
            std::unique_lock<std::mutex> g(lock_);
            if (state_.has_value() && *state_ == state) {
                return;
            }
            handleTokenExpired(state);
            handleAutoAccept(state, users);
            handleCallState(state);
            state_.emplace(state);
            ipc::SharedMessage message = createStatusMessageNoLock();
            g.unlock();

            statusChangedSignal_(std::move(message));
        });

    callAcceptedSubscription_ = session_->subscribeCallAccepted([this](const std::string& /*chatId*/) {
        std::lock_guard<std::mutex> g(lock_);

        stopSound();
    });

    callDeclinedSubscription_ = session_->subscribeCallDeclined([this](const std::string& /*chatId*/) {
        std::lock_guard<std::mutex> g(lock_);

        declined_ = true;

        if (!state_.has_value()) {
            YIO_LOG_WARN("Call declined callback called without call state");
            return;
        }

        device_->telemetry()->reportEvent("callDeclined", quasar::jsonToString(prepareCallReportValue(*state_)));
    });

    callFailedSubscription_ = session_->subscribeCallFailed(
        [this](const std::string& /*chatId*/, messenger::CallTransport::ErrorCode errorCode, const std::string message) {
            std::lock_guard<std::mutex> g(lock_);

            lastError_.emplace(errorCode, message);

            if (!state_.has_value()) {
                YIO_LOG_WARN("Call failed callback called without call state");
                return;
            }

            Json::Value event = prepareCallReportValue(*state_);
            event["error_code"] = (int)errorCode;
            event["message"] = message;
            device_->telemetry()->reportEvent("callError", quasar::jsonToString(event));
        });

    callCreationFailedSubscription_ = session_->subscribeCallCreationFailed(
        [this] {
            handleTokenExpired(session_->getState());
        });

    deviceContext_->onAcceptIncomingCall = [this]() {
        std::lock_guard<std::mutex> g(lock_);
        session_->acceptIncomingCall();
    };
    deviceContext_->onDeclineIncomingCall = [this]() {
        std::lock_guard<std::mutex> g(lock_);
        session_->declineIncomingCall();
    };
    deviceContext_->onDeclineCurrentCall = [this]() {
        std::lock_guard<std::mutex> g(lock_);
        session_->hangupCall();
    };

    deviceContext_->connectToSDK();

    worker_.add([this] { heartbeat(); });
}

void CallEndpoint::onSDKState(const YandexIO::SDKState& state) {
    mediaPlaying_ = state.isMediaPlaying();
}

ipc::SharedMessage CallEndpoint::processQuasarMessage(const ipc::Message& message)
{
    std::lock_guard<std::mutex> g(lock_);
    ipc::SharedMessage response;

    if (message.has_call_action()) {
        if (message.call_action().has_accept_call()) {
            session_->acceptIncomingCall();

        } else if (message.call_action().has_decline_call()) {
            session_->declineIncomingCall();

        } else if (message.call_action().has_hangup_call()) {
            session_->hangupCall();

        } else if (message.call_action().has_make_call()) {
            session_->startCall(message.call_action().user_guid(), message.call_action().payload());

        } else if (message.call_action().has_make_call_to_own_device()) {
            session_->startCallToOwnDevice(message.call_action().device_id(), message.call_action().payload());
        }
    }

    if (message.has_messenger_send_heartbeat()) {
        session_->sendHeartbeat();
    }

    if (message.has_calld_state_request()) {
        response = ipc::buildMessage([&](auto& response) {
            response.set_request_id(message.request_id());
            setProtoState(session_->getState(), response.mutable_calld_state_response());
        });
    }

    if (message.has_calld_runtime_settings()) {
        const auto& settings = message.calld_runtime_settings();
        if (settings.has_allow_usual_calls()) {
            session_->setAllowUsualCalls(settings.allow_usual_calls());
        }
    }

    return response;
}

void CallEndpoint::handleTokenExpired(const messenger::Session::State& state) {
    if (state.tokenExpired) {
        authProvider_->requestAuthTokenUpdate("CallEndpoint handleTokenExpired");
    }
}

void CallEndpoint::handleAutoAccept(const messenger::Session::State& state, const std::vector<std::string>& users) {
    if (state.status == messenger::Status::RINGING && state.direction == messenger::rtc::Direction::INCOMING) {
        if (users.size() == 1 && users[0] == "*") {
            YIO_LOG_INFO("Auto accept call from " << state.userGuid);
            session_->acceptIncomingCall();

        } else if (std::find(std::begin(users), std::end(users), state.userGuid) != std::end(users)) {
            YIO_LOG_INFO("Auto accept call from " << state.userGuid);
            session_->acceptIncomingCall();
        }
    }
}

void CallEndpoint::handleCallState(const messenger::Session::State& state) {
    switch (state.status) {
        case messenger::Status::NEW: {
            shouldResumeMedia_ = mediaPlaying_.load();
            declined_ = false;
            connectedOnce_ = false;
            lastError_.reset();
            stopMusicAndDialog();
            break;
        }
        case messenger::Status::DIALING: {
            playSound("calls_checking.mp3", true);
            break;
        }

        case messenger::Status::RINGING: {
            device_->telemetry()->reportEvent("callRinging", quasar::jsonToString(prepareCallReportValue(state)));

            if (state.direction == messenger::rtc::Direction::INCOMING) {
                playSound("calls_ringing_incoming.mp3", true);
                worker_.addDelayed(
                    [this]() {
                        std::lock_guard<std::mutex> g(lock_);

                        if (state_->status == messenger::Status::RINGING) {
                            sayWhoCalls(state_->callerDeviceId, state_->callerPayload);
                        }
                    },
                    std::chrono::seconds(1));

            } else if (state.direction == messenger::rtc::Direction::OUTGOING) {
                playSound("calls_ringing_outgoing.mp3", true);
            }
            break;
        }

        case messenger::Status::ACCEPTING: {
            device_->telemetry()->reportEvent("callAccepted", quasar::jsonToString(prepareCallReportValue(state)));

            shouldResumeMedia_ = false;
            stopSound();
            break;
        }

        case messenger::Status::CONNECTING: {
            shouldResumeMedia_ = false;
            playSound("calls_connecting.mp3", true);
            break;
        }

        case messenger::Status::CONNECTED: {
            connectedOnce_ = true;
            shouldResumeMedia_ = false;
            playSound("calls_connected.mp3");
            break;
        }

        case messenger::Status::ENDED: {
            auto event = prepareCallReportValue(state);
            event["connected_at_least_once"] = connectedOnce_;

            if (state_.has_value()) {
                event["prev_status"] = messenger::toString(state_->status);

                switch (state_->status) {
                    case messenger::Status::DIALING:
                    case messenger::Status::RINGING:
                        stopSound();
                        break;

                    default:
                        break;
                }
            }

            device_->telemetry()->reportEvent("callEnded", quasar::jsonToString(event));

            if (endedWithErrorNoLock()) {
                playSound("calls_fail.mp3");

            } else if (declined_ && state.direction == messenger::rtc::Direction::OUTGOING) {
                playSound("calls_busy.mp3");

            } else if (callAcceptedNoLock()) {
                playSound("calls_ended.mp3");
            }

            break;
        }

        case messenger::Status::NOCALL: {
            if (shouldResumeMedia_) {
                resumeMusic();
            }

            shouldResumeMedia_ = false;
            declined_ = false;
            lastError_.reset();

            break;
        }
    }
}

void CallEndpoint::stopMusicAndDialog() {
    sdk_->getPlaybackControlCapability()->pause();
    sdk_->getAliceCapability()->cancelDialog();
}

void CallEndpoint::resumeMusic() {
    sdk_->getPlaybackControlCapability()->play();
}

void CallEndpoint::playSound(const std::string& name, bool infinite) {
    currentSound_ = name;
    sdk_->getFilePlayerCapability()->playSoundFile(currentSound_,
                                                   proto::AudioChannel::CONTENT_CHANNEL,
                                                   YandexIO::IFilePlayerCapability::PlayParams{
                                                       .playLooped = infinite});
}

void CallEndpoint::stopSound() {
    if (!currentSound_.empty()) {
        sdk_->getFilePlayerCapability()->stopSoundFile(currentSound_);
        currentSound_.clear();
    }
}

namespace {
    Json::Value createTypedSemanticFrame(const std::string& callerDeviceId, const std::string& callerPayload) {
        Json::Value semanticFrame;
        semanticFrame["name"] = "@@mm_semantic_frame";
        semanticFrame["type"] = "server_action";
        semanticFrame["payload"]["typed_semantic_frame"]["get_caller_name"]["caller_device_id"]["string_value"] = callerDeviceId;
        semanticFrame["payload"]["typed_semantic_frame"]["get_caller_name"]["caller_payload"]["string_value"] = callerPayload;
        semanticFrame["utterance"] = "";

        auto& analytics = semanticFrame["payload"]["analytics"];
        analytics["product_scenario"] = "MessengerCall";
        analytics["origin"] = "SmartSpeaker";
        analytics["purpose"] = "say_caller_name";

        return semanticFrame;
    }
} // unnamed namespace

void CallEndpoint::sayWhoCalls(const std::string& callerDeviceId, const std::string& callerPayload) {
    auto request = std::make_shared<YandexIO::VinsRequest>(
        createTypedSemanticFrame(callerDeviceId, callerPayload),
        YandexIO::VinsRequest::createSoftwareDirectiveEventSource());

    sdk_->getAliceCapability()->startRequest(std::move(request), nullptr);
}

CallEndpoint::~CallEndpoint() {
    deviceContext_->shutdown();
    YIO_LOG_INFO("CallService destroying");
}

ipc::SharedMessage CallEndpoint::getStatusMessage()
{
    std::lock_guard<std::mutex> g(lock_);
    return createStatusMessageNoLock();
}

ipc::SharedMessage CallEndpoint::createStatusMessageNoLock() const {
    return ipc::buildMessage([this](auto& msg) {
        if (state_.has_value()) {
            setProtoState(*state_, msg.mutable_calld_state_changed());

        } else {
            msg.mutable_calld_state_changed()->set_status(proto::CalldSessionState::NOCALL);
        }

        if (endedWithErrorNoLock()) {
            setError(lastError_->first, lastError_->second, msg.mutable_calld_state_changed()->mutable_ended_with_error());
        }
    });
}

CallEndpoint::IStatusChangedSignal& CallEndpoint::statusChangedSignal()
{
    return statusChangedSignal_;
}

bool CallEndpoint::endedWithErrorNoLock() const {
    return state_.has_value() && lastError_.has_value() && state_->status == messenger::Status::ENDED && lastError_->first == messenger::CallTransport::ErrorCode::BAD_REQUEST;
}

bool CallEndpoint::callAcceptedNoLock() const {
    return state_.has_value() && (state_->status == messenger::Status::ACCEPTING || state_->status == messenger::Status::CONNECTING || state_->status == messenger::Status::CONNECTED);
}

void CallEndpoint::setAllowUsualCalls(bool allowUsualCalls) {
    session_->setAllowUsualCalls(allowUsualCalls);
}

void CallEndpoint::heartbeat() {
    std::lock_guard<std::mutex> g(lock_);

    if (state_.has_value() && state_->status != messenger::Status::ENDED && state_->status != messenger::Status::NOCALL) {
        device_->telemetry()->reportEvent("callHeartbeat", quasar::jsonToString(prepareCallReportValue(*state_)));
    }

    worker_.addDelayed([this] { heartbeat(); }, CALL_HEARTBEAT_PERIOD);
}
