#include "call_service.h"

#include <yandex_io/callkit/calls/call/call.h>
#include <yandex_io/callkit/session/session.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>

#include <memory>

using namespace quasar;

namespace {

    bool calldEnabled(const Json::Value& serviceConfig) {
        if (!serviceConfig.isNull()) {
            return quasar::tryGetBool(serviceConfig, "enabled", true);
        }

        return true;
    }

    const std::string MESSENGER_ALPHA_ENV = "messenger";
    const std::string MESSENGER_PROD_ENV = "messenger-prod";

} // namespace

const std::string CallService::SERVICE_NAME = "calld";

CallService::CallService(
    std::shared_ptr<YandexIO::IDevice> device,
    std::shared_ptr<ipc::IIpcFactory> ipcFactory,
    std::shared_ptr<IAuthProvider> authProvider,
    std::shared_ptr<IStereoPairProvider> stereoPairProvider,
    std::shared_ptr<YandexIO::SDKInterface> sdk)
    : device_(std::move(device))
    , ipcFactory_(std::move(ipcFactory))
    , authProvider_(std::move(authProvider))
    , stereoPairProvider_(std::move(stereoPairProvider))
    , enabled_(calldEnabled(device_->configuration()->getServiceConfig(CallService::SERVICE_NAME)))
    , stopped_(false)
    , restart_(false)
    , nonSelfCallsAllowed_(false)
    , forceNonSelfCallsAllowed_(false)
    , dndMode_(false)
    , hasMessengerXivaSubscription_(false)
    , toSyncd_(ipcFactory_->createIpcConnector("syncd"))
    , toPushd_(ipcFactory_->createIpcConnector("pushd"))
    , toDoNotDisturb_(ipcFactory_->createIpcConnector("do_not_disturb"))
    , server_(ipcFactory_->createIpcServer(SERVICE_NAME))
    , authInfo_(authProvider_->ownerAuthInfo().value())
    , sdk_(std::move(sdk))
{
    stereoPairProvider_->stereoPairState().connect(
        [this](const auto& state) {
            isStereoPairFollower_ = state->isFollower();
        }, lifetime_);
    toSyncd_->setMessageHandler([this](const auto& message) {
        if (message->has_user_config_update() && message->user_config_update().has_config()) {
            std::lock_guard<std::mutex> g(lock_);

            const auto serviceConfig = parseJson(message->user_config_update().config())["system_config"][CallService::SERVICE_NAME];

            if (!serviceConfig.isNull() && !serviceConfig["enabled"].isNull()) {
                enabled_ = tryGetBool(serviceConfig, "enabled", true);

            } else {
                enabled_ = calldEnabled(device_->configuration()->getServiceConfig(CallService::SERVICE_NAME));
            }

            std::string autoAcceptValue;

            if (!serviceConfig.isNull() && !serviceConfig["autoAcceptUsers"].isNull()) {
                autoAcceptValue = tryGetString(serviceConfig, "autoAcceptUsers", "");

            } else {
                autoAcceptValue = tryGetString(device_->configuration()->getServiceConfig(CallService::SERVICE_NAME), "autoAcceptUsers", "");
            }

            if (!autoAcceptValue.empty()) {
                autoAcceptUsers_ = split(autoAcceptValue);

            } else {
                autoAcceptUsers_.clear();
            }

            const auto deviceConfig = parseJson(message->user_config_update().config())["device_config"];

            nonSelfCallsAllowed_ = quasar::tryGetBool(deviceConfig, "allow_non_self_calls", false);

            if (!serviceConfig.isNull() && !serviceConfig["forceAllowNonSelfCalls"].isNull()) {
                forceNonSelfCallsAllowed_ = tryGetBool(serviceConfig, "forceAllowNonSelfCalls", false);

            } else {
                forceNonSelfCallsAllowed_ = false;
            }

            if (endpoint_) {
                endpoint_->setAllowUsualCalls(getNonSelfCallsAllowedNoLock());
            }

            cv_.notify_one();
        }
    });

    toPushd_->setMessageHandler([this](const auto& message) {
        if (message->has_xiva_subscriptions()) {
            std::lock_guard<std::mutex> g(lock_);

            std::vector<std::string> xivaSubscriptions(
                message->xiva_subscriptions().xiva_subscriptions().cbegin(),
                message->xiva_subscriptions().xiva_subscriptions().cend());

            if (xivaSubscriptions != xivaSubscriptions_) {
                YIO_LOG_INFO("Calld restart triggered from pushd");

                restart_ = true;

                xivaSubscriptions_ = std::move(xivaSubscriptions);

                hasMessengerXivaSubscription_ = std::any_of(
                    xivaSubscriptions_.cbegin(),
                    xivaSubscriptions_.cend(),
                    [](const auto& s) { return s == MESSENGER_ALPHA_ENV || s == MESSENGER_PROD_ENV; });

                cv_.notify_one();

            } else {
                YIO_LOG_INFO("Calld received same pushd services: " << join(xivaSubscriptions_, ", "));
            }
        }
    });

    toDoNotDisturb_->setMessageHandler([this](const auto& message) {
        if (message->has_do_not_disturb_event() && message->do_not_disturb_event().has_is_dnd_enabled()) {
            std::lock_guard<std::mutex> g(lock_);

            YIO_LOG_DEBUG("Set DND mode: " << message->do_not_disturb_event().is_dnd_enabled());

            dndMode_ = message->do_not_disturb_event().is_dnd_enabled();

            if (endpoint_) {
                endpoint_->setAllowUsualCalls(getNonSelfCallsAllowedNoLock());
            }
        }
    });

    authProvider_->ownerAuthInfo().connect(
        [this](const auto& authInfo) {
            std::lock_guard<std::mutex> g(lock_);
            if (!authInfo_->isSameAuth(*authInfo)) {
                authInfo_ = authInfo;
                YIO_LOG_INFO("Calld restart triggered from AuthProvider");
                restart_ = true;
                cv_.notify_one();
            } else {
                authInfo_ = authInfo;
                YIO_LOG_INFO("Calld received same uid=" << authInfo_->passportUid << ", token=" << maskToken(authInfo_->authToken));
            }
        }, lifetime_);

    toSyncd_->connectToService();
    toPushd_->connectToService();
    toDoNotDisturb_->connectToService();

    server_->setClientConnectedHandler(makeSafeCallback(
        [this](auto& connection) {
            std::lock_guard g(lock_);
            connection.send(getCallStatusUnsafe());
            if (endpoint_) {
                connection.send(endpoint_->getStatusMessage());
            }
        }, lifetime_));

    server_->setMessageHandler(makeSafeCallback(
        [this](const auto& message, auto& connection) {
            std::lock_guard g(lock_);
            if (endpoint_) {
                if (auto response = endpoint_->processQuasarMessage(*message)) {
                    connection.send(response);
                }
            }
        }, lifetime_));
    server_->listenService();
}

CallService::~CallService()
{
    lifetime_.die();

    {
        std::lock_guard<std::mutex> g(lock_);
        stopped_ = true;
    }

    cv_.notify_one();

    if (worker_.joinable()) {
        worker_.join();
    }
}

std::string CallService::getServiceName() const {
    return SERVICE_NAME;
}

void CallService::start() {
    worker_ = std::thread(&CallService::run, this);
}

void CallService::run() {
    std::unique_lock g(lock_);

    while (!stopped_) {
        const bool needToStart = enabled_ && !isStereoPairFollower_ && authInfo_->isAuthorized() && hasMessengerXivaSubscription_;
        YIO_LOG_INFO(
            "Calld control thread wake up: "
            << "stopped_="
            << stopped_
            << ", restart_="
            << restart_
            << ", enabled_="
            << enabled_
            << ", isStereoPairFollower="
            << (isStereoPairFollower_.load())
            << ", needToStart="
            << needToStart
            << ", passportUid_="
            << authInfo_->passportUid
            << ", authToken_="
            << maskToken(authInfo_->authToken)
            << ", xivaSubscriptions="
            << join(xivaSubscriptions_, ", "));

        restart_ = false;
        if (needToStart) {
            // explicitly remove old endpoint because it owns IPC server and listens calld port
            endpoint_.reset();
            endpoint_ = std::make_shared<CallEndpoint>(
                device_,
                ipcFactory_,
                authProvider_,
                authInfo_->passportUid,
                authInfo_->authToken,
                getNonSelfCallsAllowedNoLock(),
                autoAcceptUsers_,
                xivaSubscriptions_,
                sdk_);
            sdk_->addSDKStateObserver(endpoint_);
            endpoint_->statusChangedSignal().connect(
                [this](const auto& message) {
                    server_->sendToAll(message);
                }, lifetime_);
            server_->sendToAll(endpoint_->getStatusMessage());
        } else {
            endpoint_.reset();
        }

        cv_.wait(g,
                 [this, needToStart]() {
                     return restart_ ||
                            stopped_ ||
                            (needToStart != (enabled_ && !isStereoPairFollower_ && authInfo_->isAuthorized() && hasMessengerXivaSubscription_));
                 });
    }
}

bool CallService::getNonSelfCallsAllowedNoLock() const {
    return forceNonSelfCallsAllowed_ || (nonSelfCallsAllowed_ && !dndMode_);
}

ipc::SharedMessage CallService::getCallStatusUnsafe()
{
    proto::QuasarMessage message;
    auto& state = *message.mutable_call_message()->mutable_state();
    state.set_is_enabled(enabled_);
    state.set_is_running(endpoint_ && !stopped_);
    return ipc::SharedMessage(std::move(message));
}
