#include "do_not_disturb_endpoint.h"

#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/protos/quasar_proto.pb.h>

YIO_DEFINE_LOG_MODULE("do_not_disturb");

using namespace quasar;
using namespace proto;

DoNotDisturbEndpoint::DoNotDisturbEndpoint(std::shared_ptr<YandexIO::IDevice> device, std::shared_ptr<ipc::IIpcFactory> ipcFactory)
    : device_(std::move(device))
    , server_(ipcFactory->createIpcServer(SERVICE_NAME))
    , syncdConnector_(ipcFactory->createIpcConnector("syncd"))
    , ntpdConnector_(ipcFactory->createIpcConnector("ntpd"))
    , deviceContext_(std::make_unique<YandexIO::DeviceContext>(ipcFactory, nullptr, false))
{
    server_->setMessageHandler([this](const auto& message, auto& /*connection*/) { handleQuasarMessage(*message); });
    server_->setClientConnectedHandler([this](auto& connection) {
        dndStatusUpdateQueue_.add([this, connection = connection.share()]() {
            connection->send(createDndMessage(doNotDisturb_));
        });
    });
    server_->listenService();

    syncdConnector_->setMessageHandler([this](const auto& message) {
        handleConfigQuasarMessage(*message);
    });
    syncdConnector_->connectToService();

    ntpdConnector_->setMessageHandler([this](const auto& message) {
        handleQuasarMessage(*message);
    });

    deviceContext_->onTimezone = [this](const proto::Timezone& timezone) {
        handleTimezone(timezone);
    };

    deviceContext_->connectToSDK();

    ntpdConnector_->connectToService();
}

DoNotDisturbEndpoint::~DoNotDisturbEndpoint() {
    server_->shutdown();
    syncdConnector_->shutdown();
    ntpdConnector_->shutdown();
}

void DoNotDisturbEndpoint::handleTimezone(const proto::Timezone& timezone) {
    if (!timezone.has_timezone_offset_sec()) {
        return;
    }

    int newTimezoneOffsetSec = timezone.timezone_offset_sec();
    std::lock_guard<std::mutex> lockGuard(mutex_);
    if (newTimezoneOffsetSec == timezoneOffsetSec_ && isTimezoneDetected_) {
        return;
    }

    timezoneOffsetSec_ = newTimezoneOffsetSec;
    isTimezoneDetected_ = true;
    if (isSyncTimeSuccessful_) {
        updateDndStatus();
        dndStatusUpdateLifetime_.die();
        planPendingDndStatus();
    }
}

void DoNotDisturbEndpoint::handleQuasarMessage(const QuasarMessage& message) {
    if (!message.has_ntp_sync_event()) {
        return;
    }

    const auto& ntpSyncEvent = message.ntp_sync_event();
    if (!ntpSyncEvent.has_is_ntp_sync_successful()) {
        return;
    }
    bool isNtpSyncSuccessful = ntpSyncEvent.is_ntp_sync_successful();

    // has not changes
    std::lock_guard<std::mutex> lockGuard(mutex_);
    if (isSyncTimeSuccessful_ == isNtpSyncSuccessful) {
        return;
    }

    isSyncTimeSuccessful_ = isNtpSyncSuccessful;

    if (isNtpSyncSuccessful && isTimezoneDetected_) {
        updateDndStatus();
        dndStatusUpdateLifetime_.die();
        planPendingDndStatus();
    } else {
        dndStatusUpdateLifetime_.die();
    }
}

void DoNotDisturbEndpoint::updateDndStatus() {
    if (!startDoNotDisturbTime_ || !endDoNotDisturbTime_) {
        doNotDisturb_ = true;
        server_->sendToAll(createDndMessage(doNotDisturb_));
        return;
    }

    TimeInfo currentTime = getCurrentTimeInfo();
    doNotDisturb_ = currentTime.insideInterval(*startDoNotDisturbTime_, *endDoNotDisturbTime_);
    server_->sendToAll(createDndMessage(doNotDisturb_));
}

void DoNotDisturbEndpoint::planPendingDndStatus() {
    if (!startDoNotDisturbTime_ || !endDoNotDisturbTime_) {
        doNotDisturb_ = true;
        return;
    }

    TimeInfo currentTime = getCurrentTimeInfo();

    TimeInfo nextTime = getNextChangeDndTime(startDoNotDisturbTime_.value(), endDoNotDisturbTime_.value());
    int timeToNextChangeSeconds = currentTime.calculateNextTimeDeltaSeconds(nextTime);
    YIO_LOG_INFO("next dnd state change at: " + nextTime.to_string() << " after " << timeToNextChangeSeconds
                                                                     << " sec, now dnd enabled: " << (doNotDisturb_ ? "true" : "false"));

    dndStatusUpdateQueue_.addDelayed(
        [this]() {
            updateDndStatus();
            planPendingDndStatus();
        },
        std::chrono::seconds(timeToNextChangeSeconds),
        dndStatusUpdateLifetime_);
}

void DoNotDisturbEndpoint::sendDndTimesUpdateEvent(const std::string& oldStartTime,
                                                   const std::string& oldEndTime,
                                                   const std::string& newStartTime,
                                                   const std::string& newEndTime) {
    Json::Value eventBody;
    eventBody["current_start_time"] = oldStartTime;
    eventBody["current_end_time"] = oldEndTime;
    eventBody["new_start_time"] = newStartTime;
    eventBody["new_end_time"] = newEndTime;
    eventBody["use_default"] = useDefaultDndTime_;

    device_->telemetry()->reportEvent("DndTimesChanged", jsonToString(eventBody));
}

proto::QuasarMessage DoNotDisturbEndpoint::createDndMessage(const bool is_dnd_enabled) {
    QuasarMessage dndMsg;
    dndMsg.mutable_do_not_disturb_event()->set_is_dnd_enabled(is_dnd_enabled);

    return dndMsg;
}

TimeInfo DoNotDisturbEndpoint::getNextChangeDndTime(const TimeInfo& startTime, const TimeInfo& endTime) const {
    TimeInfo currentTime = getCurrentTimeInfo();

    if (startTime < endTime) {
        if (currentTime >= startTime && currentTime < endTime) {
            return endTime;
        }

        return startTime;
    }

    if (currentTime >= startTime || currentTime < endTime) {
        return endTime;
    }

    return startTime;
}

TimeInfo DoNotDisturbEndpoint::getCurrentTimeInfo() const {
    /* Station and Mini have different timezone settings.
     * So we use gmtime here (which doesn't rely on any timezone system settings) and set timezone explicitly. */
    const auto nowLocal = std::chrono::system_clock::now() + std::chrono::seconds(timezoneOffsetSec_);
    const std::time_t nowTimeT = std::chrono::system_clock::to_time_t(nowLocal);

    const std::tm calendar_time = *std::gmtime(std::addressof(nowTimeT));
    return TimeInfo(calendar_time.tm_hour, calendar_time.tm_min, calendar_time.tm_sec, isTimezoneDetected_, timezoneOffsetSec_);
}

void DoNotDisturbEndpoint::handleConfigQuasarMessage(const QuasarMessage& message) {
    if (message.has_user_config_update()) {
        const std::string& config = message.user_config_update().config();
        YIO_LOG_DEBUG("Handle config: " << config);
        if (!config.empty()) {
            const Json::Value systemConfig = parseJson(config)["system_config"];
            const Json::Value deviceConfig = parseJson(config)["device_config"];

            std::optional<TimeInfo> newStartDoNotDisturbTime = TimeInfo::parse(DND_START_TIME_DEFAULT);
            std::optional<TimeInfo> newEndDoNotDisturbTime = TimeInfo::parse(DND_END_TIME_DEFAULT);

            if (deviceConfig.isMember("dndMode")) {
                const Json::Value& dndModeConfig = deviceConfig["dndMode"];
                const bool dndModeEnabled = tryGetBool(dndModeConfig, "enabled", false);
                if (dndModeEnabled && parseUserDndModeConfig(dndModeConfig, newStartDoNotDisturbTime, newEndDoNotDisturbTime)) {
                    checkUpdateIfNeed(newStartDoNotDisturbTime, newEndDoNotDisturbTime, false);
                    return;
                }
            }

            if (systemConfig.isMember("dnd_prefs")) {
                const Json::Value& dndPrefsConfig = systemConfig["dnd_prefs"];
                if (parseQuasmodromDndConfig(dndPrefsConfig, newStartDoNotDisturbTime, newEndDoNotDisturbTime)) {
                    checkUpdateIfNeed(newStartDoNotDisturbTime, newEndDoNotDisturbTime, false);
                    return;
                }
            }
            std::unique_lock lock(mutex_);
            if (isTimezoneDetected_) {
                // set default timezone
                newStartDoNotDisturbTime->hasTimezone_ = true;
                newStartDoNotDisturbTime->timezoneOffsetSec_ = timezoneOffsetSec_;
                newEndDoNotDisturbTime->hasTimezone_ = true;
                newEndDoNotDisturbTime->timezoneOffsetSec_ = timezoneOffsetSec_;
            }
            lock.unlock();
            checkUpdateIfNeed(newStartDoNotDisturbTime, newEndDoNotDisturbTime, true);
        }
    }
}

bool DoNotDisturbEndpoint::parseUserDndModeConfig(const Json::Value& dndModeConfig,
                                                  std::optional<TimeInfo>& startTime,
                                                  std::optional<TimeInfo>& endTime) {
    std::optional<TimeInfo> currentStartTime;
    std::optional<TimeInfo> currentEndTime;

    try {
        const std::string startDoNotDisturbTimeStr = getString(dndModeConfig, "starts");
        currentStartTime = TimeInfo::parse(startDoNotDisturbTimeStr);
    } catch (const std::exception& e) {
        YIO_LOG_ERROR_EVENT("DoNotDisturbEndpoint.FailedParseTime.User.Starts", "dndMode starts parse error: " << e.what());
        return false;
    }

    try {
        const std::string endDoNotDisturbTimeStr = getString(dndModeConfig, "ends");
        currentEndTime = TimeInfo::parse(endDoNotDisturbTimeStr);
    } catch (const std::exception& e) {
        YIO_LOG_ERROR_EVENT("DoNotDisturbEndpoint.FailedParseTime.User.Ends", "dndMode ends parse error: " << e.what());
        return false;
    }

    startTime = currentStartTime;
    endTime = currentEndTime;

    return true;
}

bool DoNotDisturbEndpoint::parseQuasmodromDndConfig(const Json::Value& dndPrefsConfig,
                                                    std::optional<TimeInfo>& startTime,
                                                    std::optional<TimeInfo>& endTime) {
    std::optional<TimeInfo> currentStartTime;
    std::optional<TimeInfo> currentEndTime;

    try {
        const std::string startDoNotDisturbTimeStr = getString(dndPrefsConfig, "start_time");
        currentStartTime = TimeInfo::parse(startDoNotDisturbTimeStr);
    } catch (const std::exception& e) {
        YIO_LOG_ERROR_EVENT("DoNotDisturbEndpoint.FailedParseTime.Global.Starts", "dndPrefs start_time parse error: " << e.what());
        return false;
    }

    try {
        const std::string endDoNotDisturbTimeStr = getString(dndPrefsConfig, "end_time");
        currentEndTime = TimeInfo::parse(endDoNotDisturbTimeStr);
    } catch (const std::exception& e) {
        YIO_LOG_ERROR_EVENT("DoNotDisturbEndpoint.FailedParseTime.Global.Ends", "dndPrefs end_time parse error: " << e.what());
        return false;
    }

    startTime = currentStartTime;
    endTime = currentEndTime;

    return true;
}

void DoNotDisturbEndpoint::checkUpdateIfNeed(std::optional<TimeInfo>& startTime, std::optional<TimeInfo>& endTime, bool useDefaultDndTime) {
    if (!startTime || !endTime) {
        YIO_LOG_ERROR_EVENT("DoNotDisturbEndpoint.checkUpdateIfNeed", "startTime or endTime is empty");
        return;
    }

    // if hasn't changes
    bool timeChanged = !startDoNotDisturbTime_ || !endDoNotDisturbTime_ || *startDoNotDisturbTime_ != *startTime || *endDoNotDisturbTime_ != *endTime;
    bool defaultDndTimeChanged = useDefaultDndTime_ != useDefaultDndTime;

    std::lock_guard<std::mutex> lockGuard(mutex_);
    useDefaultDndTime_ = useDefaultDndTime;
    // We don't want to react to default time setting after start
    if (timeChanged || defaultDndTimeChanged) {
        std::string currentStartDoNotDisturbTimeStr = (!startDoNotDisturbTime_ ? "null" : (*startDoNotDisturbTime_).to_string());
        std::string currentEndDoNotDisturbTimeStr = (!endDoNotDisturbTime_ ? "null" : (*endDoNotDisturbTime_).to_string());

        sendDndTimesUpdateEvent(currentStartDoNotDisturbTimeStr, currentEndDoNotDisturbTimeStr,
                                (*startTime).to_string(), (*endTime).to_string());
    }

    if (!timeChanged) {
        return;
    }

    startDoNotDisturbTime_ = startTime;
    endDoNotDisturbTime_ = endTime;

    updateDndStatus();
    dndStatusUpdateLifetime_.die();
    planPendingDndStatus();
}
