#include "time_info.h"

#include <yandex_io/libs/logging/logging.h>

#include <iomanip>
#include <ios>
#include <sstream>
#include <stdexcept>

YIO_DEFINE_LOG_MODULE("do_not_disturb");

using namespace quasar;

TimeInfo::TimeInfo()
    : TimeInfo(0, 0, 0)
{
}

TimeInfo::TimeInfo(int hours, int minutes, int seconds, bool hasTimezone, int timezoneOffsetSec)
    : hours_(hours)
    , minutes_(minutes)
    , seconds_(seconds)
    , hasTimezone_(hasTimezone)
    , timezoneOffsetSec_(timezoneOffsetSec)
{
}

int TimeInfo::calculateNextTimeDeltaSeconds(const TimeInfo& other) const {
    if (*this < other) {
        return other.getDeltaSeconds(*this);
    }

    int totalSeconds = getDeltaSeconds(other);

    // invert result
    return PER_DAY_SECONDS - totalSeconds;
}

bool TimeInfo::operator<(const TimeInfo& other) const {
    return getDeltaSeconds(other) < 0;
}

bool TimeInfo::operator>(const TimeInfo& other) const {
    return getDeltaSeconds(other) > 0;
}

bool TimeInfo::operator==(const TimeInfo& other) const {
    return getDeltaSeconds(other) == 0;
}

bool TimeInfo::operator!=(const TimeInfo& other) const {
    return getDeltaSeconds(other) != 0;
}

bool TimeInfo::operator>=(const TimeInfo& other) const {
    return *this > other || *this == other;
}

bool TimeInfo::insideInterval(const TimeInfo& leftBorder, const TimeInfo& rightBorder) const {
    if (rightBorder < leftBorder) {
        return *this < rightBorder || *this >= leftBorder;
    }

    return *this >= leftBorder && *this < rightBorder;
}

std::string TimeInfo::to_string() const {
    std::stringstream ss;
    ss << *this;
    return ss.str();
}

int TimeInfo::getDeltaSeconds(const TimeInfo& other) const {
    int currentTotalSeconds = getTotalSeconds();
    int otherTotalSeconds = other.getTotalSeconds();

    if (hasTimezone_ && other.hasTimezone_) {
        currentTotalSeconds -= timezoneOffsetSec_;
        if (currentTotalSeconds < 0) {
            currentTotalSeconds += PER_DAY_SECONDS;
        }

        if (currentTotalSeconds >= PER_DAY_SECONDS) {
            currentTotalSeconds %= PER_DAY_SECONDS;
        }

        otherTotalSeconds -= other.timezoneOffsetSec_;
        if (otherTotalSeconds < 0) {
            otherTotalSeconds += PER_DAY_SECONDS;
        }

        if (otherTotalSeconds >= PER_DAY_SECONDS) {
            otherTotalSeconds %= PER_DAY_SECONDS;
        }
    }

    return currentTotalSeconds - otherTotalSeconds;
}

int TimeInfo::getTotalSeconds() const {
    return hours_ * 60 * 60 + minutes_ * 60 + seconds_;
}

std::ostream& operator<<(std::ostream& os, const TimeInfo& timeInfo) {
    os << std::setfill('0')
       << std::setw(2) << timeInfo.hours_ << ":"
       << std::setw(2) << timeInfo.minutes_ << ":"
       << std::setw(2) << timeInfo.seconds_;
    if (timeInfo.hasTimezone_) {
        os << (timeInfo.timezoneOffsetSec_ >= 0 ? "+" : "-");
        int currentTimezoneOffsetSec = abs(timeInfo.timezoneOffsetSec_);
        int timezoneHours = currentTimezoneOffsetSec / (60 * 60);
        int timezoneMinutes = currentTimezoneOffsetSec / 60 - timezoneHours * 60;

        os << std::setfill('0')
           << std::setw(2) << timezoneHours << ":"
           << std::setw(2) << timezoneMinutes;
    }

    return os;
}

void tryReadChar(std::istream& is, char expectedNextChar) {
    if (is.peek() == expectedNextChar) {
        char throwAway;
        is >> throwAway;
    } else {
        is.setstate(std::ios::failbit);
    }
}

int tryReadNumber(std::istream& is, int length) {
    if (length <= 0) {
        throw std::logic_error("length should be greater than zero");
    }

    char currentChar;
    std::vector<char> numberChars;
    numberChars.resize(length);

    for (int i = 0; i < length && is.get(currentChar); ++i) {
        if (currentChar == ':') {
            break;
        }
        numberChars[i] = currentChar;
    }
    std::string str(numberChars.begin(), numberChars.end());

    return std::stoi(str);
}

std::istream& operator>>(std::istream& is, quasar::TimeInfo& timeInfo) {
    int hours;
    int minutes;
    int seconds;
    int timezoneHours;
    int timezoneOffsetSec;
    int timezoneMinutes = 0;
    bool hasTimezone = false;
    is.clear();

    is >> hours;
    if (is.fail()) {
        throw std::logic_error("Cannot read hours");
    }

    tryReadChar(is, ':');
    if (is.fail()) {
        throw std::logic_error("Expected ':' char as separator");
    }

    is >> minutes;
    if (is.fail()) {
        throw std::logic_error("Cannot read minutes");
    }

    tryReadChar(is, ':');

    if (is.fail()) {
        is.clear();
        seconds = 0;
    } else {
        is >> seconds;
        if (is.fail()) {
            throw std::logic_error("Cannot read seconds");
        }
    }

    char sign;
    if (is.get(sign) && (sign == '+' || sign == '-')) {
        hasTimezone = true;

        timezoneHours = tryReadNumber(is, 2);
        if (is.fail()) {
            throw std::logic_error("Cannot read timezone hours");
        }

        if (is.peek() == ':') {
            char throwAway;
            is >> throwAway;
        }

        if (!is.eof()) {
            is >> timezoneMinutes;
            if (is.fail()) {
                throw std::logic_error("Cannot read timezone minutes");
            }
        }
    }

    if (hours < 0 || hours > 24) {
        throw std::logic_error("hours should be in range [0,24]");
    }
    if (minutes < 0 || minutes > 59) {
        throw std::logic_error("minutes should be in range [0,59]");
    }
    if (seconds < 0 || seconds > 59) {
        throw std::logic_error("seconds should be in range [0,59]");
    }
    if (hasTimezone) {
        if (timezoneHours < -12 || timezoneHours > 14) {
            // most western timezone is -12, most eastern timezone is 14
            throw std::logic_error("timezone hours should be in range [-12,14]");
        }
        if (timezoneMinutes < 0 || timezoneMinutes > 59) {
            throw std::logic_error("timezone minutes should be in range [0,59]");
        }
    }

    if (hasTimezone) {
        timezoneOffsetSec = timezoneHours * 60 * 60;
        if (timezoneMinutes) {
            timezoneOffsetSec += timezoneMinutes * 60;
        }

        timezoneOffsetSec *= (sign == '+') ? 1 : -1;

        timeInfo.timezoneOffsetSec_ = timezoneOffsetSec;
        timeInfo.hasTimezone_ = true;
    }

    timeInfo.hours_ = hours;
    timeInfo.minutes_ = minutes;
    timeInfo.seconds_ = seconds;

    return is;
}

std::optional<TimeInfo> TimeInfo::parse(const std::string& timeStr) {
    std::istringstream input(timeStr);
    try {
        TimeInfo timeInfo;
        input >> timeInfo;
        return timeInfo;
    } catch (std::exception& e) {
        YIO_LOG_ERROR_EVENT("TimeInfo.FailedParse", "TimeInfo::parse failed on input: " << timeStr << ", reason: " << e.what());
        return std::nullopt;
    }
}
