#include "check_token.h"

#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/configuration/configuration.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/jwt/jwt.h>
#include <yandex_io/libs/logging/logging.h>

#include <memory>
#include <mutex>

YIO_DEFINE_LOG_MODULE("firstrun");

using namespace quasar;
using namespace quasar::check_token;

CheckToken::CheckToken(
    std::shared_ptr<YandexIO::IDevice> device,
    std::shared_ptr<IAuthProvider> authProvider,
    std::shared_ptr<IDeviceStateProvider> deviceStateProvider,
    std::chrono::milliseconds timeout)
    : device_(std::move(device))
    , authProvider_(std::move(authProvider))
    , deviceStateProvider_(std::move(deviceStateProvider))
    , backendUrl_(device_->configuration()->getServiceConfig("common")["backendUrl"].asString())
    , authInfo_(authProvider_->ownerAuthInfo().value())
    , deviceState_(deviceStateProvider_->deviceState().value())
    , backendClient_("jwt-quasar-backend", device_)

{
    backendClient_.setTimeout(timeout);
    authProvider_->ownerAuthInfo().connect(
        [this](const auto& newAuthInfo) {
            std::lock_guard<std::mutex> lock(mutex_);
            if (newAuthInfo->authToken != authInfo_->authToken) {
                authInfo_ = newAuthInfo;
                condVar_.notify_all();
            }
        }, lifetime_);
    deviceStateProvider_->deviceState().connect(
        [this](const auto& newDeviceState) {
            std::lock_guard<std::mutex> lock(mutex_);
            deviceState_ = newDeviceState;
            condVar_.notify_all();
        }, lifetime_);
}

HttpClient::Headers CheckToken::getHeaders() {
    return HttpClient::Headers{
        {"Authorization", "OAuth " + authInfo_->authToken},
        {"Content-Type", "application/json"},
    };
}

bool CheckToken::check(const std::string& token) {
    try {
        const auto parsedToken = decodeJWT(token);
        const std::string tokenDeviceId = getStringGrantFromJWT(parsedToken.get(), "sub");
        const std::string actualDeviceId = device_->deviceId();
        if (tokenDeviceId != actualDeviceId) {
            YIO_LOG_ERROR_EVENT("CheckToken.BadToken.DeviceIdMismatch", "Token device id (" + tokenDeviceId + ") doesn't match actual device id (" + actualDeviceId + ")");
            return false;
        }

        const unsigned long expiresSec = jwt_get_grant_int(parsedToken.get(), "exp");
        const unsigned long nowSec = time(nullptr);

        if (expiresSec < nowSec) {
            YIO_LOG_ERROR_EVENT("CheckToken.BadToken.Expired", "JWT token has already expired: now=" + std::to_string(nowSec) + " expires=" + std::to_string(expiresSec));
            return false;
        }
    } catch (const std::exception& e) {
        YIO_LOG_ERROR_EVENT("CheckToken.FailedParseToken", "Failed to parse JWT token: " << e.what());
        return false;
    }

    std::unique_lock<std::mutex> lock(mutex_);
    condVar_.wait(lock, [this]() { return deviceState_->configuration == DeviceState::Configuration::CONFIGURED; });
    const HttpClient::HttpResponse response = backendClient_.post("check-token", backendUrl_ + "/check_token", token, getHeaders());
    YIO_LOG_TRACE("check_token response is ready");

    const auto jsonBody = checkAndParseResponse(response);
    return jsonBody["valid"].asBool();
}

Json::Value CheckToken::checkAndParseResponse(const HttpClient::HttpResponse& response) {
    YIO_LOG_DEBUG("check_token response" << response.body);
    if (isSuccessHttpCode(response.responseCode)) {
        try {
            YIO_LOG_TRACE("check_token response: " + response.body);
            auto registerBody = parseJson(response.body);
            if ("ok" != registerBody["status"].asString()) {
                throw Exception("Non-ok response status: " + registerBody["status"].asString());
            }
            return registerBody;
        } catch (const Json::Exception& exception) {
            throw Exception("Json parsing error: " + std::string(exception.what()));
        }
    }
    throw Non200ResponseCodeException(response.responseCode);
}

CheckToken::~CheckToken() {
    lifetime_.die();
    condVar_.notify_all();
}
