#include "first_run_endpoint.h"
#include "yandex_io/interfaces/auth/i_auth_provider.h"

#include <yandex_io/services/pushd/xiva_operations.h>

#include <yandex_io/libs/base/crc32.h>
#include <yandex_io/libs/base/persistent_file.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/jwt/jwt.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/protobuf_utils/debug.h>
#include <yandex_io/libs/threading/steady_condition_variable.h>
#include <yandex_io/protos/model_objects.pb.h>
#include <yandex_io/protos/quasar_proto.pb.h>
#include <yandex_io/protos/enum_names/enum_names.h>

#include <library/cpp/http/misc/parsed_request.h>
#include <library/cpp/http/server/http.h>
#include <library/cpp/http/server/response.h>

#include <fstream>
#include <memory>
#include <optional>
#include <string>

YIO_DEFINE_LOG_MODULE("firstrun");

using namespace quasar;

namespace {

    constexpr const char* FORCE_WIFI_RECONFIGURE = "forceWifiReconfigure";
    bool getForceWifiReconigure(YandexIO::IDevice& device) {
        return tryGetBool(device.configuration()->getServiceConfig("common"), FORCE_WIFI_RECONFIGURE, false);
    };
} // namespace

const std::string FirstRunEndpoint::SERVICE_NAME = "firstrund";
class FirstRunEndpoint::FirstRunHttpServer: public THttpServer {
public:
    using THttpServer::THttpServer;
};

class FirstRunEndpoint::FirstRunHttpCallback: public THttpServer::ICallBack {
public:
    FirstRunHttpCallback(FirstRunEndpoint* endpoint);
    TClientRequest* CreateClient() override;

private:
    FirstRunEndpoint* endpoint_;
};

class FirstRunEndpoint::FirstRunHttpRequest: public TRequestReplier {
public:
    FirstRunHttpRequest(FirstRunEndpoint* endpoint);
    bool DoReply(const TReplyParams& params) override;

private:
    FirstRunEndpoint* endpoint_;
};

FirstRunEndpoint::FirstRunEndpoint(
    std::shared_ptr<YandexIO::IDevice> device,
    std::shared_ptr<ipc::IIpcFactory> ipcFactory,
    std::shared_ptr<IAuthProvider> authProvider,
    std::shared_ptr<IDeviceStateProvider> deviceStateProvider,
    std::shared_ptr<IUpdatesProvider> updatesProvider,
    std::shared_ptr<IUserConfigProvider> userConfigProvider,
    std::shared_ptr<YandexIO::IFilePlayerCapability> filePlayerCapability)
    : device_(std::move(device))
    , authProvider_(std::move(authProvider))
    , updatesProvider_(std::move(updatesProvider))
    , wifidConnector_(ipcFactory->createIpcConnector("wifid"))
    , networkdConnector_(ipcFactory->createIpcConnector("networkd"))
    , pushdConnector_(ipcFactory->createIpcConnector("pushd"))
    , bleInitConnector_(ipcFactory->createIpcConnector("ble_initd"))
    , backendClient_("firstrun-quasar-backend", device_)
    , firstRunServer_(ipcFactory->createIpcServer(SERVICE_NAME))
    , deviceContext_(ipcFactory, [this]() { handleObserverConnect(); }, false)
    , revision_(device_->platformRevision())
    , filePlayerCapability_(std::move(filePlayerCapability))
{
    const auto config = device_->configuration()->getServiceConfig(SERVICE_NAME);
    const int postponesPeriodSec = tryGetInt(config, "otaPostponesPeriodSec", 60);
    otaPostponesPeriod_ = std::chrono::seconds(postponesPeriodSec);

    firstGreetingDoneFilePath_ = tryGetString(config, "firstGreetingDoneFilePath", "first_greeting_done");
    registeredFilePath_ = tryGetString(config, "registeredFilePath", "registered");

    /* true - if should start setup  mode and false if should stop setup mode*/
    deviceContext_.onSetupMode = [this](bool start) mutable {
        bool isConfigurationMode;
        {
            std::lock_guard<std::mutex> guard(mutex_);
            isConfigurationMode = configurationMode_;
        }
        if (start) {
            if (!isConfigurationMode) {
                startInit(false);
            }
        } else {
            if (isConfigurationMode) {
                manualStopInit();
            }
        }
    };

    deviceContext_.onToggleSetupMode = [this]() mutable {
        bool isConfigurationMode;
        {
            std::lock_guard<std::mutex> guard(mutex_);
            isConfigurationMode = configurationMode_;
        }
        if (isConfigurationMode && !isRegistered()) {
            /* Do not allow to skip Init mode if device is not registered */
            return;
        }
        if (isConfigurationMode) {
            manualStopInit();
        } else {
            startInit(false);
        }
    };

    deviceContext_.onChangeAccount = [this](const std::string& oauthCode) {
        const auto res = changeAccount(oauthCode);
        YIO_LOG_INFO("Changing account by SDK call: " << res.registerSuccess << " " << res.registerResponseBody << " " << IAuthProvider::AddUserResponse::statusName(res.addUserResponse));
        Json::Value bodyResponse;
        bodyResponse["addUserStatus"] = IAuthProvider::AddUserResponse::statusName(res.addUserResponse);
        bodyResponse["registerResponse"] = res.registerResponseBody;
        if (isConfigurationMode() && res.registerSuccess) {
            stopInit();
        };
        deviceContext_.fireAuthenticationStatus(oauthCode, res.registerSuccess, jsonToString(bodyResponse));
    };

    checkToken_ = std::make_unique<CheckToken>(device_, authProvider_, deviceStateProvider);

    userConfigProvider->jsonChangedSignal(IUserConfigProvider::ConfigScope::SYSTEM, "/").connect([this](const auto& systemConfig) {
        shouldCheckWifi_ = tryGetBool(*systemConfig, "shouldCheckWifi", true);
        if (auto value = getOptionalBool(*systemConfig, FORCE_WIFI_RECONFIGURE)) {
            forceWifiReconfigure_ = *value;
        } else {
            forceWifiReconfigure_ = getForceWifiReconigure(*device_);
        }
        YIO_LOG_INFO("forceWifiReconfigure is set to " << forceWifiReconfigure_);
    }, lifetime_);

    deviceContext_.onAllowInitConfigurationState = [this]() {
        auto authInfo = authProvider_->ownerAuthInfo().value();
        YIO_LOG_INFO("OnQuasarStart: Connecting to authd, isAuthorized=" << authInfo->isAuthorized() << ", source=" << (int)authInfo->source);
        authProvider_->ownerAuthInfo().connect(
            [this](const auto& authInfo) {
                if (authInfo->source == AuthInfo2::Source::UNDEFINED) {
                    YIO_LOG_INFO("Awaiting authorization...");
                    return;
                }

                // This callback is supposed to be called once on a speaker's start
                // So check whether we've initialized configurationState_
                if (std::scoped_lock lock{mutex_}; configurationState_ != proto::ConfigurationState::UNKNOWN_STATE) {
                    YIO_LOG_INFO("Configuration state valid. Skip callback.");
                    return;
                }

                YIO_LOG_INFO("Auth info changed: isAuthorized=" << authInfo->isAuthorized() << ", isRegistered=" << isRegistered() << ", auth source=" << (int)authInfo->source);
                if (!authInfo->isAuthorized() || !isRegistered()) {
                    startInit(true);
                } else {
                    std::lock_guard<std::mutex> lock(mutex_);
                    started_ = true;
                    configurationMode_ = false;
                    /* If authd already have auth token in startup_info -> device is configured */
                    setConfigurationStateUnlocked(proto::ConfigurationState::CONFIGURED);
                }
            }, lifetime_);
    };

    YIO_LOG_INFO("Creating and starting HTTP server...");
    THttpServer::TOptions httpOptions;
    httpOptions.Port = getInt(config, "httpPort");
    httpOptions.nThreads = 1;
    httpOptions.KeepAliveEnabled = true;
    httpCallback_ = std::make_unique<FirstRunHttpCallback>(this);
    httpServer_ = std::make_unique<FirstRunHttpServer>(httpCallback_.get(), httpOptions);
    httpServer_->Start();
    YIO_LOG_INFO("Creating and starting HTTP server... Done");
}

void FirstRunEndpoint::start() {
    const auto config = device_->configuration()->getServiceConfig(SERVICE_NAME);

    /* Register callback that will handle Updater "ready apply ota" messages. If device is already configured ->
     * allow to apply it asap, otherwise run "postpone" periodic executor
     * NOTE: Updater has it's Hard Deadline when you can't postpone update apply, so it will be applied anyway
     *       when hard deadline exceeded
     */
    updatesProvider_->readyApplyUpdateSignal().connect(
        [this](bool ready) {
            if (ready) {
                YIO_LOG_INFO("Updater is ready to apply update");
                std::scoped_lock<std::mutex> guard(mutex_);
                if (configurationState_ == proto::ConfigurationState::CONFIGURED) {
                    /* If device is already configured -> apply OTA asap */
                    updatesProvider_->confirmUpdateApply();
                } else {
                    updatePostponeExecutor_ = std::make_unique<PeriodicExecutor>(
                        makeSafeCallback([this]() {
                            updatesProvider_->postponeUpdateApply();
                        }, lifetime_), otaPostponesPeriod_, PeriodicExecutor::PeriodicType::CALLBACK_FIRST);
                }
            }
        }, lifetime_);

    YIO_LOG_INFO("Starting -- v1");

    wifidConnector_->setSilentMode(true);
    wifidConnector_->setConnectHandler([=]() {
        YIO_LOG_INFO("Connected to wifid.");
        std::lock_guard<std::mutex> lock(mutex_);
        if (started_ && configurationMode_) {
            disableAllNetworks();
        }
    });

    wifidConnector_->connectToService();

    networkdConnector_->setMessageHandler(std::bind(&FirstRunEndpoint::handleNetworkMessage, this, std::placeholders::_1));
    networkdConnector_->connectToService();

    wifidConnector_->waitUntilConnected();
    YIO_LOG_INFO("Waited for wifid");

    accessPointName_ = getString(device_->configuration()->getServiceConfig("common"), "accessPointName");
    if (!accessPoint_) {
        std::string startAPScript = "/system/bin/startap.sh";
        std::string stopAPScript = "/system/bin/stopap.sh";
        if (config.isMember("startAccessPointScript")) {
            startAPScript = config["startAccessPointScript"].asString();
        }
        if (config.isMember("stopAccessPointScript")) {
            stopAPScript = config["stopAccessPointScript"].asString();
        }
        accessPoint_ = std::make_unique<ScriptsAccessPoint>(device_, std::move(startAPScript), std::move(stopAPScript));
    }

    pushdConnector_->setSilentMode(true);
    pushdConnector_->setMessageHandler(std::bind(&FirstRunEndpoint::handlePushdMessage, this, std::placeholders::_1));
    pushdConnector_->connectToService();

    firstRunServer_->setClientConnectedHandler([=](auto& connection) {
        std::unique_lock<std::mutex> lock(mutex_);
        {
            proto::QuasarMessage message;
            message.set_configuration_state(configurationState_);
            connection.send(std::move(message));
        }
        if (started_) {
            if (configurationMode_ && accessPoint_->isStarted()) {
                /* Send message that AccessPoint is enabled */
                proto::QuasarMessage readyMessage;
                readyMessage.mutable_configuration_mode_ap_up();
                connection.send(std::move(readyMessage));
            }
        }
    });
    firstRunServer_->listenService();

    /* Prepare authd Connector. NOTE: Connection will be established when DeviceContext will receive onAllowInitConfigurationState */
    if (device_->hal()->getBluetoothCapabilities().hasBle()) {
        bleInitConnector_->connectToService();
    }

    /* NOTE: Connect to yandexio sdk after setting up all callbacks AND setting up authdConnector callbacks */
    deviceContext_.connectToSDK();

    auto cryptographyConfig = getJson(device_->configuration()->getServiceConfig("common"), "cryptography");
    deviceCryptography_ = device_->hal()->createDeviceCryptography(cryptographyConfig);

    wifiStoragePath_ = getString(config, "wifiStoragePath");
    wifiStoragePath_.Parent().MkDirs();

    forceWifiReconfigure_ = getForceWifiReconigure(*device_);

    backendUrl_ = getString(device_->configuration()->getServiceConfig("common"), "backendUrl");
    softwareVersion_ = getString(device_->configuration()->getServiceConfig("common"), "softwareVersion");
    try {
        macAddress_ = getFileContent("/sys/class/net/wlan0/address");
    } catch (const std::exception& e) {
        YIO_LOG_ERROR_EVENT("FirstRunEndpoint.FailedGetMacAddress", "Cannot get MAC address: " << e.what());
    }
    backendClient_.setTimeout(std::chrono::milliseconds{9900}); // Borrowed 5*100ms for a pre-sleep, return after SK-4122
    backendClient_.setRetriesCount(5);

    loadSavedWifi();
    if (!savedWifi_.IsInitialized()) {
        loadSavedWifiFromWPASupplicant();
        storeSavedWifi();
    }

    lastConnectedMonotonic_ = std::chrono::steady_clock::now();
    wifiMonitor_ = std::make_unique<PeriodicExecutor>(std::bind(&FirstRunEndpoint::checkWifi, this), std::chrono::minutes(1));

    if (isRegistered() && !isFirstGreetingDone()) {
        /* If device was registered and right after that had critical update ->
         * check if there is an update on boot up. If there is no update -> Device was updated for the first
         * time, so play "configure_success" sound and set up FirstGreetingDone flag to filesystem
         */
        updatesProvider_->waitUpdateState(makeSafeCallback(
                                              [this](UpdatesState2::Critical critical) {
                                                  switch (critical) {
                                                      case UpdatesState2::Critical::UNDEFINED:
                                                          YIO_LOG_INFO("Check was canceled or could not receive value in hour");
                                                          break;
                                                      case UpdatesState2::Critical::YES:
                                                          YIO_LOG_INFO("There is a critical update");
                                                          break;
                                                      case UpdatesState2::Critical::NO:
                                                          YIO_LOG_INFO("No a critical update");
                                                          device_->telemetry()->reportEvent("firstConfigureDone");
                                                          deviceContext_.fireStartingConfigureSuccess(false);
                                                          setFirstGreetingDone();
                                                          break;
                                                  }
                                              }, lifetime_), std::chrono::hours(1));
    }
}

FirstRunEndpoint::FirstRunHttpResponse FirstRunEndpoint::httpPing()
{
    Json::Value result;
    result["status"] = "ok";
    return FirstRunHttpResponse{
        .code = 200,
        .contentType = "application/json",
        .contentBody = jsonToString(result),
    };
}

FirstRunEndpoint::FirstRunHttpResponse FirstRunEndpoint::httpInfo() {
    YIO_LOG_INFO("Received info request");
    std::string deviceId = device_->deviceId();
    Json::Value result;
    result["uuid"] = deviceId;
    result["crypto_key"] = deviceCryptography_->getEncryptPublicKey();
    result["mac"] = macAddress_;
    result["version"] = softwareVersion_;

    return FirstRunHttpResponse{
        .code = 200,
        .contentType = "application/json",
        .contentBody = jsonToString(result),
    };
}

FirstRunEndpoint::FirstRunHttpResponse FirstRunEndpoint::httpSsid() {
    YIO_LOG_INFO("Received /ssid request");

    Json::Value response;
    auto& ssid = response["ssid"];

    ssid = Json::arrayValue;

    YIO_LOG_INFO("Retrieving ssids from wifimanager");

    proto::QuasarMessage ssidListRequest;
    ssidListRequest.mutable_wifi_list_request();

    auto ssidList = wifidConnector_->sendRequestSync(std::move(ssidListRequest), std::chrono::seconds(5));

    int ssidsCounter = 0;
    for (int i = 0; i < ssidList->wifi_list().hotspots().size(); ++i) {
        const auto& hotspot = ssidList->wifi_list().hotspots(i);
        if (hotspot.is_corporate()) {
            /* Do not send to User Application corporate networks, since we can't connect to it
             * (it needs special certificates)
             */
            YIO_LOG_DEBUG("Skip corporate network");
            continue;
        }
        ssid[ssidsCounter]["ssid"] = hotspot.ssid();
        ssid[ssidsCounter]["secure"] = hotspot.secure();
        ssid[ssidsCounter]["level"] = hotspot.rssi();
        /* Use external counter for ssids, otherwise skipped networks will be NULL in Json */
        ++ssidsCounter;
    }

    return FirstRunHttpResponse{
        .code = 200,
        .contentType = "application/json",
        .contentBody = jsonToString(response),
    };
}

FirstRunEndpoint::FirstRunHttpResponse FirstRunEndpoint::httpConnect(const std::string& payload) {
    proto::QuasarMessage ledMessage;
    bool success = false;
    Json::Value response;
    const auto minErrorMonotonic = (std::chrono::steady_clock::now() + minConnectError_);
    ConnectParameters parameters;

    try {
        YIO_LOG_INFO("Received /connect request");
        parameters = parseConnectParameters(payload);

        // WIFI_TYPE_NONE means we are connected via ethernet, without using any wi-fi.
        // Applicable for Station-2 only.
        const bool shouldConnectToWifi = parameters.wifiType != quasar::proto::WifiType::NONE_WIFI_TYPE;

        deviceContext_.fireConnectingToNetwork();

        if (parameters.stopAccessPoint && accessPoint_->isStarted()) {
            stopAccessPoint();
        }

        proto::QuasarMessage connectMessage;
        auto wifiConnect = connectMessage.mutable_wifi_connect();

        if (!parameters.password.empty()) {
            wifiConnect->set_password(TString(parameters.password));
        }

        YIO_LOG_INFO("WifiType is " << parameters.wifiType);
        wifiConnect->set_wifi_type(parameters.wifiType);

        bool hasCriticalUpdate = false;
        do {
            if (shouldConnectToWifi) {
                bool wifiConnected = false;
                for (size_t i = 0; i < parameters.ssids.size(); ++i) {
                    wifiConnect->set_wifi_id(TString(parameters.ssids[i]));
                    YIO_LOG_INFO("connect: wifiId: " << wifiConnect->wifi_id());
                    auto connectResponse = wifidConnector_->sendRequestSync(proto::QuasarMessage{connectMessage}, std::chrono::seconds(70));
                    if (connectResponse->has_wifi_connect_response() &&
                        proto::WifiConnectResponse::OK == connectResponse->wifi_connect_response().status())
                    {
                        wifiConnected = true;
                        break;
                    } else if (i + 1 == parameters.ssids.size()) {
                        response["status"] = "error";
                        lastErrorCode = connectResponse->wifi_connect_response().status();
                        response["error_code"] = lastErrorCode;
                        lastErrorData = "WIFI_" + wifiConnectResponseStatusName(
                                                      connectResponse->wifi_connect_response().status());
                        response["data"] = lastErrorData;
                        auto setupStatus = proto::SetupStatusMessage::SOMETHING_WENT_WRONG;
                        if (connectResponse->has_wifi_connect_response()) {
                            setupStatus = getSetupStatus(connectResponse->wifi_connect_response().status());
                        }

                        YIO_LOG_ERROR_EVENT("FirstRunEndpoint.FailedConnectWifi", "Failed to connect to any wifi ssid. Error code: " << response["error_code"] << ". Data: " << response["data"]);
                        indicateError(setupStatus, minErrorMonotonic, parameters);
                        break;
                    }
                }

                if (!wifiConnected) {
                    break;
                }
            }

            notifySetupStatusViaBle(proto::SetupStatusMessage::NETWORK_CONNECT_OK);

            const ChangeAccountResult changeUserSuccess = changeAccount(parameters.authCode);

            if (changeUserSuccess.addUserResponse != IAuthProvider::AddUserResponse::Status::OK) {
                response["status"] = "error";
                lastErrorCode = (int)changeUserSuccess.addUserResponse;
                response["error_code"] = lastErrorCode;
                lastErrorData = IAuthProvider::AddUserResponse::statusName(changeUserSuccess.addUserResponse);
                response["data"] = lastErrorData;
                auto setupStatus = proto::SetupStatusMessage::SOMETHING_WENT_WRONG;
                if (IAuthProvider::AddUserResponse::Status::NO_INTERNET == changeUserSuccess.addUserResponse) {
                    if (shouldConnectToWifi) {
                        setupStatus = proto::SetupStatusMessage::NO_ACCESS_WIFI;
                    } else {
                        setupStatus = proto::SetupStatusMessage::NO_ACCESS_ETHERNET;
                    }
                }
                indicateError(setupStatus, minErrorMonotonic, parameters);
                break;
            }

            if (changeUserSuccess.registerSuccess) {
                if ("ok" == changeUserSuccess.registerResponseBody["status"].asString()) {
                    notifySetupStatusViaBle(proto::SetupStatusMessage::BACKEND_REGISTER_OK);
                    success = true;
                    hasCriticalUpdate = changeUserSuccess.registerResponseBody["hasCriticalUpdate"].asBool();
                }
            }

            if (!success) {
                indicateError(proto::SetupStatusMessage::SOMETHING_WENT_WRONG, minErrorMonotonic, parameters);
                const int backendErrorCode = 6;
                response["status"] = "error";
                lastErrorCode = backendErrorCode;
                lastErrorData = "BACKEND_ERROR";
                response["error_code"] = lastErrorCode;
                response["data"] = lastErrorData;
                break;
            }

            response["status"] = "ok";
            response["has_critical_update"] = hasCriticalUpdate;
            lastErrorCode = 0;
            YIO_LOG_INFO("Successfully initialized");
            if (shouldConnectToWifi) {
                savedWifi_ = *wifiConnect;
            }
            if (!hasCriticalUpdate) {
                deviceContext_.fireStartingConfigureSuccess(false);
                notifySetupStatusViaBle(proto::SetupStatusMessage::UPDATE_NOT_REQUIRED);
                notifySetupStatusViaBle(proto::SetupStatusMessage::OK);
                setFirstGreetingDone();
            } else {
                deviceContext_.fireStartingConfigureSuccess(true);
                notifySetupStatusViaBle(proto::SetupStatusMessage::UPDATE_IS_REQUIRED);
            }
        } while (false);
    } catch (const std::exception& e) {
        YIO_LOG_ERROR_EVENT("FirstRunEndpoint.FailedHandleConnectRequest", "Exception during processing /connect request: " << e.what());
        indicateError(proto::SetupStatusMessage::SOMETHING_WENT_WRONG, minErrorMonotonic, parameters);
        response["error"] = e.what();
    }

    if (success) {
        stopInit();
        setRegistered();
        storeSavedWifi();
    } else {
        if (accessPoint_->isStarted()) {
            stopAccessPoint();
        }

        // sometimes this fails with timeout
        try {
            disableAllNetworks();
        } catch (const std::exception& e) {
            YIO_LOG_ERROR_EVENT("FirstRunEndpoint.FailedDisableAllNetworks", "Exception during network disabling: " << e.what());
        }

        startAccessPoint();
    }

    return FirstRunHttpResponse{
        .code = 200,
        .contentType = "application/json",
        .contentBody = jsonToString(response),
    };
}

FirstRunEndpoint::ChangeAccountResult FirstRunEndpoint::changeAccount(const std::string& authCode)
{
    const bool withXToken = true;
    auto response = authProvider_->addUser(authCode, UserType::OWNER, withXToken, std::chrono::minutes(1));
    if (response.status != IAuthProvider::AddUserResponse::Status::OK) {
        return ChangeAccountResult::makeError(response.status);
    }

    std::stringstream urlParams;
    urlParams << "device_id=" << urlEncode(device_->deviceId()) << "&name=quasar&activation_code=" << std::to_string(getCrc32(authCode)) << "&firmware_version=" << urlEncode(softwareVersion_) << "&platform=" << urlEncode(device_->platform()) << "&ts=" << std::to_string(time(nullptr));

    if (!revision_.empty()) {
        urlParams << "&revision=" << urlEncode(revision_);
    }
    auto urlParamsString = urlParams.str();
    const std::string backendUrl = backendUrl_ + "/register?" + urlParamsString;

    YIO_LOG_INFO("Backend request: " << backendUrl);

    HttpClient::Headers headers = {{"Authorization", "OAuth " + response.authToken}};

    try {
        const auto signature = deviceCryptography_->sign(urlParamsString);
        HttpClient::addSignatureHeaders(headers, signature, deviceCryptography_->getType());
    } catch (const std::exception& e) {
        YIO_LOG_ERROR_EVENT("FirstRunEndpoint.FailedSignRequestParams", "Can't sign urlParams (signature version 2): " << e.what());
        return ChangeAccountResult::makeError(IAuthProvider::AddUserResponse::Status::CRYPTO_ERROR);
    }

    // Sleep so that passport's database gets a chance to sync newly issued oauth-token, drop this abomination after resolving SK-4122
    std::this_thread::sleep_for(std::chrono::milliseconds(500u));
    const auto registerResponse = backendClient_.post("register", backendUrl, "{}", headers);
    YIO_LOG_INFO("Backend returned code: " << registerResponse.responseCode << ". Body: " << registerResponse.body);

    if (!isSuccessHttpCode(registerResponse.responseCode)) {
        return ChangeAccountResult::makeError(response.status);
    }

    YIO_LOG_INFO("Calling authd.change_user with uid=" << response.id);
    authProvider_->changeUser(response.id, std::chrono::minutes(1));

    auto registerBody = parseJson(registerResponse.body);
    YIO_LOG_INFO("authd.change_user response: " << registerResponse.body);
    return ChangeAccountResult::makeSuccess(response.status, std::move(registerBody));
}

FirstRunEndpoint::ConnectParameters FirstRunEndpoint::parseConnectParameters(const std::string& payload)
{
    ConnectParameters parameters;
    auto request = parseJson(payload);

    const auto& id = request["ssid"];
    if (id.isString()) {
        parameters.ssids.push_back(request["ssid"].asString());
    } else { // id is array of ssids
        for (const auto& ssid : id) {
            parameters.ssids.push_back(ssid.asString());
        }
    }

    auto& cryptography = *deviceCryptography_;
    parameters.password = request["password"].asString();
    bool plain = request["plain"].asBool();
    if (!parameters.password.empty()) {
        YIO_LOG_DEBUG("Password encoded: " << parameters.password);
        if (!plain) {
            parameters.password = cryptography.decrypt(base64Decode(parameters.password));
        }
    }

    parameters.authCode = request["xtoken_code"].asString();
    if (!plain) {
        parameters.authCode = cryptography.decrypt(base64Decode(parameters.authCode));
    }

    if (request.isMember("wifi_type")) {
        parameters.wifiType = proto::WifiType::UNKNOWN_WIFI_TYPE;
        wifiTypeParse(request["wifi_type"].asString(), &parameters.wifiType);
    }

    if (request.isMember("stop_access_point")) {
        parameters.stopAccessPoint = request["stop_access_point"].asBool();
    }

    YIO_LOG_DEBUG("Stopping AP: " << parameters.stopAccessPoint);
    YIO_LOG_DEBUG("WifiType " << parameters.wifiType);
    YIO_LOG_DEBUG("Raw WifiType " << request["wifi_type"].asString());

    return parameters;
}

FirstRunEndpoint::FirstRunHttpResponse FirstRunEndpoint::httpStartInit() {
    startInit(false);
    return FirstRunHttpResponse{
        .code = 200,
        .contentType = "none",
        .contentBody = "OK",
    };
}

FirstRunEndpoint::FirstRunHttpResponse FirstRunEndpoint::httpStopInit() {
    stopInit();
    return FirstRunHttpResponse{
        .code = 200,
        .contentType = "none",
        .contentBody = "OK",
    };
}

void FirstRunEndpoint::startInit(bool firstTime) {
    std::unique_lock<std::mutex> lock(mutex_);
    if (configurationMode_) {
        return;
    }
    firstTimeSetup_ = firstTime;
    started_ = true;
    configurationMode_ = true;

    if (firstTimeSetup_) {
        // start AP before messages on firstTime
        if (!accessPoint_->isStarted()) {
            coldStartAccessPoint();
        }
    }
    deviceContext_.setSetupMode(true, firstTime);

    setConfigurationStateUnlocked(proto::ConfigurationState::CONFIGURING);
    if (firstTimeSetup_) {
        proto::QuasarMessage readyMessage;
        readyMessage.mutable_configuration_mode_ap_up();
        firstRunServer_->sendToAll(std::move(readyMessage));
    }

    lock.unlock();
    try {
        disableAllNetworks();
    } catch (const std::exception& e) // Avoid reconnecting from authd and repeating welcome phrase. Will disable networks on wifid connect
    {
        YIO_LOG_WARN("Cannot send disable networks request to wifid: " << e.what());
    }
    lock.lock();

    if (!firstTimeSetup_) {
        // start AP in parallel if not first time
        if (!accessPoint_->isStarted()) {
            coldStartAccessPoint();
        }
        proto::QuasarMessage readyMessage;
        readyMessage.mutable_configuration_mode_ap_up();
        firstRunServer_->sendToAll(std::move(readyMessage));
    }
}

void FirstRunEndpoint::stopInit() {
    deviceContext_.setSetupMode(false);

    std::lock_guard<std::mutex> lock(mutex_);
    configurationMode_ = false;
    setConfigurationStateUnlocked(proto::ConfigurationState::CONFIGURED);
    stopAccessPoint();
    enableAllNetworks(); // FIXME: this is the least invasive way to work around wifi management split between wifid and firstrund, should be moved out in QUASAR-5619

    notifySetupStatusViaBle(proto::SetupStatusMessage::CONFIGURATION_FINISHED);
}

void FirstRunEndpoint::manualStopInit() {
    stopInit();

    proto::QuasarMessage message;
    message.mutable_reset_wifi();
    wifidConnector_->sendMessage(std::move(message));

    proto::QuasarMessage messageConnect;
    *messageConnect.mutable_wifi_connect() = savedWifi_;
    wifidConnector_->sendMessage(std::move(messageConnect));
}

void FirstRunEndpoint::disableAllNetworks() {
    YIO_LOG_INFO("Disabling all current wifi networks");
    proto::QuasarMessage disableNetworks;
    disableNetworks.mutable_wifi_networks_disable();
    wifidConnector_->sendRequestSync(std::move(disableNetworks), std::chrono::seconds(10));
}

void FirstRunEndpoint::enableAllNetworks() {
    YIO_LOG_INFO("Enabling all current wifi networks");
    proto::QuasarMessage enableNetworks;
    enableNetworks.mutable_wifi_networks_enable();
    wifidConnector_->sendRequestSync(std::move(enableNetworks), std::chrono::seconds(10));
}

void FirstRunEndpoint::coldStartAccessPoint() {
    YIO_LOG_INFO("Cold-Starting access point");
    accessPoint_->coldStart(accessPointName_);
}

void FirstRunEndpoint::startAccessPoint() {
    YIO_LOG_INFO("Starting access point");
    accessPoint_->start(accessPointName_);
}

void FirstRunEndpoint::stopAccessPoint() {
    if (accessPoint_->isStarted()) {
        YIO_LOG_INFO("Stopping access point");
        accessPoint_->stop();
    } else {
        YIO_LOG_WARN("Access Point is already down. Skip stop");
    }
}

void FirstRunEndpoint::indicateError(const proto::SetupStatusMessage::SetupStatus& status, std::chrono::steady_clock::time_point minMonotonicTime,
                                     const FirstRunEndpoint::ConnectParameters& connectParameters) {
    std::string wavName = getSoundName(status);
    YIO_LOG_WARN("Indicate error: " << wavName);
    auto now = std::chrono::steady_clock::now();
    if (now < minMonotonicTime) {
        std::this_thread::sleep_for(std::chrono::duration_cast<std::chrono::milliseconds>(minMonotonicTime - now));
    }

    deviceContext_.fireSetupError();

    filePlayerCapability_->playSoundFile(wavName, quasar::proto::AudioChannel::DIALOG_CHANNEL);
    notifySetupStatusViaBle(status);

    std::string networkType;
    if (connectParameters.wifiType == proto::WifiType::NONE_WIFI_TYPE) {
        networkType = "ethernet";
    } else {
        networkType = "wifi";
    }

    Json::Value eventBody;
    eventBody["networkType"] = networkType;
    eventBody["status"] = status;
    device_->telemetry()->reportEvent("initConnectError", jsonToString(eventBody));
}

std::string FirstRunEndpoint::getSoundName(proto::SetupStatusMessage::SetupStatus status) {
    switch (status) {
        case proto::SetupStatusMessage::SOMETHING_WENT_WRONG:
            return "something_went_wrong.wav";
        case proto::SetupStatusMessage::CANNOT_CONNECT_TO_THE_WIFI:
            return "cannot_connect_to_the_wifi.wav";
        case proto::SetupStatusMessage::WRONG_WIFI_PASSWORD:
            return "wrong_wifi_password.wav";
        case proto::SetupStatusMessage::NO_ACCESS_WIFI:
            return "no_access.wav";
        case proto::SetupStatusMessage::NO_ACCESS_ETHERNET:
            return "activation_no_network_in_ethernet.wav";
        default:
            return "something_went_wrong.wav";
    }
}

proto::SetupStatusMessage::SetupStatus FirstRunEndpoint::getSetupStatus(proto::WifiConnectResponse::Status status) {
    static_assert(proto::WifiConnectResponse::Status_ARRAYSIZE == 4, "Added new wifi status. Please process it");

    switch (status) {
        case proto::WifiConnectResponse::SSID_NOT_FOUND:
        case proto::WifiConnectResponse::TIMEOUT:
            return proto::SetupStatusMessage::CANNOT_CONNECT_TO_THE_WIFI;
        case proto::WifiConnectResponse::AUTH_ERROR:
            return proto::SetupStatusMessage::WRONG_WIFI_PASSWORD;
        default:
            throw std::runtime_error(std::string("No sound for wifi error: ") + wifiConnectResponseStatusName(status));
    }
}

void FirstRunEndpoint::notifySetupStatusViaBle(proto::SetupStatusMessage::SetupStatus status) {
    if (device_->hal()->getBluetoothCapabilities().hasBle()) {
        proto::QuasarMessage message;
        message.mutable_setup_status_message()->set_status(status);
        bleInitConnector_->sendMessage(std::move(message));
    }
}

FirstRunEndpoint::FirstRunHttpResponse FirstRunEndpoint::httpStopAccessPoint() {
    std::unique_lock<std::mutex> lock(mutex_);
    if (accessPoint_->isStarted()) {
        stopAccessPoint();
    }
    lock.unlock();
    return FirstRunHttpResponse{
        .code = 200,
        .contentType = "application/json",
        .contentBody = "{\"status\":\"ok\"}"};
}

FirstRunEndpoint::FirstRunHttpResponse FirstRunEndpoint::httpPauseAccessPoint(int durationSec) {
    std::unique_lock<std::mutex> lock(mutex_);
    if (accessPoint_->isStarted()) {
        stopAccessPoint();
        apCallbackQueue_.addDelayed([this]() {
            /* start AP after durationSec pause */
            std::lock_guard<std::mutex> lock(mutex_);
            startAccessPoint();
        }, std::chrono::seconds(durationSec));
    }
    lock.unlock();
    return FirstRunHttpResponse{
        .code = 200,
        .contentType = "application/json",
        .contentBody = "{\"status\":\"ok\"}"};
}

FirstRunEndpoint::FirstRunHttpResponse FirstRunEndpoint::httpGetLastError() {
    if (lastErrorCode == 0) {
        return FirstRunHttpResponse{
            .code = 200,
            .contentType = "application/json",
            .contentBody = "{\"status\":\"ok\"}"};
    }

    Json::Value response;
    response["error_code"] = lastErrorCode;
    response["data"] = lastErrorData;
    lastErrorCode = 0;
    return FirstRunHttpResponse{
        .code = 200,
        .contentType = "application/json",
        .contentBody = jsonToString(response),
    };
}

void FirstRunEndpoint::handlePushdMessage(const ipc::SharedMessage& message) {
    if (message->has_push_notification() && message->push_notification().operation() == XivaOperations::SWITCH_USER) {
        YIO_LOG_TRACE("switch_user push notification accepted");
        const auto& payload = message->push_notification().message();

        if (payload.empty()) {
            YIO_LOG_ERROR_EVENT("FirstRunEndpoint.BadJson.SwitchUserPush.Empty", "Empty switch_user event payload");
            return;
        }

        Json::Value messageJson;
        try {
            messageJson = parseJson(payload);
        } catch (const Json::Exception& e) {
            YIO_LOG_ERROR_EVENT("FirstRunEndpoint.BadJson.SwitchUserPush", "Invalid JSON in switch_user payload: " << e.what() << ' ' << payload);
            return;
        }

        if (!messageJson.isMember("token")) {
            YIO_LOG_ERROR_EVENT("FirstRunEndpoint.MissingSwitchUserJwtToken", "JWT token not found in switch_user payload: " << payload);
            return;
        }

        const std::string jwtToken = messageJson["token"].asString();
        if (checkToken_->check(jwtToken)) {
            YIO_LOG_DEBUG("JWT token from switch_user payload is ok");
            const auto jwt = decodeJWT(jwtToken);
            const std::string xCode = getStringGrantFromJWT(jwt.get(), "x_code");

            const auto changeAccountResult = changeAccount(xCode);
            if (changeAccountResult.registerSuccess) {
                YIO_LOG_DEBUG("User switched successfully");
            } else {
                YIO_LOG_ERROR_EVENT("FirstRunEndpoint.FailedSwitchUser", "Failed to switch user");
            }
        } else {
            YIO_LOG_ERROR_EVENT("FirstRunEndpoint.InvalidSwitchUserJwtToken", "JWT token from switch_user payload is invalid");
            return;
        }
    }
}

void FirstRunEndpoint::setRegistered() {
    YIO_LOG_DEBUG("SET registered := true");
    PersistentFile file(registeredFilePath_, PersistentFile::Mode::TRUNCATE);
}

bool FirstRunEndpoint::isRegistered() const {
    bool result = fileExists(registeredFilePath_);
    YIO_LOG_DEBUG("GET registered? -> " << (result ? "true" : "false"));
    return result;
}

void FirstRunEndpoint::setFirstGreetingDone() {
    PersistentFile file(firstGreetingDoneFilePath_, PersistentFile::Mode::TRUNCATE);
}

bool FirstRunEndpoint::isFirstGreetingDone() const {
    return fileExists(firstGreetingDoneFilePath_);
}

void FirstRunEndpoint::handleNetworkMessage(const ipc::SharedMessage& message) {
    if (message->has_network_status()) {
        std::lock_guard<std::mutex> lock(mutex_);
        networkStatus_ = message->network_status();
        if (proto::NetworkStatus::CONNECTED == networkStatus_.status()) {
            lastConnectedMonotonic_ = std::chrono::steady_clock::now();
        }
    }
}

void FirstRunEndpoint::checkWifi() {
    std::lock_guard<std::mutex> lock(mutex_);

    if (configurationMode_ || !shouldCheckWifi_.load()) {
        return;
    }

    if (!savedWifi_.IsInitialized()) {
        YIO_LOG_WARN("No saved wifi networks during wifi check");
        device_->telemetry()->reportEvent("savedWifiNotInitialized");
        return;
    }

    // Attempts to restore the connection no earlier than one minute after receiving the last status CONNECTED
    auto monotonicSeconds = std::chrono::steady_clock::now();
    if (std::chrono::duration_cast<std::chrono::seconds>(monotonicSeconds - lastConnectedMonotonic_).count() > 60) {
        if (networkStatus_.status() != proto::NetworkStatus::CONNECTED && networkStatus_.status() != proto::NetworkStatus::CONNECTED_NO_INTERNET) {
            YIO_LOG_INFO("Manually reconnecting to wifi " << savedWifi_.wifi_id());
            proto::QuasarMessage message;
            *message.mutable_async_wifi_connect() = savedWifi_; // TODO: we should remember last network type, or maybe try to connect to both ethernet and wifi
            wifidConnector_->sendMessage(std::move(message));
            device_->telemetry()->reportEvent("wifiReconnectAsync");
        } else if (networkStatus_.status() == proto::NetworkStatus::CONNECTED_NO_INTERNET) {
            if (forceWifiReconfigure_) {
                YIO_LOG_INFO("Reloading network due to no Internet access");
                proto::QuasarMessage message;
                message.mutable_network_config_reload();
                networkdConnector_->sendMessage(std::move(message));
                device_->telemetry()->reportEvent("wifiReconnectConfigReload");
            } else {
                if (networkStatus_.has_wifi_status()) {
                    YIO_LOG_WARN("Connection has no Internet access while connected to wifi " << savedWifi_.wifi_id());
                } else if (networkStatus_.has_ethernet_status()) {
                    YIO_LOG_WARN("Connection has no Internet access while connected to ethernet");
                }
            }
        }
    }
}

void FirstRunEndpoint::loadSavedWifi() {
    if (!wifiStoragePath_.Exists()) {
        return;
    }

    TString serialized = getFileContent(wifiStoragePath_.GetPath());
    std::lock_guard<std::mutex> lock(mutex_);
    if (!savedWifi_.ParseFromString(serialized)) {
        YIO_LOG_ERROR_EVENT("FirstRunEndpoint.BadProto.WifiStorage", "Cannot parse saved wifi network from " << wifiStoragePath_);
        return;
    }
}

void FirstRunEndpoint::storeSavedWifi() {
    if (savedWifi_.IsInitialized()) {
        std::ofstream file_(wifiStoragePath_.GetPath());

        if (!file_.good()) {
            throw std::runtime_error("Cannot open file " + wifiStoragePath_.GetPath() + " for writing");
        }

        std::lock_guard<std::mutex> lock(mutex_);
        file_ << savedWifi_.SerializeAsString();
    }
}

void FirstRunEndpoint::loadSavedWifiFromWPASupplicant() {
    auto config = device_->configuration()->getServiceConfig(SERVICE_NAME);
    if (config.isMember("wifiConfigPath")) {
        YIO_LOG_INFO("Parsing wpa_supplicant.conf");
        wifiConfigPath_ = getString(config, "wifiConfigPath");
        std::ifstream conf(wifiConfigPath_.GetPath());
        if (!conf.good()) {
            YIO_LOG_WARN("Cannot open wpa_supplicant.conf");
            device_->telemetry()->reportEvent("noWPASupplicant");
            return;
        }

        std::string line;
        while (std::getline(conf, line)) {
            boost::trim(line);
            if (boost::starts_with(line, "ssid=\"") && line.length() > 7) {
                std::string ssid = line.substr(6, line.length() - 7);
                boost::trim(ssid);
                if (!ssid.empty()) {
                    savedWifi_.set_wifi_id(TString(ssid));
                    YIO_LOG_INFO("SSID = " << ssid);
                } else {
                    YIO_LOG_WARN("Parsed empty ssid from wpa_supplicant.conf");
                }
            }
            if (boost::starts_with(line, "psk=\"") && line.length() > 6) {
                std::string password = line.substr(5, line.length() - 6);
                boost::trim(password);
                if (!password.empty()) {
                    savedWifi_.set_password(TString(password));
                }
            }
            if (boost::starts_with(line, "key_mgmt=")) {
                if (line.find("WPA") != std::string::npos) {
                    savedWifi_.set_wifi_type(proto::WifiType::WPA);
                }

                if (line.find("NONE") != std::string::npos) {
                    savedWifi_.set_wifi_type(proto::WifiType::OPEN);
                }

                if (line.find("WEP") != std::string::npos) {
                    savedWifi_.set_wifi_type(proto::WifiType::WEP);
                }
            }
        }
    } else {
        YIO_LOG_INFO("wifiConfigPath not defined. Nothing to parse.");
        return;
    }
}

FirstRunEndpoint::~FirstRunEndpoint() {
    lifetime_.die();
    httpServer_->Shutdown();
    httpServer_->Wait();
    apCallbackQueue_.destroy(); // destroy cbQueue so it won't touch AP
    wifidConnector_->shutdown();
    networkdConnector_->shutdown();
    pushdConnector_->shutdown();
    bleInitConnector_->shutdown();

    updatePostponeExecutor_.reset();
}

int FirstRunEndpoint::getServicePort() const {
    return firstRunServer_->port();
}

void FirstRunEndpoint::resetAccessPoint(AccessPoint* accessPoint) {
    accessPoint_.reset(accessPoint);
}

void FirstRunEndpoint::setMinConnectError(std::chrono::milliseconds minConnectError) {
    minConnectError_ = minConnectError;
}

void FirstRunEndpoint::setConfigurationState(proto::ConfigurationState newState) {
    std::lock_guard<std::mutex> guard(mutex_);
    setConfigurationStateUnlocked(newState);
}

void FirstRunEndpoint::setConfigurationStateUnlocked(proto::ConfigurationState newState) {
    if (newState == configurationState_) {
        return;
    }
    YIO_LOG_INFO("New Configuration state: " << PROTO_ENUM_TO_STRING(proto::ConfigurationState, newState));
    configurationState_ = newState;
    proto::QuasarMessage message;
    message.set_configuration_state(newState);
    firstRunServer_->sendToAll(std::move(message));
    if (newState == proto::ConfigurationState::CONFIGURED) {
        updatePostponeExecutor_.reset();
        /* If updater is ready to apply OTA -> allow it apply it asap */
        updatesProvider_->confirmUpdateApply();
        /* ping Updatesd to check updates as soon as device enter CONFIGURED mode */
        updatesProvider_->checkUpdates();
    }
}

bool FirstRunEndpoint::isConfigurationMode() const {
    std::lock_guard<std::mutex> guard(mutex_);
    return configurationMode_;
}

void FirstRunEndpoint::waitConnectorsConnections() {
    networkdConnector_->waitUntilConnected();
    deviceContext_.waitUntilConnected();
}

FirstRunEndpoint::FirstRunHttpCallback::FirstRunHttpCallback(FirstRunEndpoint* endpoint)
    : endpoint_(endpoint)
{
}

TClientRequest* FirstRunEndpoint::FirstRunHttpCallback::CreateClient()
{
    return new FirstRunHttpRequest(endpoint_);
}

FirstRunEndpoint::FirstRunHttpRequest::FirstRunHttpRequest(FirstRunEndpoint* endpoint)
    : endpoint_(endpoint)
{
}

bool FirstRunEndpoint::FirstRunHttpRequest::DoReply(const TReplyParams& params)
{
    TParsedHttpFull request(params.Input.FirstLine());
    FirstRunHttpResponse response;
    try {
        if (!endpoint_->isConfigurationMode()) {
            YIO_LOG_WARN("Request when not in configuration mode. Rejecting. Handle: " << request.Path);
            params.Output << THttpResponse(HTTP_FORBIDDEN).SetContent("Access denied", "text/plain");
            params.Output.Flush();
            return true;
        }

        if ("/ping" == request.Path) {
            response = endpoint_->httpPing();
        } else if ("/info" == request.Path) {
            response = endpoint_->httpInfo();
        } else if ("/ssid" == request.Path) {
            response = endpoint_->httpSsid();
        } else if ("/connect" == request.Path) {
            auto payload = params.Input.ReadAll();
            response = endpoint_->httpConnect(payload);
        } else if ("/start_init" == request.Path) {
            response = endpoint_->httpStartInit();
        } else if ("/stop_init" == request.Path) {
            response = endpoint_->httpStopInit();
        } else if ("/stop_access_point" == request.Path) {
            response = endpoint_->httpStopAccessPoint();
        } else if ("/pause_access_point" == request.Path) {
            int durationSec = 0;
            for (const auto& header : params.Input.Headers()) {
                if (header.Name() == "duration") {
                    durationSec = boost::lexical_cast<int>(header.Value());
                }
            }
            response = endpoint_->httpPauseAccessPoint(durationSec);
        } else if ("/get_last_error" == request.Path) {
            response = endpoint_->httpGetLastError();
        } else {
            response = FirstRunHttpResponse{
                .code = 200,
                .contentType = "text/plain",
                .contentBody = "OK",
            };
        }
    } catch (const std::exception& ex) {
        YIO_LOG_ERROR_EVENT("FirstRunEndpoint.FailedHandleHttpRequest", "Error processing request to " << request.Path << ": " << ex.what());
        Json::Value json;
        json["error"] = ex.what();
        response = FirstRunHttpResponse{
            .code = 200,
            .contentType = "application/json",
            .contentBody = jsonToString(json),
        };
    }

    params.Output << THttpResponse(static_cast<HttpCodes>(response.code))
                         .SetContentType(response.contentType)
                         .SetContent(response.contentBody.c_str());
    params.Output.Flush();
    return true;
}

void FirstRunEndpoint::handleObserverConnect() {
    std::scoped_lock<std::mutex> lock(mutex_);
    deviceContext_.setSetupMode(configurationMode_, firstTimeSetup_);
}
