#pragma once

#include "access_point.h"
#include "check_token.h"

#include <yandex_io/capabilities/file_player/interfaces/i_file_player_capability.h>
#include <yandex_io/external_libs/datacratic/soa/service/http_endpoint.h>
#include <yandex_io/interfaces/auth/i_auth_provider.h>
#include <yandex_io/interfaces/device_state/i_device_state_provider.h>
#include <yandex_io/interfaces/updates/i_updates_provider.h>
#include <yandex_io/interfaces/user_config/i_user_config_provider.h>
#include <yandex_io/libs/base/named_callback_queue.h>
#include <yandex_io/libs/cryptography/cryptography.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/device_cryptography/device_cryptography.h>
#include <yandex_io/libs/http_client/http_client.h>
#include <yandex_io/libs/ipc/i_ipc_factory.h>
#include <yandex_io/libs/threading/periodic_executor.h>
#include <yandex_io/libs/threading/steady_condition_variable.h>
#include <yandex_io/protos/model_objects.pb.h>
#include <yandex_io/sdk/private/device_context.h>

#include <util/folder/path.h>

#include <chrono>
#include <future>
#include <thread>

namespace quasar {

    class FirstRunEndpoint {
    public:
        FirstRunEndpoint(
            std::shared_ptr<YandexIO::IDevice> device,
            std::shared_ptr<ipc::IIpcFactory> ipcFactory,
            std::shared_ptr<IAuthProvider> authProvider,
            std::shared_ptr<IDeviceStateProvider> deviceStateProvider,
            std::shared_ptr<IUpdatesProvider> updatesProvider,
            std::shared_ptr<IUserConfigProvider> userConfigProvider,
            std::shared_ptr<YandexIO::IFilePlayerCapability> filePlayerCapability);

        void start();

        ~FirstRunEndpoint();

        int getServicePort() const;

        /**
         * Wait until internal connectors set up connections to servers
         */
        void waitConnectorsConnections();

        void resetAccessPoint(AccessPoint* accessPoint);
        void setMinConnectError(std::chrono::milliseconds minConnectError);
        void setConfigurationState(quasar::proto::ConfigurationState newState);
        void setConfigurationStateUnlocked(quasar::proto::ConfigurationState newState);
        bool isConfigurationMode() const;

        static const std::string SERVICE_NAME;

    private:
        struct ConnectParameters {
            std::vector<std::string> ssids;
            std::string password;
            std::string authCode;
            quasar::proto::WifiType wifiType = quasar::proto::WifiType::UNKNOWN_WIFI_TYPE;
            bool stopAccessPoint = true;
        };

        struct ChangeAccountResult {
            using Status = quasar::IAuthProvider::AddUserResponse::Status;

            bool registerSuccess;
            Status addUserResponse;
            Json::Value registerResponseBody;

            static auto makeError(Status status) {
                return ChangeAccountResult{false, status, Json::Value()};
            }

            static auto makeSuccess(Status status, Json::Value response) {
                return ChangeAccountResult{true, status, std::move(response)};
            }

        private:
            ChangeAccountResult(
                bool registerSuccess_,
                Status addUserResponse_,
                Json::Value registerResponseBody_)
                : registerSuccess(registerSuccess_)
                , addUserResponse(addUserResponse_)
                , registerResponseBody(std::move(registerResponseBody_))
            {
            }
        };

        void startInit(bool firstTime);
        void stopInit();
        void manualStopInit();

        void disableAllNetworks();
        void enableAllNetworks();

        void startAccessPoint();
        void coldStartAccessPoint();
        void stopAccessPoint();

        void indicateError(const proto::SetupStatusMessage::SetupStatus& status, std::chrono::steady_clock::time_point minMonotonicTime, const FirstRunEndpoint::ConnectParameters& connectParameters);

        static std::string getSoundName(quasar::proto::SetupStatusMessage::SetupStatus status);
        static quasar::proto::SetupStatusMessage::SetupStatus getSetupStatus(proto::WifiConnectResponse::Status status);
        void notifySetupStatusViaBle(quasar::proto::SetupStatusMessage::SetupStatus status);

        void handleNetworkMessage(const ipc::SharedMessage& message);
        void handlePushdMessage(const ipc::SharedMessage& message);

        void handleObserverConnect();

        FirstRunEndpoint::ChangeAccountResult changeAccount(const std::string& authCode);

        void setRegistered();
        bool isRegistered() const;

        void setFirstGreetingDone();
        bool isFirstGreetingDone() const;

        void loadSavedWifi();
        void storeSavedWifi();
        void loadSavedWifiFromWPASupplicant();

        void checkWifi();

        struct FirstRunHttpResponse {
            int code{500};
            std::string contentType;
            std::string contentBody;
        };

        static FirstRunHttpResponse httpPing();
        FirstRunHttpResponse httpInfo();
        FirstRunHttpResponse httpSsid();
        FirstRunHttpResponse httpConnect(const std::string& payload);
        FirstRunHttpResponse httpStartInit();
        FirstRunHttpResponse httpStopInit();
        FirstRunHttpResponse httpStopAccessPoint();
        FirstRunHttpResponse httpPauseAccessPoint(int durationSec);
        FirstRunHttpResponse httpGetLastError();
        ConnectParameters parseConnectParameters(const std::string& payload);

        class FirstRunHttpServer;
        class FirstRunHttpCallback;
        class FirstRunHttpRequest;

        Lifetime lifetime_;
        const std::shared_ptr<YandexIO::IDevice> device_;
        const std::shared_ptr<IAuthProvider> authProvider_;
        const std::shared_ptr<IUpdatesProvider> updatesProvider_;
        std::unique_ptr<FirstRunHttpCallback> httpCallback_;
        std::unique_ptr<FirstRunHttpServer> httpServer_;

        int lastErrorCode = 0;
        std::string lastErrorData;

        std::shared_ptr<ipc::IConnector> wifidConnector_;
        std::shared_ptr<ipc::IConnector> networkdConnector_;
        std::shared_ptr<ipc::IConnector> pushdConnector_;
        std::shared_ptr<ipc::IConnector> bleInitConnector_;

        std::unique_ptr<CheckToken> checkToken_;
        std::unique_ptr<YandexIO::DeviceCryptography> deviceCryptography_;

        std::string backendUrl_;
        std::string accessPointName_;
        std::string softwareVersion_;
        std::string macAddress_;

        TFsPath wifiStoragePath_;
        TFsPath wifiConfigPath_;

        std::atomic<bool> shouldCheckWifi_{true};

        // force wpa_supplicant config reload on Internet access lost - QUASAR-2536
        std::atomic_bool forceWifiReconfigure_{false};

        mutable std::mutex mutex_;

        /* ConfigurationState hold advanced state (with data reset, etc), configurationMode_ hold actual information
         * if device is in configurations mode or not.
         * FIXME: remove configurationMode_ along with DATA_RESET from ConfigurationState
         */
        bool configurationMode_ = false;
        quasar::proto::ConfigurationState configurationState_{quasar::proto::ConfigurationState::UNKNOWN_STATE};

        bool started_ = false;

        HttpClient backendClient_;

        std::shared_ptr<ipc::IServer> firstRunServer_;

        std::unique_ptr<AccessPoint> accessPoint_;

        quasar::proto::WifiConnect savedWifi_;
        quasar::proto::NetworkStatus networkStatus_;
        std::chrono::steady_clock::time_point lastConnectedMonotonic_;

        std::unique_ptr<PeriodicExecutor> updatePostponeExecutor_;

        std::unique_ptr<PeriodicExecutor> wifiMonitor_;

        std::chrono::milliseconds minConnectError_{20000};

        std::chrono::seconds otaPostponesPeriod_{std::chrono::minutes(1)};

        /* Callback queue to work with AccessPoint */
        NamedCallbackQueue apCallbackQueue_{"FirstRunEndpoint"};
        YandexIO::DeviceContext deviceContext_;

        bool firstTimeSetup_ = false;

        std::string firstGreetingDoneFilePath_;
        std::string registeredFilePath_;
        const std::string revision_;

        const std::shared_ptr<YandexIO::IFilePlayerCapability> filePlayerCapability_;
    };

} // namespace quasar
