#include "fluent_bit_ctx.h"

extern "C" {
#include <contrib/tools/fluent-bit-minimal/include/fluent-bit/flb_lib.h>
#include <contrib/tools/fluent-bit-minimal/include/fluent-bit/flb_input.h>
}

#include <stdexcept>
#include <contrib/tools/fluent-bit-minimal/include/fluent-bit/flb_str.h>

using namespace quasar;

extern struct flb_input_plugin in_logcat_plugin;

namespace {

    template <auto FlbCreate, auto FlbSet>
    void addPlugin(flb_ctx_t* ctx, const std::string& name, const std::unordered_multimap<std::string, std::string>& properties) {
        int ffd = FlbCreate(ctx, name.c_str(), nullptr);
        if (ffd < 0) {
            throw std::runtime_error("Failed to create plugin '" + name + "'");
        }

        for (const auto& [key, value] : properties) {
            int ret = FlbSet(
                ctx, ffd,
                key.c_str(), value.c_str(),
                nullptr);
            if (ret != 0) {
                throw std::runtime_error("Failed to set plugin property '" + key + "'");
            }
        }
    }

} // namespace

FluentBitCtx::FluentBitCtx()
    : ctx_(flb_create())
{
    if (!ctx_) {
        throw std::runtime_error("Failed to create flb context");
    }

    libInput_ = flb_input(ctx_, "lib", nullptr);
    if (libInput_ == -1) {
        throw std::runtime_error("Failed to create 'lib' input");
    }
}

FluentBitCtx::~FluentBitCtx()
{
    flb_stop(ctx_);
    flb_destroy(ctx_);
}

void FluentBitCtx::addInput(const std::string& name, const std::unordered_multimap<std::string, std::string>& properties) {
    addPlugin<flb_input, flb_input_set>(ctx_, name, properties);
}

void FluentBitCtx::addFilter(const std::string& name, const std::unordered_multimap<std::string, std::string>& properties) {
    addPlugin<flb_filter, flb_filter_set>(ctx_, name, properties);
}

void FluentBitCtx::addOutput(const std::string& name, const std::unordered_multimap<std::string, std::string>& properties) {
    addPlugin<flb_output, flb_output_set>(ctx_, name, properties);
}

void FluentBitCtx::setProperties(const std::unordered_multimap<std::string, std::string>& properties) {
    for (const auto& [key, value] : properties) {
        int ret = flb_service_set(
            ctx_,
            key.c_str(), value.c_str(),
            nullptr);
        if (ret != 0) {
            throw std::runtime_error("Failed to set service property '" + key + "'");
        }
    }
}

void FluentBitCtx::start() {
    int ret = flb_start(ctx_);
    if (ret != 0) {
        throw std::runtime_error("Failed to start fluent-bit engine");
    }
}

void FluentBitCtx::log(const std::string& data) {
    flb_lib_push(ctx_, libInput_, data.data(), data.size());
}

int FluentBitCtx::loadLogcatPlugin() {
    struct flb_input_plugin* plugin = (flb_input_plugin*)flb_malloc(sizeof(struct flb_input_plugin));
    if (!plugin) {
        flb_errno();
        return -1;
    }
    memcpy(plugin, &in_logcat_plugin, sizeof(struct flb_input_plugin));
    mk_list_add(&plugin->_head, &ctx_->config->in_plugins);

    return 0;
}

void FluentBitCtx::initInternalLogging(const std::string& logPath) {
    ctx_->config->log_file = flb_strdup(logPath.c_str());
}
