#include "fluent_bit_lib.h"

#include "fluent_bit_ctx.h"

#include <yandex_io/services/fluent-bitd/fluent_bit_service.h>

#include <yandex_io/libs/configuration/configuration.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/protos/quasar_proto.pb.h>

using namespace quasar;

FluentBitLib::FluentBitLib(
    std::shared_ptr<YandexIO::IDevice> device,
    const Json::Value& config,
    const std::shared_ptr<ipc::IIpcFactory>& ipcFactory)
    : FluentBitBase(std::move(device), config)
    , server_(ipcFactory->createIpcServer(FluentBitService::SERVICE_NAME))
{
    if (config["fluent-bit"].isMember("logging")) {
        internalLogEnabled_ = tryGetBool(config["fluent-bit"]["logging"], "enabled", false);
        internalLogPath_ = tryGetString(config["fluent-bit"]["logging"], "fileName", DEFAULT_LOG_PATH);
    }

    const auto& paths = config["fluent-bit"]["tailPaths"];
    tailPaths_.reserve(paths.size());
    for (const auto& path : paths) {
        tailPaths_.push_back(path.asString());
    }

    server_->setMessageHandler([this](const auto& message, ipc::IServer::IClientConnection& /*connection*/) {
        processQuasarMessage(*message);
    });
    server_->listenService();
}

FluentBitLib::~FluentBitLib() = default;

void FluentBitLib::setupFluentBit() {
    std::scoped_lock lock(mutex_);

    flb_.reset();

    if (!enabled_) {
        YIO_LOG_INFO("Fluent Bit is disabled");
        return;
    }

    YIO_LOG_INFO("Setting up Fluent Bit");

    flb_ = std::make_unique<FluentBitCtx>();
    if (internalLogEnabled_) {
        flb_->initInternalLogging(internalLogPath_);
    }

    flb_->setProperties({{"Flush", variables_["flushInterval"]},
                         {"HTTP_Server", variables_["httpServer"]},
                         {"HTTP_Listen", "0.0.0.0"},
                         {"HTTP_PORT", variables_["httpServerPort"]}});

    if (variables_["kernelLogsDst"] == "http") {
        YIO_LOG_INFO("Enabling kernel logs");

        flb_->addInput("kmsg", {{"Alias", "kernel"},
                                {"Tag", "kernel"}});
        flb_->addFilter("modify", {{"Match", "kernel"},
                                   {"Add", "Path dmesg"},
                                   {"Rename", "msg Data"}});
    }

    if (variables_["logcatDst"] == "http") {
        int ret = flb_->loadLogcatPlugin();
        if (ret < 0) {
            throw std::runtime_error("Failed to load 'logcat' plugin");
        }

        YIO_LOG_INFO("Enabling logcat plugin");
        std::unordered_multimap<std::string, std::string> properties = {{"Alias", "logcat"},
                                                                        {"Tag", "logcat"},
                                                                        {"Path", "logcat"}};
        if (auto it = variables_.find("logcatLogBufferSize"); it != variables_.end()) {
            properties.insert({"Log_Buffer_Size", it->second});
        }

        flb_->addInput("logcat", properties);
    }

    if (variables_["daemonsLogsDst"] == "http") {
        YIO_LOG_INFO("Enabling daemons logs");

        for (const auto& path : tailPaths_) {
            YIO_LOG_INFO("Tailing '" + path + "'");

            flb_->addInput("tail", {{"Alias", "daemons"},
                                    {"Tag", "daemons"},
                                    {"Path", path},
                                    {"DB", variables_["tailDbFile"]},
                                    {"Path_Key", "Path"},
                                    {"Key", "Data"},
                                    {"Buffer_Chunk_Size", variables_["tailBufferChunkSize"]},
                                    {"Buffer_Max_Size", variables_["tailBufferMaxSize"]},
                                    {"Skip_Long_Lines", variables_["tailSkipLongLines"]}});
        }
    }

    libLogsEnabled_ = variables_["libLogsDst"] == "http";

    const auto deviceId = device_->deviceId();
    const auto devicePlatform = device_->configuration()->getDeviceType();

    flb_->addFilter("record_modifier", {{"Match", "*"},
                                        {"Whitelist_key", "Data"},
                                        {"Whitelist_key", "Path"},
                                        {"Record", "DeviceId " + deviceId},
                                        {"Record", "Platform " + devicePlatform}});

    const auto commonConfig = device_->configuration()->getServiceConfig("common");
    const auto crtFilePath = getString(commonConfig, "caCertsFile");

    flb_->addOutput("http", {{"Alias", "clickdaemon"},
                             {"Match", "*"},
                             {"Host", variables_["clickdaemonHost"]},
                             {"Port", variables_["clickdaemonPort"]},
                             {"URI", variables_["clickdaemonUri"]},
                             {"IPv6", variables_["httpIPv6"]},
                             {"tls", variables_["httpTls"]},
                             {"tls.ca_file", crtFilePath},
                             {"Format", "json"},
                             {"Compress", variables_["httpCompress"]},
                             {"json_date_format", "epoch"},
                             {"json_date_key", "Timestamp"},
                             {"Retry_Limit", variables_["httpRetryLimit"]}});

    YIO_LOG_INFO("Starting Fluent Bit");

    flb_->start();
}

void FluentBitLib::processQuasarMessage(const proto::QuasarMessage& message) {
    if (!message.has_log_message()) {
        return;
    }

    std::scoped_lock lock(mutex_);
    if (!flb_) {
        return;
    }

    if (!libLogsEnabled_) {
        return;
    }

    const auto& log = message.log_message();

    Json::Value value;
    value.append(log.timestamp_ms() / 1000);
    value.append(Json::Value{Json::objectValue});
    value[1]["Data"] = log.msg();
    value[1]["Path"] = "lib";

    flb_->log(jsonToString(value));
}
