#include "fluent_bit_base.h"

#include <yandex_io/libs/base/persistent_file.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/sqlite/sqlite_database.h>

#include <util/folder/path.h>

YIO_DEFINE_LOG_MODULE("fluent-bit");

using namespace quasar;

FluentBitBase::FluentBitBase(std::shared_ptr<YandexIO::IDevice> device, const Json::Value& config)
    : device_(std::move(device))
    , configHelper_(config["fluent-bit"])
    , tailDbBackupFileName_(getString(config, "tailDbBackupFileName"))
    , httpClient_("local-fluent-bit", device_)
{
    enabled_ = config["fluent-bit"]["enabled"].asBool();

    const auto& variables = config["fluent-bit"]["variables"];
    for (const auto& key : variables.getMemberNames()) {
        variables_[key] = variables[key].asString();
    }

    httpClient_.setTimeout(std::chrono::milliseconds{5000});
    httpClient_.setRetriesCount(0);

    const auto collectMetricsPeriod = std::chrono::milliseconds(config["collectMetricsPeriodMs"].asUInt64());
    collectMetricsExecutor_ = std::make_unique<PeriodicExecutor>(
        std::bind(&FluentBitBase::collectMetrics, this),
        collectMetricsPeriod);
}

FluentBitBase::~FluentBitBase() {
}

void FluentBitBase::init() {
    loadTailDbBackup();

    try {
        setupFluentBit();
    } catch (const std::runtime_error& e) {
        YIO_LOG_ERROR_EVENT("FluentBitBase.InitFailure.Exception", "Setup failed: " << e.what());
    }
}

void FluentBitBase::teardown() {
    makeTailDbBackup();
}

void FluentBitBase::processNewConfig(const Json::Value& config) {
    const auto configUpdate = configHelper_.getConfigurationUpdate(config);
    if (configUpdate.isNull()) {
        return;
    }

    YIO_LOG_INFO("Got fluent-bit config update: " << jsonToString(configUpdate));

    std::lock_guard<std::mutex> guard(configMutex_);
    if (configUpdate.isMember("enabled")) {
        enabled_ = configUpdate["enabled"].asBool();
    }

    if (configUpdate.isMember("variables")) {
        auto variablesUpdate = configUpdate["variables"];
        for (const auto& key : variablesUpdate.getMemberNames()) {
            variables_[key] = variablesUpdate[key].asString();
        }
    }

    setupFluentBit();
}

void FluentBitBase::collectMetrics() {
    std::lock_guard<std::mutex> guard(configMutex_);

    if (!enabled_) {
        YIO_LOG_DEBUG("No metrics collected: fluent-bit is not enabled");
        return;
    }

    if (variables_["httpServer"] != "On") {
        YIO_LOG_DEBUG("No metrics collected: fluent-bit metrics server is not enabled");
        return;
    }

    const auto metricsEndpoint = "http://localhost:" + variables_["httpServerPort"] + "/api/v1";

    auto uptimeJson = makeRequest("uptime", metricsEndpoint, "/uptime");
    if (uptimeJson.isNull()) {
        // if we can't get uptime, we will not be able to get metrics
        return;
    }

    uptimeJson.removeMember("uptime_hr"); // it's just human readable version of uptime_sec
    reportMetrics("fluentBitUptime", uptimeJson);

    auto metricsJson = makeRequest("metrics", metricsEndpoint, "/metrics");
    if (metricsJson.isNull()) {
        return;
    }

    reportMetrics("fluentBitMetrics", metricsJson);
}

Json::Value FluentBitBase::makeRequest(std::string_view tag, const std::string& endpoint, const std::string& path) {
    try {
        auto response = httpClient_.get(tag, endpoint + path);

        if (response.responseCode != 200) {
            YIO_LOG_WARN(
                "Got non-200 response on GET " << path
                                               << ": [" << response.responseCode << "] " << response.body);
        } else {
            return parseJson(response.body);
        }
    } catch (const std::runtime_error& ex) {
        YIO_LOG_WARN("Error while executing request: " << ex.what());
    }

    return Json::Value();
}

void FluentBitBase::reportMetrics(const std::string& name, const Json::Value& value) {
    device_->telemetry()->reportEvent(name, jsonToString(value));
}

void FluentBitBase::makeTailDbBackup() {
    std::lock_guard<std::mutex> guard(configMutex_);
    const auto tailDbFile = variables_["tailDbFile"];

    try {
        SqliteDatabase database(tailDbFile);
        database.backupToFile(tailDbBackupFileName_);
        YIO_LOG_INFO("Tail database backup successfully created");
    } catch (const std::exception& ex) {
        YIO_LOG_ERROR_EVENT("FluentBitClient.FailedSaveDatabaseBackup", "Failed to create database backup. " << ex.what());
    }
}

void FluentBitBase::loadTailDbBackup() {
    if (fileExists(tailDbBackupFileName_)) {
        YIO_LOG_INFO("Found tail database backup file: " << tailDbBackupFileName_);
        try {
            SqliteDatabase database(tailDbBackupFileName_);
            database.checkIntegrity();

            const auto tailDbFile = variables_["tailDbFile"];
            TransactionFile backupFile(tailDbFile, tailDbFile + "_tmp");
            TFsPath(tailDbBackupFileName_).CopyTo(backupFile.temporaryFilename(), true);
            backupFile.commit();
            YIO_LOG_INFO("Tail database successfully restored from backup file");
        } catch (const std::exception& ex) {
            YIO_LOG_ERROR_EVENT("FluentBitClient.FailedRestoreDatabaseBackup", "Failed to load database backup. " << ex.what());
        }

        std::remove(tailDbBackupFileName_.c_str());
    }
}
