#include "fluent_bit_client.h"

#include <yandex_io/libs/base/persistent_file.h>
#include <yandex_io/libs/logging/logging.h>

YIO_DEFINE_LOG_MODULE("fluent-bit");

using namespace quasar;

FluentBitClient::FluentBitClient(std::shared_ptr<YandexIO::IDevice> device, const Json::Value& config)
    : FluentBitBase(std::move(device), config)
    , deviceCfgFileName_(getString(config, "deviceCfgFileName"))
    , variablesCfgFileName_(getString(config, "variablesCfgFileName"))
    , setModeScriptFileName_(getString(config, "setModeScriptFileName"))
    , killScriptFileName_(getString(config, "killScriptFileName"))
{
}

FluentBitClient::~FluentBitClient() {
}

void FluentBitClient::setupFluentBit() {
    writeDeviceConfig();
    writeVariablesConfig();
    updateConfigSymlink();
    restartFluentBit();
}

void FluentBitClient::writeDeviceConfig() {
    YIO_LOG_INFO("Writing device data to " << deviceCfgFileName_);

    AtomicFile deviceCfgFile(deviceCfgFileName_);
    const auto deviceId = device_->deviceId();
    const auto devicePlatform = device_->configuration()->getDeviceType();
    auto commonConfig = device_->configuration()->getServiceConfig("common");
    auto crtFilePath = getString(commonConfig, "caCertsFile");

    std::stringstream ss;
    ss << "@SET device_id=" << deviceId << std::endl;
    ss << "@SET device_platform=" << devicePlatform << std::endl;
    ss << "@SET device_crtFile=" << crtFilePath << std::endl;
    deviceCfgFile.write(ss.str());
}

void FluentBitClient::writeVariablesConfig() {
    YIO_LOG_INFO("Updating " << variablesCfgFileName_);
    AtomicFile variablesFile(variablesCfgFileName_);
    std::stringstream ss;

    for (const auto& item : variables_) {
        if (!item.second.empty()) {
            ss << "@SET " << item.first << "=" << item.second << std::endl;
        }
    }
    variablesFile.write(ss.str());
}

void FluentBitClient::updateConfigSymlink() {
    if (enabled_) {
        YIO_LOG_INFO("fluent-bit is now enabled, resetting symlink to logs.cfg");
        std::system((setModeScriptFileName_ + " enabled").c_str());
    } else {
        YIO_LOG_INFO("fluent-bit is now disabled, resetting symlink to idle.cfg");
        std::system((setModeScriptFileName_ + " disabled").c_str());
    }
}

void FluentBitClient::restartFluentBit() {
    YIO_LOG_INFO("Killing fluent-bit process");
    std::system((killScriptFileName_).c_str()); // quasar launcher should do the rest
}
