#include "backend_device_state_updater.h"

#include <yandex_io/libs/http_client/http_client.h>
#include <yandex_io/libs/base/named_callback_queue.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/base/retry_delay_counter.h>

namespace {

    using namespace quasar;

    constexpr std::chrono::seconds DEFAULT_MINIMAL_SENDING_INTERVAL = std::chrono::minutes(15);

    bool is200Response(const HttpClient::HttpResponse& response) {
        return response.responseCode >= 200 && response.responseCode < 300;
    }

    bool isAuthTokenInvalidResponse(const HttpClient::HttpResponse& response) {
        return response.responseCode == 403 && getStringSafe(response.body, ".message") == "AUTH_TOKEN_INVALID";
    }

    std::string getBackendUrl(YandexIO::IDevice& device) {
        return device.configuration()->getServiceConfig("common")["backendUrl"].asString();
    }

    class UpdaterImpl: public glagol::BackendDeviceStateUpdater {
        const std::string url_;
        std::shared_ptr<IAuthProvider> authProvider_;
        HttpClient::Headers headers_;
        RetryDelayCounter retryDelayCounter_;
        std::chrono::seconds minimalSendingInterval_{DEFAULT_MINIMAL_SENDING_INTERVAL};
        std::shared_ptr<ISimpleHttpClient> httpClient_;
        std::string payload_;
        std::chrono::steady_clock::time_point lastSent_;
        Lifetime lifetime_;
        NamedCallbackQueue queue_;

        static std::string makeUrl(const std::string& backendUrl, const std::string& deviceId, const std::string& platform) {
            return backendUrl + "/update_device_state?device_id=" + deviceId + "&platform=" + platform + "&field=networkInfo&send_push";
        }

        static HttpClient::Headers makeHeaders(const std::string& token) {
            return HttpClient::Headers{
                {"Authorization", "OAuth " + token},
                {"Content-Type", "application/json"},
            };
        }

        void doSending() {
            YIO_LOG_DEBUG(payload_);
            try {
                if (const auto owner = authProvider_->ownerAuthInfo().value(); owner == nullptr || owner->authToken.empty()) {
                    throw std::runtime_error("Not authorized yet, delaying");
                }
                auto response = httpClient_->post("update_network_info", url_, payload_, headers_);
                if (!is200Response(response)) {
                    if (isAuthTokenInvalidResponse(response)) {
                        authProvider_->requestAuthTokenUpdate("update_device_state NetworkInfo");
                    }
                    throw std::runtime_error("Non 200 response code: " + std::to_string(response.responseCode) + " " + response.responseStatus);
                }
                payload_.clear();
                lastSent_ = std::chrono::steady_clock::now();
                retryDelayCounter_.reset();
            } catch (const std::runtime_error& e) {
                YIO_LOG_WARN("Failed to do update_device_state " << e.what());
                scheduleSending(retryDelayCounter_.get());
                retryDelayCounter_.increase();
            }
        }

        void scheduleSending(std::chrono::milliseconds delayTime) {
            YIO_LOG_INFO("Schedule sending for " << delayTime.count() << " ms");
            queue_.addDelayed([this] {
                doSending();
            }, delayTime);
        }

    public:
        UpdaterImpl(std::shared_ptr<YandexIO::IDevice> device, std::shared_ptr<IAuthProvider> authProvider, std::shared_ptr<quasar::ISimpleHttpClient> backendClient)
            : url_(makeUrl(getBackendUrl(*device), device->deviceId(), device->configuration()->getDeviceType()))
            , authProvider_(std::move(authProvider))
            , httpClient_(std::move(backendClient))
            , lastSent_(std::chrono::steady_clock::now() - minimalSendingInterval_)
            , queue_("BackendDeviceStateUpdater")
        {
            retryDelayCounter_.setSettings(RetryDelayCounter::Settings{
                .init = std::chrono::seconds(15),
                .max = std::chrono::minutes(5),
            });

            authProvider_->ownerAuthInfo().connect([this](std::shared_ptr<const AuthInfo2> authInfo) {
                queue_.add([this, authInfo = std::move(authInfo)]() {
                    headers_ = makeHeaders(authInfo->authToken);
                });
            }, lifetime_);
        }

        void updateNetworkInfoField(glagol::IBackendApi::NetworkInfo info) override {
            queue_.add([this, info = std::move(info)]() {
                const bool alreadyDelayed = !payload_.empty();
                payload_ = jsonToString(info.serialize());
                if (!alreadyDelayed) {
                    const auto passedTime = std::chrono::steady_clock::now() - lastSent_;
                    if (passedTime >= minimalSendingInterval_) {
                        doSending();
                    } else {
                        auto delayTime = std::chrono::duration_cast<std::chrono::milliseconds>(minimalSendingInterval_ - passedTime);
                        scheduleSending(delayTime);
                    }
                }
            });
        }

        void setMinimalSendingInterval(std::chrono::seconds interval) override {
            queue_.add([this, interval]() {
                if (interval == std::chrono::seconds(0)) {
                    minimalSendingInterval_ = DEFAULT_MINIMAL_SENDING_INTERVAL;
                } else {
                    minimalSendingInterval_ = interval;
                }
            });
        }
    };
} // namespace

std::unique_ptr<glagol::BackendDeviceStateUpdater> glagol::makeBackendDeviceStateUpdater(std::shared_ptr<YandexIO::IDevice> device, std::shared_ptr<quasar::IAuthProvider> authProvider, std::shared_ptr<quasar::ISimpleHttpClient> backendClient) {
    return std::make_unique<UpdaterImpl>(std::move(device), std::move(authProvider), std::move(backendClient));
}
