#include "connecting_queue.h"

using namespace glagol;

ConnectingQueue::ConnectionData::ConnectionData(DeviceId id, DiscoveredItemPtr item)
    : deviceId(id)
    , discoveredItem(std::move(item))
          {};

ConnectingQueue::ConnectingQueue(unsigned poolSize,
                                 ConnectorsFactory connectorsFactory,
                                 OnConnected onConnected)
    : poolSize_(poolSize)
    , connectorsFactory_(std::move(connectorsFactory))
    , onConnected_(std::move(onConnected))
{
}

void ConnectingQueue::queueConnection(const DeviceId& id, DiscoveredItemPtr item) {
    std::scoped_lock<std::mutex> lock(mutex_);
    waitingQueue_.emplace_back(id, std::move(item));
    scheduleConnect();
}

void ConnectingQueue::checkTimeouted(std::chrono::seconds interval) {
    checkTimeoutedImpl(interval);
    std::scoped_lock<std::mutex> lock(mutex_);
    scheduleConnect();
}

void ConnectingQueue::setPoolSize(unsigned size) {
    std::scoped_lock<std::mutex> lock(mutex_);
    poolSize_ = size;
}

void ConnectingQueue::checkTimeoutedImpl(std::chrono::seconds interval) {
    ConnectionsList timeouted;
    {
        std::scoped_lock<std::mutex> lock(mutex_);
        const auto now = std::chrono::steady_clock::now();
        while (!connectingQueue_.empty() && now - connectingQueue_.front().started > interval) {
            auto iter = connectingQueue_.begin();
            iter->timeouted = true;
            timeouted.splice(timeouted.end(), connectingQueue_, iter);
        }
    };
    for (auto& data : timeouted) {
        data.connector.reset(); // to sure that iterator used in onStateChangedCb is valid until connector will be destructed
    }
}

void ConnectingQueue::scheduleConnect() {
    if (connectorsFactory_ && !waitingQueue_.empty() && connectingQueue_.size() < poolSize_) {
        // get from begining of waitingQueue and add into the end of connectingQueue
        auto iter = waitingQueue_.begin();
        connectingQueue_.splice(connectingQueue_.end(), waitingQueue_, iter);
        auto& data = *iter;
        data.started = std::chrono::steady_clock::now();
        data.connector = connectorsFactory_();
        data.connector->setOnStateChangedCallback(
            [this, iter, called = false](auto state) mutable {
                YIO_LOG_INFO("Connector state chagned to " << std::to_string(state));
                if (state == glagol::ext::Connector::State::CONNECTED && !called) {
                    called = true;
                    connected(iter);
                }
            });
        data.connector->connect(data.discoveredItem, {});
    }
};

void ConnectingQueue::connected(ConnectionsList::iterator iter) {
    std::unique_lock<std::mutex> lock(mutex_);
    if (iter->timeouted) {
        return;
    }
    auto connector = std::move(iter->connector);
    auto deviceId = iter->deviceId;
    connectingQueue_.erase(iter);
    lock.unlock();
    onConnected_(deviceId, std::move(connector));
}
