#include "discovered_items.h"

#include "connecting_queue.h"
#include "resolves_tracker.h"

#include <yandex_io/libs/glagol_sdk/i_discovery.h>
#include <yandex_io/libs/glagol_sdk/avahi_wrapper/avahi_browse_client.h>
#include <yandex_io/libs/logging/logging.h>

YIO_DEFINE_LOG_MODULE("glagol");

bool glagol::isLocalIPAddress(const std::string& address) {
    return (address.starts_with("127.") || address == "::1" || address.starts_with("::ffff:127.") || address.starts_with("fe80:") // local-link non routable
    );
}

using namespace glagol;

namespace {
    bool invalidAddress(const std::string& address) {
        return (address.empty() || isLocalIPAddress(address));
    }

    constexpr const char* YANDEX_IO_SERVICENAME_PREFIX = "YandexIOReceiver-";
    constexpr const char* YANDEX_IO_SERVICETYPE = "_yandexio._tcp";

    bool invalidResolve(const ResolveInfo& resolve) {
        if (resolve.port <= 0 || resolve.port >= 65536) {
            return true;
        }
        return invalidAddress(resolve.address);
    }

    std::string toString(const ResolveItem::Info& info) {
        std::string result;
        result += "hostname=" + info.hostname + ", ";
        result += "address=" + info.address + ", ";
        result += "port=" + std::to_string(info.port) + ", ";
        result += "txt={";
        for (const auto& [name, val] : info.txt) {
            result += name;
            result += '=';
            result += val;
            result += ',';
        };
        result += '}';
        return result;
    }

    std::string toString(const IDiscovery::Result::Item& info) {
        // clang-format off
        return("stereopair=" + quasar::proto::GlagolDiscoveryItem_GroupRole_Name(info.stereopairRole)
               + "tandem=" + quasar::proto::GlagolDiscoveryItem_GroupRole_Name(info.tandemRole));
        // clang-format on
    }

    GroupRole roleFromTxtField(const auto& srcMap, const std::string& field) {
        auto iter = srcMap.find(field);
        if (iter != srcMap.end()) {
            if (iter->second == "leader") {
                return RoleNest::LEADER;
            }
            if (iter->second == "follower") {
                return RoleNest::FOLLOWER;
            }
        }
        return RoleNest::STAND_ALONE;
    }

    std::string sourceToStr(ResolveSource source) {
        switch (source) {
            case ResolveSource::MDNS:
                return "mdns";
            case ResolveSource::PEERS:
                return "peers";
            case ResolveSource::BACKEND:
                return "backend";
            case ResolveSource::CONNINFO:
                return "conninfo";
        }
        return "ERR";
    }

    std::string makeDiscoveryTelemetryPayload(const DeviceId& id, const ResolveInfo& resolve, ResolveSource source) {
        Json::Value json(Json::objectValue);
        json["id"] = id.id;
        json["platform"] = id.platform;
        json["address"] = resolve.address;
        json["port"] = resolve.port;
        json["src"] = sourceToStr(source);
        return quasar::jsonToString(json);
    }

    class DiscoveredItems: public IDiscoveredItems {
        using ConnectionData = glagol::DiscoveredItem::ConnectionData;
        using Protocol = glagol::DiscoveredItem::Protocol;

        class Item;
        using ItemPtr = std::shared_ptr<Item>;

        class Item: public glagol::DiscoveredItem {
            mutable std::mutex mutex_;
            DiscoveredItems* owner_;
            const glagol::DeviceId id_;
            ConnectionData data_;
            OnDiscovery discoveryCb_;
            bool wasConnected_{false};

            bool dataIsFull() const {
                return !data_.uri.empty() && !data_.tlsCertificate.empty();
            }

        public:
            Item(DiscoveredItems* owner, const glagol::DeviceId& id, ConnectionData data)
                : owner_(owner)
                , id_(id)
                , data_(std::move(data))
            {
            }

            void connected() {
                std::scoped_lock<std::mutex> lock(mutex_);
                wasConnected_ = true;
            }

            bool isConnected() const {
                std::scoped_lock<std::mutex> lock(mutex_);
                return wasConnected_;
            }

            void invalidCert() override {
                {
                    std::scoped_lock<std::mutex> lock(mutex_);
                    data_.tlsCertificate.clear();
                }
                owner_->needNewCert(id_);
            }

            void discover(OnDiscovery cb) override {
                std::unique_lock<std::mutex> lock(mutex_);
                discoveryCb_ = cb;
                if (!dataIsFull()) {
                    return;
                }
                auto dataCopy = data_;
                lock.unlock();
                cb(dataCopy);
            }

            const glagol::DeviceId& getDeviceId() const override {
                return id_;
            }

            ConnectionData getData() const {
                std::scoped_lock<std::mutex> lock(mutex_);
                return data_;
            }

            void stopDiscovery() override {
                std::string uri;
                {
                    std::scoped_lock<std::mutex> lock(mutex_);
                    discoveryCb_ = nullptr;
                    uri.swap(data_.uri);
                    data_.tlsCertificate.clear();
                }
                owner_->stopItem(id_, wasConnected_, uri);
            }

            void updateCert(const std::string& cert) {
                std::unique_lock<std::mutex> lock(mutex_);
                if (data_.tlsCertificate != cert) {
                    YIO_LOG_INFO("Update certificate of " << id_.id << ' ' << id_.platform);
                    data_.tlsCertificate = cert;
                    if (discoveryCb_ && dataIsFull()) {
                        auto dataCopy = data_;
                        auto cbCopy = discoveryCb_;
                        lock.unlock();
                        cbCopy(dataCopy);
                    }
                }
            }
        };

        /* member fields */
        using ResolvedItems = IAvahiBrowseClient::ResolvedItems;
        using SourceFlag = ResolveItem::SourceFlag;
        std::shared_ptr<YandexIO::ITelemetry> telemetry_;
        mutable std::mutex mutex_;
        bool stopped_{false};
        Settings settings_;
        const DeviceId myDeviceId_;
        std::shared_ptr<IBackendApi> backendApi_;
        struct WholeDeviceData {
            struct Trait {};

            WholeDeviceData(const WholeDeviceData&) = delete;

            WholeDeviceData(Trait /*trait*/){};
            WholeDeviceData(const IBackendApi::Device& device)
                : accountDevice(device)
            {
            }

            ResolvesTracker resolves;
            std::optional<IBackendApi::Device> accountDevice;
            ItemPtr discoveryItem; // proxy class to interact with connector

            bool isConnected() const {
                return discoveryItem && discoveryItem->isConnected();
            }

            std::string accountDeviceName() const {
                return accountDevice ? accountDevice->name : std::string();
            };

            bool canBeQueued() const {
                return (resolves.hasClusterResolve() && (bool)accountDevice && !discoveryItem);
            }

            bool uriChanged(const std::string& uri) const {
                if (uri.empty()) {
                    return false;
                }
                if (!discoveryItem) {
                    return true;
                }
                return discoveryItem->getData().uri != uri;
            }
        };
        using ActiveItemsMap = std::map<DeviceId, WholeDeviceData>;
        ActiveItemsMap active_;
        unsigned resolvedActives_{0};

        std::list<ActiveItemsMap::iterator> waitNewCert_;
        std::function<void()> neighboursUpdated_;
        OnDiscoveryCallback onDiscovery_;
        ConnectingQueue connectingQueue_;

        using ConnectorPtr = std::shared_ptr<glagol::ext::Connector>;
        StoragePtr storage_;
        std::chrono::steady_clock::time_point lastSave_;

        /* routines */
        void needNewCert(const DeviceId& id) {
            {
                std::scoped_lock<std::mutex> lock(mutex_);
                auto activeIter = active_.find(id);
                if (activeIter == std::end(active_)) {
                    return;
                }
                waitNewCert_.push_back(activeIter);
            }
            backendApi_->invalidCert(id);
        }

        void stopItem(const DeviceId& id, bool wasConnected, const std::string& uri) {
            YIO_LOG_INFO("Stop discoveryItem " << id.id);
            if (stopItemImpl(id, wasConnected, uri)) {
                callNeighborsUpdated();
            }
        }

        bool stopItemImpl(const DeviceId& id, bool wasConnected, const std::string& usedUri) {
            std::scoped_lock<std::mutex> lock(mutex_);
            auto activeIter = active_.find(id);
            auto rmIter = std::find(std::begin(waitNewCert_), std::end(waitNewCert_), activeIter);
            if (rmIter != std::end(waitNewCert_)) {
                waitNewCert_.erase(rmIter);
            }

            if (activeIter != active_.end()) {
                activeIter->second.discoveryItem.reset();
                const auto oldResolvesCount = resolvedActives_;
                activeIter->second.resolves.unlock(wasConnected);
                removeOutdates(activeIter->second.resolves);
                if (activeIter->second.canBeQueued()) {
                    queueConnect(id);
                }
                YIO_LOG_INFO(usedUri);
                return oldResolvesCount != resolvedActives_ || activeIter->second.uriChanged(usedUri);
            }
            YIO_LOG_ERROR_EVENT("DiscoveredItems.stopItemImpl", "Failed to find devices id '" << id.id << ":" << id.platform << "'");
            return false;
        }

        // helpers
        static ConnectionData makeConnectionData(const ResolveInfo& item,
                                                 std::string cert) {
            return {
                .protocol = item.protocol,
                .uri = item.uri(),
                .tlsCertificate = std::move(cert),
            };
        }

        static ResolveSource resolveSourceFromItem(const ResolveItem::Info& info) {
            if (info.source == ResolveItem::SourceFlag::PEERS_EXCHANGE) {
                return ResolveSource::PEERS;
            }
            return ResolveSource::MDNS;
        }

        void removeOutdates(ResolvesTracker& resolves) {
            if (resolves.removeOutdates(settings_.outdatingPeriod)) {
                --resolvedActives_;
            }
        }

        ActiveItemsMap::iterator updateItem(const DeviceId& id, ResolveInfo resolve, ResolveSource source) {
            auto [activeIter, activeInserted] = active_.emplace(id, WholeDeviceData::Trait());

            auto result = active_.end();
            const bool hadResolves = activeIter->second.resolves.hasResolve();

            if (activeIter->second.resolves.addResolve(resolve, source)) {
                telemetry_->reportEvent("glagolNewDiscovery", makeDiscoveryTelemetryPayload(id, resolve, source));
                result = activeIter;
            }
            if (!hadResolves) {
                ++resolvedActives_;
            }
            removeOutdates(activeIter->second.resolves);

            return result;
        }

        DiscoveredItemPtr makeDiscoveryItem(const DeviceId& id) {
            auto [iter, inserted] = active_.emplace(id, WholeDeviceData::Trait());
            auto& [resolves, accountDevice, discoveryItem] = iter->second;
            if (inserted || discoveryItem || !resolves.hasClusterResolve() || !accountDevice) {
                throw std::runtime_error("makeDiscoveryItem called on not fully completed data");
            }

            discoveryItem = std::make_shared<Item>(this,
                                                   iter->first,
                                                   makeConnectionData(resolves.getLockedResolve(), accountDevice->glagol.security.serverCertificate));
            return discoveryItem;
        }

        static bool isYandexIOItem(const ResolveItem& item) {
            return (item.domain == "local" && item.type == YANDEX_IO_SERVICETYPE && item.name.starts_with(YANDEX_IO_SERVICENAME_PREFIX));
        }

        void queueConnect(const DeviceId& id) {
            if (stopped_) {
                YIO_LOG_DEBUG("Stopping, no more queuing connections");
                return;
            }
            YIO_LOG_INFO("Queue connecting to " << id.id << ' ' << id.platform);
            auto discoveryItem = makeDiscoveryItem(id);
            connectingQueue_.queueConnection(id, discoveryItem);
        }

        std::function<void()> getNeighborsUpdated() {
            std::lock_guard<std::mutex> lock(mutex_);
            return neighboursUpdated_;
        }

        void callNeighborsUpdated() {
            if (auto neighboursUpdatedCopy = getNeighborsUpdated(); neighboursUpdatedCopy) {
                neighboursUpdated_();
            }
        }

        OnDiscoveryCallback getOnDiscoveryCb() {
            std::scoped_lock<std::mutex> lock(mutex_);
            return onDiscovery_;
        }

        void exportConnection(const DeviceId& id, ConnectorPtr connector) {
            {
                std::scoped_lock<std::mutex> lock(mutex_);
                auto iter = active_.find(id);
                if (iter != active_.end()) {
                    iter->second.discoveryItem->connected();
                } else {
                    YIO_LOG_WARN("exportConnection: Not found " << id.id << ':' << id.platform);
                }
            };
            getOnDiscoveryCb()(id, connector);
        }

    public:
        DiscoveredItems(DeviceId myDeviceId,
                        std::shared_ptr<YandexIO::ITelemetry> telemetry,
                        std::shared_ptr<IBackendApi> backendApi,
                        StoragePtr storage,
                        ConnectorsFactory connectorsFactory)
            : telemetry_(std::move(telemetry))
            , myDeviceId_(std::move(myDeviceId))
            , backendApi_(std::move(backendApi))
            , connectingQueue_(settings_.connectingPoolSize,
                               std::move(connectorsFactory),
                               [this](const DeviceId& id, ConnectorPtr connector) {
                                   exportConnection(id, connector);
                               })
            , storage_(std::move(storage))
        {
            if (storage_) {
                YIO_LOG_INFO("Loading resolves");
                try {
                    deserializeResolves(quasar::parseJson(storage_->getContent()));
                } catch (...) {
                }
            }
        }

        ~DiscoveredItems() {
            std::scoped_lock<std::mutex> lock(mutex_);
            stopped_ = true; // no more queueConnect();
        }

        void setSettings(const Settings& settings) override {
            std::scoped_lock<std::mutex> lock(mutex_);
            settings_ = settings;
            connectingQueue_.setPoolSize(settings_.connectingPoolSize);
        }

        void addResolve(const DeviceId& deviceId, ResolveInfo resolve, ResolveSource source) override {
            if (invalidResolve(resolve)) {
                YIO_LOG_INFO("Ignoring invalid resolve " << resolve.address << ':' << resolve.port << " for " << deviceId.id);
                return;
            }
            std::unique_lock<std::mutex> lock(mutex_);
            if (auto activeIter = updateItem(deviceId, resolve, source); activeIter != std::end(active_)) {
                YIO_LOG_INFO("Discovering " << deviceId.id << ' ' << deviceId.platform
                                            << ' ' << resolve.address << ':' << resolve.port << " updated");

                if (myDeviceId_ != deviceId && activeIter->second.canBeQueued()) {
                    queueConnect(deviceId);
                }
            } else {
                return;
            }
            lock.unlock();
            callNeighborsUpdated();
        }

        void newResolve(const ResolveItem& item, const ResolveItem::Info& details) override {
            YIO_LOG_INFO(myDeviceId_.id << " new resolve " << item.name << ' ' << details.address);
            if (!isYandexIOItem(item)) {
                return;
            }

            auto idIter = details.txt.find("deviceId");
            if (idIter == std::end(details.txt) || idIter->second.empty()) {
                return;
            }

            auto platformIter = details.txt.find("platform");
            if (platformIter == std::end(details.txt) || platformIter->second.empty()) {
                return;
            }
            YIO_LOG_DEBUG("new resolve came. Details = " << toString(details));

            DeviceId deviceId{.id = idIter->second, .platform = platformIter->second};
            ResolveInfo res{
                .address = details.address,
                .protocol = ResolveInfo::protoByAddress(details.address),
                .port = details.port,
                .cluster = (bool)details.txt.count("cluster"),
                .tandemRole = roleFromTxtField(details.txt, "td"),
                .stereopairRole = roleFromTxtField(details.txt, "sp"),
            };

            addResolve(deviceId, res, resolveSourceFromItem(details));
        }

        void removeResolve(const ResolveItem& item) override {
            removeResolve(item, SourceFlag::MDNS);
        }

        void removeResolve(const ResolveItem& item, ResolveItem::SourceFlag source) override {
            YIO_LOG_DEBUG("Notified about resolve removing " << item.name << " src " << (int)source);
        }

        void updateCert(const DeviceId& id, const std::string& cert) {
            const auto endIt = std::end(waitNewCert_);
            auto predicate = [&id](ActiveItemsMap::iterator iter) -> bool {
                return iter->first == id;
            };
            auto it = std::find_if(std::begin(waitNewCert_), endIt, predicate);
            while (it != endIt) {
                (*it)->second.discoveryItem->updateCert(cert);
                it = waitNewCert_.erase(it);
                it = std::find_if(it, endIt, predicate);
            }
        }

        void updateAccountDevices(const glagol::IBackendApi::DevicesMap& accountDevices) override {
            if (updateAccountDevicesImpl(accountDevices)) {
                callNeighborsUpdated();
            }

            handleNetworkInfo(accountDevices);
        }

        void handleNetworkInfo(const glagol::IBackendApi::DevicesMap& accountDevices) { // aka "server discovery"
            if (settings_.disableBackendDiscovery) {
                return;
            }
            for (const auto& [deviceId, device] : accountDevices) {
                if (device.networkInfo.externalPort && !device.networkInfo.IPs.empty()) {
                    ResolveInfo resolveInfo{
                        .port = device.networkInfo.externalPort,
                        .cluster = true,
                    };
                    for (const auto& ip : device.networkInfo.IPs) {
                        resolveInfo.address = ip;
                        resolveInfo.protocol = ResolveInfo::protoByAddress(resolveInfo.address);
                        addResolve(deviceId, resolveInfo, ResolveSource::BACKEND);
                    }
                }
            }
        }

        bool updateAccountDevicesImpl(const glagol::IBackendApi::DevicesMap& accountDevices) {
            std::scoped_lock<std::mutex> lock(mutex_);
            bool neighborsUpdated = false;
            YIO_LOG_INFO("Updating account devices of size " << accountDevices.size());
            auto srcit = std::begin(accountDevices);
            auto srcend = std::end(accountDevices);
            auto dstit = std::begin(active_);
            auto dstend = std::end(active_);
            while (srcit != srcend && dstit != dstend) {
                if (srcit->first < dstit->first) {
                    YIO_LOG_INFO("New account device " << srcit->first.id << ' ' << srcit->first.platform);
                    // newitem
                    active_.emplace_hint(dstit, srcit->first, srcit->second);
                    ++srcit;
                } else if (dstit->first < srcit->first) {
                    // disappeared item
                    if (dstit->second.accountDevice) {
                        YIO_LOG_INFO("Device " << dstit->first.id << " " << dstit->first.platform << " isn't assigned to account anymore");
                        dstit->second.accountDevice.reset(); // FIXME: arase if dstit->second is not filled ?
                        if (dstit->second.resolves.hasResolve()) {
                            neighborsUpdated = true;
                        }
                    }
                    ++dstit;
                } else { // same item
                    if (!dstit->second.accountDevice) {
                        YIO_LOG_INFO("New account device " << srcit->first.id << ' ' << srcit->first.platform);
                        auto& wholeData = dstit->second;
                        wholeData.accountDevice = srcit->second;
                        if (wholeData.resolves.hasResolve()) {
                            neighborsUpdated = true;
                        }
                        if (myDeviceId_ != dstit->first && wholeData.canBeQueued()) {
                            queueConnect(dstit->first);
                        } else {
                            YIO_LOG_DEBUG("Do not queue connection: "
                                          << (wholeData.resolves.hasClusterResolve() ? "hasClusterResolve" : "noClusterResolve") << ", "
                                          << ((bool)wholeData.accountDevice ? "hasAccountDevice" : "noAccountDevice") << ", "
                                          << ((bool)wholeData.discoveryItem ? "alreadyQueued" : "notQueuedYet"));
                        }
                    } else if (srcit->second != *(dstit->second.accountDevice)) { // something updated
                        YIO_LOG_INFO("Update account device " << srcit->first.id << ' ' << srcit->first.platform);
                        updateCert(srcit->first, srcit->second.glagol.security.serverCertificate);
                        dstit->second.accountDevice = srcit->second;
                        if (dstit->second.resolves.hasResolve()) {
                            neighborsUpdated = true;
                        }
                    }
                    ++dstit;
                    ++srcit;
                }
            }

            if (srcit != srcend) {
                while (srcit != srcend) {
                    YIO_LOG_INFO("New account device " << srcit->first.id << ' ' << srcit->first.platform);
                    active_.emplace_hint(dstit, srcit->first, srcit->second);
                    ++srcit;
                }
            } else {
                while (dstit != dstend) {
                    if (dstit->second.accountDevice) {
                        YIO_LOG_INFO("Device " << dstit->first.id << " " << dstit->first.platform << " isn't assigned to account anymore");
                        dstit->second.accountDevice.reset();
                        if (dstit->second.resolves.hasResolve()) {
                            neighborsUpdated = true;
                        }
                    }
                    ++dstit; // FIXME: erase if dstit->second is not filled?
                }
            }

            return neighborsUpdated;
        }

        void eachResolvedItem(ResolvedItemCallback callback) const override {
            if (!callback) {
                return;
            }
            auto discoveries = getDiscoveries().items;
            for (const auto& [id, discovery] : discoveries) {
                ResolveItem item{
                    .name = YANDEX_IO_SERVICENAME_PREFIX + id.id,
                    .type = YANDEX_IO_SERVICETYPE,
                    .domain = "local",
                };
                ResolveItem::Info info{
                    .hostname = id.id + '_' + id.platform,
                    .port = discovery.port,
                    .address = discovery.address,
                    .txt = {
                        {"deviceId", id.id},
                        {"platform", id.platform}}};
                if (discovery.cluster) {
                    info.txt.emplace("cluster", "yes");
                };
                callback(item, info);
            }
        }

        IDiscovery::Result getDiscoveries() const override {
            IDiscovery::Result result;
            {
                std::scoped_lock<std::mutex> lock(mutex_);
                for (const auto& [id, wholeData] : active_) {
                    const auto& resolves = wholeData.resolves;
                    if (resolves.hasResolve()) {
                        auto res = resolves.getLastResolve();
                        auto [iter, inserted] =
                            result.items.emplace(id,
                                                 IDiscovery::Result::Item{
                                                     .name = YANDEX_IO_SERVICENAME_PREFIX + id.id,
                                                     .accountDeviceName = wholeData.accountDeviceName(),
                                                     .host = id.id + '_' + id.platform,
                                                     .address = res.address,
                                                     .port = res.port,
                                                     .uri = res.uri(),
                                                     .isAccountDevice = (bool)wholeData.accountDevice,
                                                     .cluster = resolves.isCluster(),
                                                     .protocol = res.protocol == Protocol::IPV4 ? IDiscovery::Result::Protocol::ipv4 : IDiscovery::Result::Protocol::ipv6,
                                                     .tandemRole = wholeData.accountDevice ? wholeData.accountDevice->tandemRole : resolves.tandemRole(),
                                                     .stereopairRole = resolves.stereopairRole(),
                                                     .connectionActive = wholeData.isConnected(),
                                                 });
                        YIO_LOG_DEBUG("Discovery::Item [" << id.id << ":" << id.platform << "] = " << toString(iter->second));
                    }
                }
            }
            return result;
        }

        DevicesArround getDevicesAround() const override {
            DevicesArround result;
            std::scoped_lock<std::mutex> lock(mutex_);
            for (const auto& [id, wholeData] : active_) {
                if (wholeData.resolves.hasResolve()) {
                    auto res = wholeData.resolves.getLastResolve();
                    result.emplace_hint(std::end(result), id, NearbyDevice{.isAccountDevice = bool(wholeData.accountDevice), .address = res.address});
                }
            }
            return result;
        }

        bool noDevicesAround() const override { // FIXME: currently returns also non account devices
            std::scoped_lock<std::mutex> lock(mutex_);
            return resolvedActives_ == 0;
        }

        void setOnDiscoveryCallback(OnDiscoveryCallback cb) override {
            std::scoped_lock<std::mutex> lock(mutex_);
            onDiscovery_ = std::move(cb);
        }

        void setOnNeighborsUpdated(std::function<void()> cb) override {
            std::scoped_lock<std::mutex> lock(mutex_);
            neighboursUpdated_ = std::move(cb);
        }

        Json::Value serializResolves() {
            Json::Value result(Json::arrayValue);
            for (const auto& [id, data] : active_) {
                if (data.resolves.hasResolve() && data.accountDevice && id != myDeviceId_) {
                    Json::Value item(Json::objectValue);
                    item["id"] = id.id;
                    item["platform"] = id.platform;
                    item["resolves"] = data.resolves.serialize();
                    result.append(std::move(item));
                }
            }
            return result;
        }

        void deserializeResolves(const Json::Value& src) {
            if (!src.isArray()) {
                YIO_LOG_DEBUG("cannot deserialize resolves from non-array json");
                return;
            }
            const unsigned size = src.size();
            for (unsigned i = 0; i < size; ++i) {
                const auto& item = src[i];
                if (item.isMember("id") && item.isMember("platform") && item.isMember("resolves")) {
                    DeviceId deviceId{
                        .id = item["id"].asString(),
                        .platform = item["platform"].asString(),
                    };
                    if (!deviceId.id.empty() && !deviceId.platform.empty()) {
                        auto [iter, inserted] = active_.emplace(std::move(deviceId), WholeDeviceData::Trait());
                        if (inserted) {
                            auto& dst = iter->second.resolves;
                            dst.deserialize(item["resolves"]);
                            if (dst.hasResolve()) {
                                ++resolvedActives_;
                            }
                        } else {
                            YIO_LOG_WARN("Repeated resolves for " << iter->first.id << "/" << iter->first.platform);
                        }
                    }
                }
            }
            YIO_LOG_INFO("Loaded " << resolvedActives_ << " resolves");
        }

        void periodicCheck() override {
            connectingQueue_.checkTimeouted(settings_.connectingTimeout);

            std::scoped_lock<std::mutex> lock(mutex_);
            const auto now = std::chrono::steady_clock::now();
            if (storage_ && (now - lastSave_) > settings_.saveInterval) {
                YIO_LOG_INFO("Saving resolves");
                storage_->save(quasar::jsonToString(serializResolves()));
                lastSave_ = now;
                YIO_LOG_INFO("Saving resolves done");
            }
        }
    };

} // namespace

std::shared_ptr<IDiscoveredItems> glagol::createDiscoveredItems(DeviceId selfDeviceId,
                                                                std::shared_ptr<YandexIO::ITelemetry> telemetry,
                                                                std::shared_ptr<IBackendApi> backendApi,
                                                                IDiscoveredItems::StoragePtr storage,
                                                                IDiscoveredItems::ConnectorsFactory connectorsFactory) {
    return std::make_shared<DiscoveredItems>(std::move(selfDeviceId), std::move(telemetry), std::move(backendApi), std::move(storage), std::move(connectorsFactory));
}
