#include "glagol.h"

#include <yandex_io/interfaces/auth/connector/auth_provider.h>
#include <yandex_io/libs/base/directives.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/glagol_sdk/backend_api.h>
#include <yandex_io/libs/protobuf_utils/debug.h>
#include <yandex_io/libs/telemetry/telemetry.h>
#include <yandex_io/libs/http_client/http_client.h>
#include <yandex_io/sdk/converters/glagol_device_list_converter.h>
#include <yandex_io/sdk/converters/glagol_discovery_result_converter.h>
#include <yandex_io/sdk/interfaces/directive.h>

#include <chrono>
#include <csignal>
#include <limits>
#include <memory>

YIO_DEFINE_LOG_MODULE("glagol");

using namespace quasar;
using namespace quasar::proto;
using namespace YandexIO;

using std::placeholders::_1;
using std::placeholders::_2;

// name for quasar server
const std::string Glagol::SERVICE_NAME = "glagold";

// eligible values for 'status' response field
const Glagol::Status Glagol::UNSUPPORTED = "UNSUPPORTED";
const Glagol::Status Glagol::FAILURE = "FAILURE";
const Glagol::Status Glagol::SUCCESS = "SUCCESS";

const std::string Glagol::IDLE = "IDLE";
const std::string Glagol::LISTENING = "LISTENING";
const std::string Glagol::SPEAKING = "SPEAKING";
const std::string Glagol::BUSY = "BUSY";
const std::string Glagol::SHAZAM = "SHAZAM";
const std::string Glagol::NONE = "NONE";
const std::string Glagol::GUEST_FORBIDDEN_ERROR = "Command not allowed in guest mode";

namespace {
    const int HEARTBEAT_NOTIFY_PERIOD_SEC = 5;                          // every 5 seconds
    const int NUMBER_OF_HEARTBEAT_NOTIFY_PASS_TO_HEARTBEAT_METRIC = 12; // every 12 HEARTBEAT_NOTIFIES send HEARTBEAT_BETRIC (every minute)
    const std::string SETRACE_REQUEST_ID = "setraceRequestId";

    QuasarMessage makeGlagolDiscoveryResult(const glagol::IDiscovery::Result& discoveryResult) {
        QuasarMessage msg;
        auto protoResult = YandexIO::convertDiscoveryResultToProto(discoveryResult);
        msg.mutable_glagol_discovery_result()->CopyFrom(protoResult);
        return msg;
    }

    glagol::DeviceId makeGlagolDeviceId(std::shared_ptr<YandexIO::IDevice>& device) {
        return glagol::DeviceId{.id = device->deviceId(), .platform = device->configuration()->getDeviceType()};
    }

    MDNSHolderPtr initMdns(MDNSHolderFactory& mdnsFactory,
                           std::shared_ptr<YandexIO::IDevice> device,
                           const Json::Value& config,
                           const Glagol::Settings& glagolSettings,
                           GlagolCluster& glagolCluster) {
        MdnsSettings mdnsSettings{
            glagolSettings.avahi.flags,
            quasar::getUInt32(config, "externalPort"),
            quasar::tryGetString(config, "hostnamePrefix", ""),
        };

        if (glagolSettings.nsdInsteadOfAvahi) {
            return mdnsFactory(device, mdnsSettings, glagolCluster.getResolveHandler(), NsdSettings());
        }
        return mdnsFactory(device, mdnsSettings, glagolCluster.getResolveHandler(), glagolSettings.avahi);
    }

    bool isConnectedNetworkStatus(const quasar::proto::NetworkStatus& networkStatus) {
        using NetworkStatus = quasar::proto::NetworkStatus;
        return (networkStatus.status() == NetworkStatus::CONNECTED || networkStatus.status() == NetworkStatus::CONNECTED_NO_INTERNET);
    };

    std::optional<std::string> getSsidFromWifiInfo(const quasar::proto::WifiInfo& message) {
        if (message.has_ssid()) {
            return message.ssid();
        }
        return {};
    }

    std::optional<std::string> getMacFromWifiInfo(const quasar::proto::WifiInfo& message) {
        if (message.has_mac()) {
            return message.mac();
        }
        return {};
    }

    std::string toString(quasar::MultiroomState::Mode mode) {
        using Mode = quasar::MultiroomState::Mode;
        switch (mode) {
            case Mode::UNDEFINED:
                return "UNDEFINED";
            case Mode::NONE:
                return "NONE";
            case Mode::MASTER:
                return "MASTER";
            case Mode::SLAVE:
                return "SLAVE";
        }
        return "UNDEFINED";
    }

    std::string toString(quasar::MultiroomState::SyncLevel level) {
        using Level = quasar::MultiroomState::SyncLevel;
        switch (level) {
            case Level::UNDEFINED:
                return "UNDEFINED";
            case Level::NOSYNC:
                return "NOSYNC";
            case Level::WEAK:
                return "WEAK";
            case Level::STRONG:
                return "STRONG";
        }
        return "UNDEFINED";
    }

    std::string toString(glagol::GroupRole role) {
        return quasar::proto::GlagolDiscoveryItem_GroupRole_Name(role);
    }

    std::string toString(quasar::StereoPairState::Role role) {
        return quasar::StereoPairState::roleName(role);
    }

    std::string toString(quasar::StereoPairState::Channel channel) {
        return quasar::StereoPairState::channelName(channel);
    }

    std::string toString(quasar::StereoPairState::Connectivity conn) {
        return quasar::StereoPairState::connectivityName(conn);
    }

    std::string toString(quasar::StereoPairState::StereoPlayerStatus status) {
        return quasar::StereoPairState::stereoPlayerStatusName(status);
    }

    bool operator!=(const glagol::AvahiSettings& a, const glagol::AvahiSettings& b) {
        return a.restrictToIPv4 != b.restrictToIPv4 ||
               a.ratelimitIntervalUsec != b.ratelimitIntervalUsec ||
               a.ratelimitBurst != b.ratelimitBurst ||
               a.recreateBrowserInterval != b.recreateBrowserInterval ||
               a.queuingResolves != b.queuingResolves;
    }

    std::string capitalizeFirstChar(std::string str) {
        str[0] = std::toupper(str[0]);
        return str;
    }

    glagol::ResolveInfo::Protocol toResolveProtocol(glagol::NetDevicesMonitor::Family family) {
        return (family == glagol::NetDevicesMonitor::Family::IPV4 ? glagol::ResolveInfo::Protocol::IPV4 : glagol::ResolveInfo::Protocol::IPV6);
    }

    glagol::GroupRole getTandemRole(const std::string& groupConfig, const std::string& deviceId, const std::string& platform) {
        try {
            auto config = parseJson(groupConfig);
            return glagol::IBackendApi::Device::roleFromGroupJson(config, deviceId, platform);
        } catch (...) {
        }
        return glagol::RoleNest::STAND_ALONE;
    }

    std::tuple<glagol::NetDevicesMonitor::Address, std::vector<std::string>, std::vector<std::string>>
    getNetDeviceDetails(glagol::NetDevicesMonitor& netDevicesMonitor) {
        using Family = glagol::NetDevicesMonitor::Family;
        glagol::NetDevicesMonitor::Address bestAddr{.family = Family::IPV6};
        std::vector<std::string> allIPs;
        std::vector<std::string> allMACs; // station-max have eth
        netDevicesMonitor.eachRunningDevice([&bestAddr, &allIPs, &allMACs](const glagol::NetDevicesMonitor::NetDevice& netDev) {
            // skip 'usbX' adb interface
            if (netDev.interfaceName.starts_with("usb") || bestAddr.family == Family::IPV4) {
                return;
            }
            allMACs.push_back(netDev.MAC);
            for (const auto& addr : netDev.addresses) {
                if (!glagol::isLocalIPAddress(addr.addr)) {
                    allIPs.push_back(addr.addr);
                }
                if (bestAddr.addr.empty() || addr.family == Family::IPV4) {
                    bestAddr = addr;
                }
            }
        });
        return {std::move(bestAddr), std::move(allIPs), std::move(allMACs)};
    }

    // helpers to eliminate copypastes and double searches in Json::Value
    void fillPlayerState(Json::Value& dst, const SimplePlayerState& simplePlayerState) {
        dst["id"] = simplePlayerState.id();
        dst["type"] = simplePlayerState.type();
        dst["hasPause"] = simplePlayerState.has_pause();
        dst["hasPlay"] = simplePlayerState.has_play();
        dst["hasNext"] = simplePlayerState.has_next();
        dst["hasPrev"] = simplePlayerState.has_prev();
        dst["progress"] = simplePlayerState.position();
        dst["duration"] = simplePlayerState.duration();
        dst["title"] = simplePlayerState.title();
        dst["subtitle"] = simplePlayerState.subtitle();
        dst["playerType"] = simplePlayerState.player_type();
        const auto& map = simplePlayerState.extra();
        // FIXME: actually extra is not filled at all
        Json::Value& extra = dst["extra"];
        extra.clear();
        for (const auto& item : map) {
            extra[item.first] = item.second;
        }

        dst["hasProgressBar"] = simplePlayerState.has_progress_bar();
        dst["liveStreamText"] = simplePlayerState.live_stream_text();
        dst["showPlayer"] = simplePlayerState.show_player();

        const auto& entity = simplePlayerState.entity_info();
        auto& jsonEntity = dst["entityInfo"];
        jsonEntity["id"] = entity.id();
        jsonEntity["type"] = entity.type();
        jsonEntity["description"] = entity.description();
        if (entity.has_shuffled()) {
            jsonEntity["shuffled"] = entity.shuffled();
        }
        if (entity.has_repeat_mode() && !entity.repeat_mode().empty()) {
            jsonEntity["repeatMode"] = entity.repeat_mode();
        }

        if (entity.has_prev()) {
            auto& jsonPrev = jsonEntity["prev"];
            jsonPrev["id"] = entity.prev().id();
            jsonPrev["type"] = entity.prev().type();
        }

        if (entity.has_next()) {
            auto& jsonNext = jsonEntity["next"];
            jsonNext["id"] = entity.next().id();
            jsonNext["type"] = entity.next().type();
        }

        // backward compability
        dst["playlistId"] = entity.id();
        dst["playlistType"] = entity.type();
        dst["playlistDescription"] = entity.description();
    }

    std::shared_ptr<HttpClient> makeHttpClient(std::shared_ptr<YandexIO::IDevice> device, const std::string& debugName) {
        return std::make_shared<HttpClient>(debugName, std::move(device));
    }
} // namespace

const std::map<AliceState_State, std::string> Glagol::stateToName{
    {AliceState_State_IDLE, Glagol::IDLE},
    {AliceState_State_LISTENING, Glagol::LISTENING},
    {AliceState_State_SPEAKING, Glagol::SPEAKING},
    {AliceState_State_BUSY, Glagol::BUSY},
    {AliceState_State_SHAZAM, Glagol::SHAZAM},
    {AliceState_State_NONE, Glagol::NONE}};

const std::map<std::string, AliceState_State> Glagol::nameToState{
    {Glagol::IDLE, AliceState_State_IDLE},
    {Glagol::LISTENING, AliceState_State_LISTENING},
    {Glagol::SPEAKING, AliceState_State_SPEAKING},
    {Glagol::BUSY, AliceState_State_BUSY},
    {Glagol::SHAZAM, AliceState_State_SHAZAM},
    {Glagol::NONE, AliceState_State_NONE}};

quasar::proto::AliceState_State Glagol::getStateByName(const std::string& stateName) {
    auto iter = nameToState.find(stateName);
    return iter == nameToState.end() ? AliceState_State_NONE : iter->second;
};

Glagol::AliceRequestEventsHandler::AliceRequestEventsHandler()
    : future_(promise_.get_future())
{
}

quasar::ipc::SharedMessage Glagol::AliceRequestEventsHandler::getResponse()
{
    const auto result = future_.wait_for(std::chrono::seconds(6));
    return result == std::future_status::ready ? future_.get() : quasar::ipc::SharedMessage();
}

void Glagol::AliceRequestEventsHandler::onAliceRequestStarted(std::shared_ptr<YandexIO::VinsRequest> request)
{
    Y_UNUSED(request);
}

void Glagol::AliceRequestEventsHandler::onAliceRequestCompleted(
    std::shared_ptr<YandexIO::VinsRequest> request, const Json::Value& response)
{
    auto message = ipc::buildMessage([&](auto& msg) {
        msg.set_request_id(TString(request->getId()));
        msg.mutable_vins_call_response()->set_status(proto::VinsCallResponse::OK);
        msg.mutable_vins_call_response()->set_vins_response(jsonToString(response));
    });

    promise_.set_value(message);
}

void Glagol::AliceRequestEventsHandler::onAliceRequestError(
    std::shared_ptr<YandexIO::VinsRequest> request, const std::string& errorCode, const std::string& errorText)
{
    auto message = ipc::buildMessage([&](auto& msg) {
        msg.set_request_id(TString(request->getId()));
        msg.mutable_vins_call_response()->set_status(proto::VinsCallResponse::ERROR);
        msg.mutable_vins_call_response()->set_vins_response("{}");
        msg.mutable_vins_call_response()->set_error_text_lang("ru");
        msg.mutable_vins_call_response()->set_error_code(TString(errorCode));
        msg.mutable_vins_call_response()->set_error_text(TString(errorText));
    });

    promise_.set_value(message);
}

Glagol::Glagol(std::shared_ptr<YandexIO::IDevice> device,
               std::shared_ptr<ipc::IIpcFactory> ipcFactory,
               std::shared_ptr<IAuthProvider> authProvider,
               std::shared_ptr<IDeviceStateProvider> deviceStateProvider,
               std::shared_ptr<IMultiroomProvider> multiroomProvider,
               std::shared_ptr<IStereoPairProvider> stereoPairProvider,
               MDNSHolderFactory mdnsFactory,
               const Json::Value& config,
               const Settings& settings,
               std::shared_ptr<YandexIO::SDKInterface> sdk)
    : device_(std::move(device))
    , featuresConfig_(device_->configuration()->getServiceConfig("aliced"))
    , quasarServer_(ipcFactory->createIpcServer(SERVICE_NAME))
    , aliceConnector_(ipcFactory->createIpcConnector("aliced"))
    , syncdConnector_(ipcFactory->createIpcConnector("syncd"))
    , videodConnector_(ipcFactory->createIpcConnector("videod"))
    , networkdConnector_(ipcFactory->createIpcConnector("networkd"))
    , multiroomProvider_(std::move(multiroomProvider))
    , stereoPairProvider_(std::move(stereoPairProvider))
    , deviceContext_(ipcFactory, /* on connect */ nullptr, /* autoconnect */ false)
    , nsdInsteadOfAvahi_(settings.nsdInsteadOfAvahi)
    , avahiSettings_(settings.avahi)
    , extraLogging_(settings.extraLogging)
    , heartbeatTelemetry_(settings.heartbeatTelemetry)
    , skipConnectionResolves_(settings.skipConnectionResolves)
    , backendDeviceStateUpdater_(glagol::makeBackendDeviceStateUpdater(device_, authProvider, makeHttpClient(device_, "glagol-network-info")))
    , wsServer_(device_, ipcFactory, authProvider, deviceStateProvider)
    , glagolCluster_(device_,
                     std::move(ipcFactory),
                     std::move(authProvider),
                     std::bind(&Glagol::onDiscovery, this, std::placeholders::_1),
                     std::bind(&Glagol::reRequestAccountDevices, this),
                     sdk->getEndpointStorage(),
                     settings.cluster)
    , mdnsFactory_(std::move(mdnsFactory))
    , mdnsHolder_(initMdns(mdnsFactory_, device_, config, settings, glagolCluster_))
    , startTime_(std::chrono::steady_clock::now())
    , sdk_(std::move(sdk))
{
    YIO_LOG_DEBUG("Construct Glagol service with features "
                  << "supported: " << jsonToString(FeaturesConfig::formatFeatures(featuresConfig_.getSupportedFeatures()), true)
                  << ", unsupported: " << jsonToString(FeaturesConfig::formatFeatures(featuresConfig_.getUnsupportedFeatures()), true));
    netDevicesMonitor_ = glagol::makeNetDevicesMonitor(
        [this, myDeviceId = makeGlagolDeviceId(device_)](const glagol::NetDevicesMonitor::NetDevice& netDev) {
            netDeviceUpdated(myDeviceId, netDev);
        });
    // default state
    state_["volume"] = 0;
    state_["timeSinceLastVoiceActivity"] = std::numeric_limits<int64_t>::max();
    // partial backward compatibility
    state_["canStop"] = false;
    state_["playing"] = false;
    state_["aliceState"] = "UNKNOWN";

    if (multiroomProvider_) {
        multiroomProvider_->multiroomState().connect(
            [this](std::shared_ptr<const MultiroomState> mrState) {
                std::scoped_lock<std::mutex> lock(multiRoom_.mutex);
                multiRoom_.mode = mrState->mode;
                multiRoom_.syncLevel = mrState->slaveSyncLevel;
                if (mrState->broadcast) {
                    multiRoom_.masterId = mrState->broadcast->masterDeviceId;
                }
            }, lifetime_);
    }

    // we want to watch wsServer's state and relay it to all our clients
    // in future, we may modify / use state in this class
    wsServer_.setOnConnectionsChanged(makeSafeCallback(
        [this](auto&&... args) {
            onGlagolWSConnectionsChanged(std::forward<decltype(args)>(args)...);
        }, lifetime_));

    wsServer_.setOnMessage(makeSafeCallback(
        [this](auto&&... args) {
            onGlagolWSMessage(std::forward<decltype(args)>(args)...);
        }, lifetime_));

    wsServer_.setOnClose(makeSafeCallback(
        [this](const GlagolWsServer::ConnectionDetails& details) {
            if (details.type == proto::GlagoldState::CLUSTER && details.deviceId) {
                glagolCluster_.clientPeerDisconnected(details.deviceId.value());
            }
        }, lifetime_));

    syncdConnector_->setMessageHandler(makeSafeCallback([this](const ipc::SharedMessage& message) {
        if (message->has_user_config_update()) {
            std::unique_lock<std::mutex> lock(quasarStateMutex_);
            const auto systemConfig = quasar::tryParseJson(message->user_config_update().config()).value_or(Json::objectValue)["system_config"];
            featuresConfig_.processNewConfig(systemConfig);
            if (message->user_config_update().has_group_config()) {
                changeTandemRole(getTandemRole(message->user_config_update().group_config(),
                                               device_->deviceId(),
                                               device_->configuration()->getDeviceType()));
            }
            lock.unlock();
            if (systemConfig.isObject()) {
                applySystemConfig(systemConfig);
            }
        }

        if (message->has_account_devices_list()) {
            auto devicesList = YandexIO::convertDeviceListFromProto(message->account_devices_list());
            auto iterToSelf = devicesList.find(glagol::DeviceId{device_->deviceId(), device_->configuration()->getDeviceType()});
            if (iterToSelf != devicesList.end()) {
                changeGuestMode(iterToSelf->second.guestMode);
            };
            wsServer_.updateAccountDevices(devicesList);
            glagolCluster_.updateAccountDevices(devicesList);
        }
    }, lifetime_));

    videodConnector_->setMessageHandler(makeSafeCallback(
        [this](const auto& message) {
            if (message->has_watched_video()) {
                std::lock_guard lock(quasarStateMutex_);
                YIO_LOG_DEBUG("New watched video state");

                watchedVideoState_ = message->watched_video();
            }
        }, lifetime_));

    aliceConnector_->setMessageHandler(makeSafeCallback(
        [this](const auto& message) {
            onAlicedMessage(message);
        }, lifetime_));

    networkdConnector_->setMessageHandler(makeSafeCallback(
        [this](const auto& message) {
            if (message->has_network_status()) {
                const auto& networkStatus = message->network_status();
                if (networkStatus.has_wifi_status()) {
                    const auto& wifiStatus = networkStatus.wifi_status();
                    if (wifiStatus.has_current_network()) {
                        const auto& currentNetwork = wifiStatus.current_network();
                        std::lock_guard<std::mutex> guard(glagoldStateMutex_);
                        wifiSsid_ = getSsidFromWifiInfo(currentNetwork);
                        wifiMac_ = getMacFromWifiInfo(currentNetwork);
                    }
                }
                std::scoped_lock<std::mutex> lock(mdnsHolderMutex_);
                mdnsHolder_->setNetworkAvailability(isConnectedNetworkStatus(networkStatus));
            }
        }, lifetime_));

    quasarServer_->setClientConnectedHandler(makeSafeCallback(
        [this](auto& connection) {
            QuasarMessage msg;
            std::lock_guard<std::mutex> guard(glagoldStateMutex_);
            msg.mutable_glagold_state()->CopyFrom(glagoldState_);
            connection.send(std::move(msg));

            YIO_LOG_DEBUG("Client connected -- sent state");
        }, lifetime_));

    quasarServer_->setMessageHandler(makeSafeCallback(
        [this](const auto& message, auto& connection)
        {
            if (message->has_glagol_cluster_message()) {
                glagolCluster_.processClusterMessage(message->glagol_cluster_message());
            } else if (message->has_glagol_discovery_request()) {
                connection.send(makeGlagolDiscoveryResultMessage());
            } else if (message->has_simple_player_state()) {
                {
                    std::lock_guard lock(stateMutex_);

                    const SimplePlayerState& simplePlayerState = message->simple_player_state();
                    fillPlayerState(state_["playerState"], simplePlayerState);

                    // partial backward compatibility
                    state_["canStop"] = simplePlayerState.has_pause();
                    state_["playing"] = simplePlayerState.has_pause();
                }
                notifyHeartbeat();
            }
        }, lifetime_));

    if (stereoPairProvider_) {
        stereoPairProvider_->stereoPairState().connect(
            [this, ipcFactory](const auto& state) {
                changeStereopairRole(state->role);
                std::lock_guard<std::mutex> lock(stateMutex_);
                stereoPair_ = state;
            }, lifetime_);
    }

    syncdConnector_->connectToService();
    videodConnector_->tryConnectToService();
    aliceConnector_->connectToService();
    networkdConnector_->connectToService();

    quasarServer_->listenService();

    wsServer_.start();

    glagolCluster_.deviceList().connect(
        [this](const auto& deviceList) {
            onClusterDevicelistChanged(deviceList);
        }, lifetime_);

    static_assert(NUMBER_OF_HEARTBEAT_NOTIFY_PASS_TO_HEARTBEAT_METRIC * HEARTBEAT_NOTIFY_PERIOD_SEC == 60, "HEARTBEAT METRIC SHOULD BE SEND EVERY MINUTE!");

    heartbeatNotifier_ = std::make_unique<PeriodicExecutor>(
        [this, counter = 0]() mutable {
            notifyHeartbeat();
            if (++counter == NUMBER_OF_HEARTBEAT_NOTIFY_PASS_TO_HEARTBEAT_METRIC) {
                sendHeartbeatTelemetry();
                counter = 0;
            }
        }, std::chrono::seconds(HEARTBEAT_NOTIFY_PERIOD_SEC)); // FIXME: make that configurable!

    deviceContext_.connectToSDK();
    YIO_LOG_INFO("Started...");
}

Glagol::~Glagol()
{
    netDevicesMonitor_->stop();
    heartbeatNotifier_.reset();
    lifetime_.die();
}

void Glagol::updateLocalAddress(std::string bestAddr) {
    std::lock_guard<std::mutex> lock(glagoldStateMutex_);
    if (bestAddr.empty()) {
        localAddress_.reset();
    } else {
        localAddress_ = std::move(bestAddr);
    }

    if (glagoldState_.public_ip() != localAddress_.value_or("")) {
        if (localAddress_) {
            glagoldState_.set_public_ip(TString(*localAddress_));
        } else {
            glagoldState_.clear_public_ip();
        }
        QuasarMessage msg;
        msg.mutable_glagold_state()->CopyFrom(glagoldState_);
        quasarServer_->sendToAll(std::move(msg));
    }
}

glagol::IBackendApi::NetworkInfo Glagol::makeNetworkInfo(std::vector<std::string> macs) const {
    glagol::IBackendApi::NetworkInfo result;
    std::scoped_lock<std::mutex> lock(glagoldStateMutex_);

    result.IPs = IPsReportedToBackend_;
    result.MACs = std::move(macs);
    if (wifiSsid_) {
        result.wifiSsid = wifiSsid_.value();
    }
    result.ts = time(nullptr);
    if (auto extPort = wsServer_.getConfiguredPort()) {
        result.externalPort = extPort.value();
    }
    if (avahiSettings_.flags.stereopair) {
        result.stereopairRole = avahiSettings_.flags.stereopair.value() ? glagol::RoleNest::LEADER : glagol::RoleNest::FOLLOWER;
    };
    return result;
}

void Glagol::netDeviceUpdated(const glagol::DeviceId& myDeviceId, const glagol::NetDevicesMonitor::NetDevice& netDev) {
    if (netDev.loopback) {
        return;
    }
    // track changes to report telemetry.
    {
        InterfaceState curState = {.state = InterfaceState::State::DOWN, .updated = std::chrono::steady_clock::now()};
        if (netDev.running) {
            curState.state = netDev.addresses.empty() ? InterfaceState::State::UP_NO_ADDR : InterfaceState::State::UP_WITH_ADDR;
        }
        auto [iter, inserted] = netDevStates_.emplace(netDev.interfaceName, curState);
        if (!inserted && iter->second.state != curState.state) {
            auto& [name, prev] = *iter;
            auto makeKeyValues = [&name = name, prev = prev, &curState]() -> std::unordered_map<std::string, std::string> {
                auto diff = std::chrono::duration_cast<std::chrono::milliseconds>(curState.updated - prev.updated);
                return {
                    {"name", name},
                    {"spentMs", std::to_string(diff.count())},
                };
            };
            if (prev.state == InterfaceState::State::DOWN) {
                device_->telemetry()->reportKeyValues("NetInterfaceUp", makeKeyValues());
            }
            if (curState.state == InterfaceState::State::UP_WITH_ADDR) {
                device_->telemetry()->reportKeyValues("NetInterfaceGotAddress", makeKeyValues());
            }
            prev = curState;
        }
    }
    // self resolves
    if (auto port = wsServer_.getConfiguredPort(); port && netDev.running) {
        for (const auto& address : netDev.addresses) {
            glagolCluster_.addResolve(myDeviceId,
                                      glagol::ResolveInfo{
                                          .address = address.addr,
                                          .protocol = toResolveProtocol(address.family),
                                          .port = port.value(),
                                          .cluster = true},
                                      glagol::ResolveSource::CONNINFO);
        }
    }

    glagolCluster_.lifecycleExecute([this]() {
        auto [bestAddr, allIPs, allMACs] = getNetDeviceDetails(*netDevicesMonitor_);
        if (!allIPs.empty() && allIPs != IPsReportedToBackend_) {
            IPsReportedToBackend_ = std::move(allIPs);
            backendDeviceStateUpdater_->updateNetworkInfoField(makeNetworkInfo(std::move(allMACs)));
        }
        updateLocalAddress(std::move(bestAddr.addr));
    });
}

void Glagol::setOnReconfigure(std::function<void()> callback) {
    onReconfigure_ = std::move(callback);
}

Glagol::Settings Glagol::fromSystemConfig(const Json::Value& systemConfig) {
    Settings result;
    result.heartbeatTelemetry = tryGetBool(systemConfig, "glagolHeartbeat", true);
    result.nsdInsteadOfAvahi = tryGetBool(systemConfig, "nsdInsteadOfAvahi", false);
    result.avahi.restrictToIPv4 = tryGetBool(systemConfig, "restrictAvahiToIPv4", true);
    if (systemConfig.isMember("glagold")) { // start to move glagold related settings here
        const auto& glagoldConfig = systemConfig["glagold"];
        if (glagoldConfig.isObject()) {
            result.extraLogging = tryGetBool(glagoldConfig, "extraLogging", result.extraLogging);
            result.skipConnectionResolves = tryGetBool(glagoldConfig, "skipConnectionResolves", result.skipConnectionResolves);
            result.cluster.disableBackendDiscovery = tryGetBool(glagoldConfig, "disableBackendDiscovery", false);
            result.cluster.peersExchange = tryGetBool(glagoldConfig, "peersExchange", result.cluster.peersExchange);
            auto setSecondsVar = [&glagoldConfig](std::chrono::seconds& dst, const char* name) {
                dst = std::chrono::seconds(tryGetInt(glagoldConfig, name, dst.count()));
            };
            setSecondsVar(result.cluster.minimumPeersExchangeInterval, "minimumPeersExchangeInterval");
            setSecondsVar(result.cluster.peerCleanupPeriod, "peerCleanupPeriod");
            setSecondsVar(result.cluster.peerKpaTimeout, "peerKpaTimeout");
            setSecondsVar(result.cluster.resolvesOutdatingInterval, "resolvesOutdatinginterval");
            setSecondsVar(result.avahi.recreateBrowserInterval, "avahiRecreateBrowserInterval");
            result.avahi.ratelimitBurst = tryGetInt(glagoldConfig, "avahiRatelimitBurst", 0);
            result.avahi.ratelimitIntervalUsec = tryGetInt64(glagoldConfig, "avahiRatelimitIntervalUsec", 0);
            result.avahi.queuingResolves = tryGetBool(glagoldConfig, "avahiQueuingResolves", false);
            setSecondsVar(result.networkInfoSendInterval, "networkInfoSendInterval");
        }
    }
    return result;
}

void Glagol::reinitMdns(const Settings& settings) {
    mdnsHolder_->reconfigure();
    mdnsHolder_.reset();
    mdnsHolder_ = initMdns(mdnsFactory_, device_, device_->configuration()->getServiceConfig("glagold"), settings, glagolCluster_);
}

void Glagol::applySystemConfig(const Json::Value& systemConfig) {
    auto newSettings = fromSystemConfig(systemConfig);

    heartbeatTelemetry_ = newSettings.heartbeatTelemetry;
    extraLogging_ = newSettings.extraLogging;

    std::unique_lock lock(mdnsHolderMutex_);
    newSettings.avahi.flags.guestMode = guestMode_;
    if (stereoPair_) {
        if (stereoPair_->isFollower()) {
            newSettings.avahi.flags.stereopair = false;
        } else if (stereoPair_->isStereoPair()) {
            newSettings.avahi.flags.stereopair = true;
        }
    }
    if (newSettings.nsdInsteadOfAvahi != nsdInsteadOfAvahi_ || (avahiSettings_ != newSettings.avahi && !newSettings.nsdInsteadOfAvahi)) {
        YIO_LOG_INFO("mdns setting changed to: nsdInsteadOfAvahi = "
                     << newSettings.nsdInsteadOfAvahi
                     << ", restrictToIpv4 = " << newSettings.avahi.restrictToIPv4
                     << ", ratelimitIntervalUsec = " << newSettings.avahi.ratelimitIntervalUsec
                     << ", ratelimitBurst = " << newSettings.avahi.ratelimitBurst
                     << ", " << glagol::toString(newSettings.avahi.flags));
        reinitMdns(newSettings);
        nsdInsteadOfAvahi_ = newSettings.nsdInsteadOfAvahi;
        avahiSettings_ = newSettings.avahi;
        if (onReconfigure_) {
            onReconfigure_();
        }
    }
    lock.unlock();

    glagolCluster_.updateSettings(newSettings.cluster);
    skipConnectionResolves_ = newSettings.skipConnectionResolves;
    backendDeviceStateUpdater_->setMinimalSendingInterval(newSettings.networkInfoSendInterval);
}

void Glagol::changeGuestMode(bool guestMode) {
    std::lock_guard<std::mutex> lock(mdnsHolderMutex_);
    guestMode_ = guestMode;
    wsServer_.setGuestMode(guestMode);
    if (guestMode != avahiSettings_.flags.guestMode) {
        YIO_LOG_INFO("Guests mode chaged to: " << (guestMode ? "allowed" : "forbidden"));
        avahiSettings_.flags.guestMode = guestMode;
        Settings tmpSettings;
        tmpSettings.avahi = avahiSettings_;
        tmpSettings.nsdInsteadOfAvahi = nsdInsteadOfAvahi_;
        reinitMdns(tmpSettings);
    };
}

void Glagol::reinitMdns() {
    Settings tmpSettings;
    tmpSettings.avahi = avahiSettings_;
    tmpSettings.nsdInsteadOfAvahi = nsdInsteadOfAvahi_;
    reinitMdns(tmpSettings);
}

void Glagol::changeStereopairRole(StereoPairState::Role role) {
    std::optional<bool> optRole;
    if (role == StereoPairState::Role::FOLLOWER) {
        optRole = false;
    } else if (role == StereoPairState::Role::LEADER) {
        optRole = true;
    }
    std::lock_guard<std::mutex> lock(mdnsHolderMutex_);
    if (optRole != avahiSettings_.flags.stereopair) {
        YIO_LOG_INFO("Stereo pair mode is " << toString(role));
        avahiSettings_.flags.stereopair = optRole;
        reinitMdns();
    }
}

void Glagol::changeTandemRole(glagol::GroupRole role) {
    using RoleNest = glagol::RoleNest;
    std::optional<bool> optRole;
    if (role == RoleNest::FOLLOWER) {
        optRole = false;
    } else if (role == RoleNest::LEADER) {
        optRole = true;
    }
    std::lock_guard<std::mutex> lock(mdnsHolderMutex_);
    if (optRole != avahiSettings_.flags.stereopair) {
        YIO_LOG_INFO("Tandem mode is " << toString(role));
        avahiSettings_.flags.tandem = optRole;
        reinitMdns();
    }
}

std::string Glagol::getServiceName() const {
    return Glagol::SERVICE_NAME;
}

void Glagol::start() {
    // not implemented
}

std::shared_ptr<ResponderStatus> Glagol::getResponderStatusProvider() {
    std::lock_guard<std::mutex> lock(mdnsHolderMutex_);
    return mdnsHolder_->getResponderStatusProvider();
}

void Glagol::forceHeartbeatTelemetry() {
    sendHeartbeatTelemetry();
}

void Glagol::sendHeartbeatTelemetry() {
    if (!heartbeatTelemetry_) {
        return;
    }
    Json::Value result(Json::objectValue);
    {
        auto& stats = result["stats"];
        auto uptimeMs = std::chrono::duration_cast<std::chrono::milliseconds>(std::chrono::steady_clock::now() - startTime_);
        stats["glagol_uptime_ms"] = std::uint64_t(uptimeMs.count());
        {
            auto& multiroom = stats["multiroom"];
            std::scoped_lock<std::mutex> lock(multiRoom_.mutex);
            multiroom["mode"] = toString(multiRoom_.mode);
            multiroom["sync_level"] = toString(multiRoom_.syncLevel);
            if (multiRoom_.masterId) {
                multiroom["last_master"] = *multiRoom_.masterId;
            }
        }
        if (stereoPairProvider_) {
            std::lock_guard<std::mutex> lock(stateMutex_);
            if (stereoPair_ && stereoPair_->isStereoPair()) {
                auto& stereopair = stats["stereopair"];
                stereopair["role"] = toString(stereoPair_->role);
                stereopair["channel"] = toString(stereoPair_->channel);
                stereopair["partnerId"] = stereoPair_->partnerDeviceId;
                stereopair["connectivity"] = toString(stereoPair_->connectivity);
                stereopair["player_status"] = toString(stereoPair_->stereoPlayerStatus);
            }
        }
        std::lock_guard<std::mutex> lock(glagoldStateMutex_);
        if (wifiSsid_) {
            stats["wifi_ssid"] = *wifiSsid_;
        }
        if (wifiMac_) {
            stats["wifi_mac"] = *wifiMac_;
        }
        if (localAddress_) {
            stats["local_ip"] = *localAddress_;
        }
    };
    glagolCluster_.completeDevicesTelemetry(wsServer_.connectedDevicesTelemetry(std::move(result)));
}

QuasarMessage Glagol::makeGlagolDiscoveryResultMessage() const {
    std::lock_guard<std::mutex> lock(stateMutex_);
    return makeGlagolDiscoveryResult(discoveryResult_);
}

void Glagol::onDiscovery(const glagol::IDiscovery::Result& newResult) {
    if (std::lock_guard<std::mutex> lock(stateMutex_); newResult == discoveryResult_) {
        return;
    } else {
        discoveryResult_ = newResult;
        deviceContext_.fireDiscoveryResult(discoveryResult_);
    }
    QuasarMessage msg;
    msg.mutable_glagol_discovery_updated();
    quasarServer_->sendToAll(std::move(msg));
}

void Glagol::onCapabilityAdded(const std::shared_ptr<YandexIO::IEndpoint>& /*enpdoint*/, const std::shared_ptr<YandexIO::ICapability>& capability)
{
    if (const auto state = capability->getState(); state.HasDeviceStateCapability()) {
        YIO_LOG_INFO("Subscribe to DeviceStateCapability");
        capability->addListener(shared_from_this());
    }
}

void Glagol::onCapabilityRemoved(const std::shared_ptr<YandexIO::IEndpoint>& /*enpdoint*/, const std::shared_ptr<YandexIO::ICapability>& capability)
{
    if (const auto state = capability->getState(); state.HasDeviceStateCapability()) {
        YIO_LOG_INFO("Unsubscribe from DeviceStateCapability");
        capability->removeListener(shared_from_this());
    }
}

void Glagol::onEndpointStateChanged(const std::shared_ptr<YandexIO::IEndpoint>& /*endpoint*/)
{
    // ¯\_(ツ)_/¯
}

void Glagol::onCapabilityStateChanged(const std::shared_ptr<YandexIO::ICapability>& /*capability*/, const NAlice::TCapabilityHolder& state)
{
    if (!state.HasDeviceStateCapability() || !state.GetDeviceStateCapability().GetState().HasDeviceState()) {
        return;
    }

    if (!state.GetDeviceStateCapability().GetState().GetDeviceState().HasSoundLevel()) {
        return;
    }

    const int volume = state.GetDeviceStateCapability().GetState().GetDeviceState().GetSoundLevel();
    const int maxVolume = state.GetDeviceStateCapability().GetState().GetDeviceState().GetSoundMaxLevel();

    if (stateVolumeChanged(volume, maxVolume)) {
        notifyHeartbeat();
    }
}

void Glagol::onCapabilityEvents(const std::shared_ptr<YandexIO::ICapability>& /*capability*/, const std::vector<NAlice::TCapabilityEvent>& /*events*/)
{
    // ¯\_(ツ)_/¯
}

bool Glagol::stateVolumeChanged(int volume, int maxVolume)
{
    std::lock_guard<std::mutex> lock(stateMutex_);

    double relativeVolume = (double)volume / maxVolume;

    if (relativeVolume != state_["volume"].asDouble()) {
        state_["volume"] = relativeVolume;
        YIO_LOG_INFO("Changed volume to " << relativeVolume);
        return true;
    }
    return false;
}

void Glagol::onClusterDevicelistChanged(const GlagolCluster::DeviceList& deviceList)
{
    std::lock_guard<std::mutex> guard(glagoldStateMutex_);
    glagoldState_.clear_cluster_device_list();
    for (const auto& deviceId : *deviceList) {
        glagoldState_.add_cluster_device_list(TString(deviceId));
    }
    YIO_LOG_DEBUG("Glagol cluster device list changed: " << shortUtf8DebugString(glagoldState_));
    QuasarMessage msg;
    msg.mutable_glagold_state()->CopyFrom(glagoldState_);
    quasarServer_->sendToAll(std::move(msg));
}

void Glagol::onGlagolWSConnectionsChanged(GlagoldState state)
{
    YIO_LOG_DEBUG("WS Connections changed, now {" << state.connections().size() << "} connections");
    GlagoldState oldState;
    bool hasYandexDeviceConnection = false;
    {
        std::lock_guard<std::mutex> guard(glagoldStateMutex_);
        /* save old state to send metrics */
        oldState.CopyFrom(glagoldState_);
        /* Save new state */
        glagoldState_ = state;

        if (state.connections_size() > 0) {
            for (const auto& item : state.connections()) {
                if (item.type() == proto::GlagoldState::YANDEXIO_DEVICE) {
                    glagoldState_.set_microphone_group_state(toGroupSpotterMicrophoneState(groupSpotterMicrophoneState_));
                    hasYandexDeviceConnection = true;
                    break;
                }
            }
        }

        if (!hasYandexDeviceConnection) {
            glagoldState_.set_microphone_group_state(GroupSpotterMicrophoneState::LISTENS);
        }

        auto deviceList = glagolCluster_.deviceList().value();
        for (const auto& deviceId : *deviceList) {
            glagoldState_.add_cluster_device_list(TString(deviceId));
        }

        if (localAddress_) {
            glagoldState_.set_public_ip(TString(*localAddress_));
        }

        YIO_LOG_DEBUG("Glagol state changed: " << shortUtf8DebugString(glagoldState_));
        QuasarMessage msg;
        msg.mutable_glagold_state()->CopyFrom(glagoldState_);
        quasarServer_->sendToAll(std::move(msg));
    }

    if (hasYandexDeviceConnection) {
        /* Got new active connection -> send heartbeat, so Leader will get effective (followers) state */
        notifyHeartbeat();
    }

    sendConnectionsMetrica(oldState, state);
}

void Glagol::onAlicedMessage(const quasar::ipc::SharedMessage& message)
{
    if (message->has_environment_message()) {
        const auto env = message->environment_message();
        if (env.has_device_info()) {
            std::lock_guard<std::mutex> guard(quasarStateMutex_);
            environmentState_.mutable_device_info()->CopyFrom(env.device_info());
        }
    }
    if (message->has_alice_state()) {
        const AliceState& aliceState = message->alice_state();
        AliceState_State state = aliceState.state();
        {
            std::lock_guard<std::mutex> guard(stateMutex_);
            state_["aliceState"] = mapState(state);

            if (lastState_ != state) {
                if (state == AliceState_State_LISTENING) {
                    lastAliceActivity_ = std::chrono::steady_clock::now();
                }
                lastState_ = state;
            }
        }

        notifyHeartbeat();
    }
    if (message->has_app_state()) {
        {
            std::lock_guard lock(quasarStateMutex_);
            appState_ = message->app_state();
        }

        notifyHeartbeat();
    }
}

void Glagol::reRequestAccountDevices() {
    QuasarMessage msg;
    msg.mutable_rerequest_devices_list();
    syncdConnector_->sendMessage(std::move(msg));
}

void Glagol::logSetraceEvent(const std::string& nameSuffix, const std::string& requestId, SetraceArgs args) {
    args[SETRACE_REQUEST_ID] = requestId;
    device_->telemetry()->reportKeyValues("glagol" + capitalizeFirstChar(nameSuffix), args);
}

void Glagol::setConnectionDetails(WebsocketServer::ConnectionHdl hdl, GlagoldState::ConnectionType connType, const std::string& id) {
    auto resolve = wsServer_.setConnectionDetailsExt(hdl, connType, id);
    if (!id.empty()) {
        auto deviceId = glagolCluster_.isKnownAccountDevice(id);
        if (!deviceId) {
            YIO_LOG_INFO("Device with unknown id = '" << id << "' connected. Rerequesting account's devices");
            reRequestAccountDevices();
        } else if (resolve && !skipConnectionResolves_ && connType == GlagoldState::CLUSTER) {
            glagolCluster_.addResolve(*deviceId, *resolve, glagol::ResolveSource::CONNINFO);
        }
    }
}

bool Glagol::allowedGuestCommand(const std::string& command) {
    const char* guestCommands[] = {
        "playMusic",
        "playRadio",
        "ping",
        "play",
        "next",
        "prev",
        "rewind",
        "stop",
        "softwareVersion",
        "setVolume",
        "shuffle",
        "repeat",
    };
    return std::find(std::begin(guestCommands), std::end(guestCommands), command) != std::end(guestCommands);
}

void Glagol::onGlagolWSMessage(WebsocketServer::ConnectionHdl hdl, MsgConnInfo connInfo, const Json::Value& request) {
    const auto startTs = std::chrono::steady_clock::now();
    try {
        Status status = UNSUPPORTED;
        const Json::Value& payload = request["payload"];
        std::string command = payload["command"].asString();

        if (extraLogging_) {
            YIO_LOG_INFO("Incoming command '" << command << "', full request " << request);
        }

        if (connInfo.guestMode && !allowedGuestCommand(command)) {
            Json::Value response(Json::objectValue);
            response["error"] = GUEST_FORBIDDEN_ERROR;
            wsServer_.send(hdl, jsonToString(response));
            return;
        };

        if (connInfo.type == GlagoldState::GLAGOL_APP) { // connection type is not specified yet
            bool glagolClusterCommand = glagolCluster_.isExternalCommand(command);
            if (glagolClusterCommand) {
                std::string fromDeviceId = payload["from_device_id"].asString();
                setConnectionDetails(hdl, connInfo.type = GlagoldState::CLUSTER, fromDeviceId);
            } else if (command == "externalCommandBypass" ||
                       command == "aliceStateBypass" ||
                       command == "leaderOverrideBypass") {
                // this command is only sent from a leader, treat this connection as one now
                connInfo.type = GlagoldState::YANDEXIO_DEVICE;
                if (payload.isMember("from_device_id")) {
                    setConnectionDetails(hdl, connInfo.type, payload["from_device_id"].asString());
                } else {
                    wsServer_.setConnectionType(hdl, connInfo.type);
                }
            }
        }

        Json::Value response = getResponseMessageCore(connInfo.type);

        if (std::optional<bool> clusterResult; connInfo.type == GlagoldState::CLUSTER && (clusterResult = glagolCluster_.processExternalCommand(response, request))) {
            status = (*clusterResult ? SUCCESS : FAILURE);
        } else if (command == "softwareVersion") {
            const auto version = device_->softwareVersion();
            response["softwareVersion"] = version;
            response["extra"]["softwareVersion"] = version;
            status = SUCCESS;
        } else if (command == "ping") {
            status = SUCCESS; // just success, no other action needed
        } else if (command == "serverAction") {
            status = SUCCESS;
            try {
                auto request = std::make_shared<YandexIO::VinsRequest>(payload["serverActionEventPayload"], YandexIO::VinsRequest::createSoftwareDirectiveEventSource());
                auto requestEvents = std::make_shared<AliceRequestEventsHandler>();
                sdk_->getAliceCapability()->startRequest(std::move(request), requestEvents);

                auto aliceResponse = requestEvents->getResponse();
                if (!aliceResponse) {
                    throw std::runtime_error("aliceResponse is null");
                }

                const Json::Value value = parseJson(aliceResponse->vins_call_response().vins_response());
                if (!value.isNull()) {
                    response["vinsResponse"] = value;
                }
                response["errorCode"] = aliceResponse->vins_call_response().error_code();
                response["errorText"] = aliceResponse->vins_call_response().error_text();
                response["errorTextLang"] = aliceResponse->vins_call_response().error_text_lang();
                status = statusToString(aliceResponse->vins_call_response());
            } catch (const std::exception& e) {
                YIO_LOG_ERROR_EVENT("Glagol.FailedCommand.ServerAction", "Exception during sending serverAction command: " << e.what());
                status = FAILURE;
                response["errorCode"] = "GlagolDeviceInternalTimeout";
                response["errorText"] = "Произошла какая-то ошибка. Спросите попозже, пожалуйста.";
                response["errorTextLang"] = "ru";
            }
        } else if (command == "rewind") {
            if (payload["position"].isNull() || !payload["position"].isNumeric()) {
                status = FAILURE;
            } else {
                status = SUCCESS;

                Json::Value commandPayload;
                commandPayload["type"] = "absolute";
                commandPayload["amount"] = getInt(payload, "position");

                QuasarMessage message;
                message.mutable_directive()->set_is_route_locally(true);
                message.mutable_directive()->set_name(Directives::PLAYBACK_REWIND);
                message.mutable_directive()->set_json_payload(jsonToString(commandPayload));
                aliceConnector_->sendMessage(std::move(message));
            }
        } else if (command == "stop") {
            status = SUCCESS;

            QuasarMessage message;
            message.mutable_directive()->set_is_route_locally(true);
            message.mutable_directive()->set_name(Directives::PLAYBACK_PAUSE);
            aliceConnector_->sendMessage(std::move(message));
        } else if (command == "play") {
            status = SUCCESS;

            Json::Value messagePayload;
            messagePayload["player"] = "current";

            QuasarMessage message;
            message.mutable_directive()->set_is_route_locally(true);
            message.mutable_directive()->set_name(Directives::PLAYBACK_PLAY);
            message.mutable_directive()->set_json_payload(jsonToString(messagePayload));
            message.mutable_directive()->set_request_id(request["id"].asString());
            aliceConnector_->sendMessage(std::move(message));
        } else if (command == "next") {
            status = SUCCESS;

            QuasarMessage message;
            message.mutable_directive()->set_is_route_locally(true);
            message.mutable_directive()->set_name(Directives::PLAYBACK_NEXT);
            message.mutable_directive()->set_json_payload(jsonToString(payload));
            message.mutable_directive()->set_request_id(request["id"].asString());
            aliceConnector_->sendMessage(std::move(message));
        } else if (command == "prev") {
            status = SUCCESS;

            QuasarMessage message;
            message.mutable_directive()->set_is_route_locally(true);
            message.mutable_directive()->set_name(Directives::PLAYBACK_PREV);
            message.mutable_directive()->set_json_payload(jsonToString(payload));
            message.mutable_directive()->set_request_id(request["id"].asString());
            aliceConnector_->sendMessage(std::move(message));
        } else if (command == "setVolume") {
            status = SUCCESS;

            Json::Value messagePayload;
            messagePayload["new_level"] = round(payload["volume"].asDouble() * 10); // FIXME: glagol sends volume as float 0-to-1, volumed expects it as 1 to 10

            QuasarMessage message;
            message.mutable_directive()->set_is_route_locally(true);
            message.mutable_directive()->set_name(Directives::SOUND_SET_LEVEL);
            message.mutable_directive()->set_json_payload(jsonToString(messagePayload));
            aliceConnector_->sendMessage(std::move(message));
        } else if (command == "control") {
            QuasarMessage message;
            if (buildControlRequest(payload, message, request["id"].asString())) {
                status = SUCCESS;
                aliceConnector_->sendMessage(std::move(message));
            } else {
                YIO_LOG_WARN("Cannot build control request from: " << payload);
                status = FAILURE;
            }
        } else if (command == "externalCommandBypass") {
            status = SUCCESS;

            try {
                QuasarMessage commandMessage;
                Y_PROTOBUF_SUPPRESS_NODISCARD commandMessage.mutable_external_command()->ParseFromString(base64Decode(payload["data"].asString()));
                commandMessage.mutable_external_command()->set_is_route_locally(true);

                proto::QuasarMessage directiveMessage;
                auto directive = YandexIO::Directive::createFromExternalCommandMessage(commandMessage.external_command());
                directiveMessage.mutable_directive()->CopyFrom(YandexIO::Directive::convertToDirectiveProtobuf(directive));
                aliceConnector_->sendMessage(std::move(directiveMessage));

                QuasarMessage responseMessage;
                responseMessage.mutable_external_command_response()->set_processed(true);
                TString bypassResponse;
                Y_PROTOBUF_SUPPRESS_NODISCARD responseMessage.SerializeToString(&bypassResponse);

                response["extra"]["bypassResponse"] = base64Encode(bypassResponse.c_str(), bypassResponse.length());
            } catch (const std::exception& e)
            {
                YIO_LOG_ERROR_EVENT("Glagol.FailedCommand.ExternalCommandBypass", "Exception sending external command: " << e.what());
                status = FAILURE;
            }
        } else if (command == "environmentMessage") {
            status = SUCCESS;

            try {
                QuasarMessage environmentMessage;
                Y_PROTOBUF_SUPPRESS_NODISCARD environmentMessage.mutable_environment_message()->ParseFromString(base64Decode(payload["data"].asString()));
                aliceConnector_->sendMessage(std::move(environmentMessage));
            } catch (const std::exception& e) {
                YIO_LOG_ERROR_EVENT("Glagol.FailedCommand.EnvironmentMessage", "Exception sending environment message: " << e.what());
                status = FAILURE;
            }
        } else if (command == "aliceStateBypass") {
            status = SUCCESS;
            AliceState aliceState;
            Y_PROTOBUF_SUPPRESS_NODISCARD aliceState.ParseFromString(base64Decode(payload["data"].asString()));
            QuasarMessage bypassMessage;
            bypassMessage.mutable_external_alice_state()->set_state(aliceState.state());
            bypassMessage.mutable_external_alice_state()->set_recognized_phrase(aliceState.recognized_phrase());
            bypassMessage.mutable_external_alice_state()->mutable_vins_response()->CopyFrom(aliceState.vins_response());
            bypassMessage.mutable_external_alice_state()->set_doa_angle(aliceState.doa_angle());
            aliceConnector_->sendMessage(std::move(bypassMessage));
        } else if (command == "showAliceVisualState") {
            try {
                // last message we got is good enough for visualisation purposes
                groupSpotterMicrophoneState_ = payload["microphone"].asString();
                wsServer_.notifyGlagoldStatus();
                if (!payload["aliceStateName"].empty()) {
                    QuasarMessage aliceStateMessage;
                    AliceState_State aliceState = getStateByName(payload["aliceStateName"].asString());
                    aliceStateMessage.mutable_external_alice_state()->set_state(aliceState);
                    aliceStateMessage.mutable_external_alice_state()->set_recognized_phrase(payload["recognizedPhrase"].asString());
                    aliceConnector_->sendMessage(std::move(aliceStateMessage));
                }
            } catch (const std::exception& e)
            {
                YIO_LOG_ERROR_EVENT("Glagol.FailedCommand.SendShowAliceVisualState", "Exception processing showAliceVisualState: " << e.what());
                status = FAILURE;
            }
        } else if (command == "leaderOverrideBypass") {
            status = SUCCESS;

            QuasarMessage bypassMessage;
            Y_PROTOBUF_SUPPRESS_NODISCARD bypassMessage.mutable_leader_override()->ParseFromString(base64Decode(payload["data"].asString()));

            aliceConnector_->sendMessage(std::move(bypassMessage));
        } else if (command == "playRadio") {
            if (payload["id"].isNull()) {
                status = FAILURE;
                logSetraceEvent(command, request["id"].asString(), {{"status", "failure"}});
            } else {
                status = SUCCESS;
                const std::string id = payload["id"].asString();

                Json::Value event;
                event["name"] = "@@mm_semantic_frame";
                event["type"] = "server_action";

                auto& semanticFramePayload = event["payload"];
                auto& radioPlay = semanticFramePayload["typed_semantic_frame"]["radio_play_semantic_frame"];
                radioPlay["fm_radio"]["fm_radio_value"] = payload["id"].asString();
                radioPlay["disable_nlg"]["bool_value"] = true;

                auto& analytics = semanticFramePayload["analytics"];
                analytics["origin"] = "RemoteControl";
                analytics["purpose"] = "play_fm_radio";

                const auto requestId = request["id"].asString();
                auto request = std::make_shared<YandexIO::VinsRequest>(event, YandexIO::VinsRequest::createSoftwareDirectiveEventSource(), requestId);
                sdk_->getAliceCapability()->startRequest(std::move(request), nullptr);

                SetraceArgs setraceArgs = {{"status", "success"}, {"id", id}};
                logSetraceEvent(command, requestId, setraceArgs);
            }
        } else if (command == "playMusic") {
            if (payload["id"].isNull() || payload["type"].isNull() || payload["type"].asString().length() == 0) {
                status = FAILURE;
                logSetraceEvent(command, request["id"].asString(), {{"status", "failure"}});
            } else {
                status = SUCCESS;

                // In quasar.music_play_object type was lowercase
                // In music_play_semantic_frame type capitalize
                const std::string type = capitalizeFirstChar(payload["type"].asString());
                const std::string id = payload["id"].asString();
                SetraceArgs setraceArgs = {
                    {"status", "success"},
                    {"type", type},
                    {"id", id}};

                Json::Value event;
                event["name"] = "@@mm_semantic_frame";
                event["type"] = "server_action";

                auto& semanticFramePayload = event["payload"];

                auto& musicPlay = semanticFramePayload["typed_semantic_frame"]["music_play_semantic_frame"];
                musicPlay["object_id"]["string_value"] = id;
                musicPlay["object_type"]["enum_value"] = type;
                musicPlay["disable_nlg"]["bool_value"] = true;

                if (payload.isMember("startFromId")) {
                    const std::string startFromId = payload["startFromId"].asString();
                    musicPlay["start_from_track_id"]["string_value"] = startFromId;
                    setraceArgs["startFromId"] = startFromId;
                }

                if (payload.isMember("startFromPosition")) {
                    const auto startFromPosition = payload["startFromPosition"].asInt();
                    musicPlay["track_offset_index"]["num_value"] = startFromPosition;
                    setraceArgs["startFromPosition"] = std::to_string(startFromPosition);
                }

                if (payload.isMember("offset")) {
                    const auto offset = payload["offset"].asDouble();
                    musicPlay["offset_sec"]["double_value"] = offset;
                    setraceArgs["offset_sec"] = std::to_string(offset);
                }

                if (payload.isMember("from")) {
                    const auto from = payload["from"].asString();
                    musicPlay["from"]["string_value"] = from;
                    setraceArgs["from"] = from;
                }

                if (payload.isMember("shuffle") && payload["shuffle"].asBool()) {
                    musicPlay["order"]["order_value"] = "shuffle";
                }

                if (payload.isMember("repeat")) {
                    musicPlay["repeat"]["repeat_value"] = payload["repeat"].asString();
                }

                auto& analytics = semanticFramePayload["analytics"];
                analytics["origin"] = "RemoteControl";
                analytics["purpose"] = type == "Radio" ? "play_radio" : "play_music";

                const auto requestId = request["id"].asString();
                auto request = std::make_shared<YandexIO::VinsRequest>(event, YandexIO::VinsRequest::createSoftwareDirectiveEventSource(), requestId);
                sdk_->getAliceCapability()->startRequest(std::move(request), nullptr);

                logSetraceEvent(command, requestId, setraceArgs);
            }
        } else if (command == "shuffle") {
            status = SUCCESS;

            Json::Value event;
            event["name"] = "@@mm_semantic_frame";
            event["type"] = "server_action";

            const std::string shuffle = payload["enable"].asBool() ? "player_shuffle_semantic_frame" : "player_unshuffle_semantic_frame";
            auto& semanticFramePayload = event["payload"];
            auto& shuffleSemanticFrame = semanticFramePayload["typed_semantic_frame"][shuffle];
            shuffleSemanticFrame["disable_nlg"]["bool_value"] = true;

            auto& analytics = semanticFramePayload["analytics"];
            analytics["origin"] = "RemoteControl";
            analytics["purpose"] = "shuffle";

            const auto requestId = request["id"].asString();
            auto request = std::make_shared<YandexIO::VinsRequest>(event, YandexIO::VinsRequest::createSoftwareDirectiveEventSource(), requestId);
            sdk_->getAliceCapability()->startRequest(std::move(request), nullptr);
        } else if (command == "repeat") {
            status = SUCCESS;

            Json::Value event;
            event["name"] = "@@mm_semantic_frame";
            event["type"] = "server_action";

            auto& semanticFramePayload = event["payload"];
            auto& repeat = semanticFramePayload["typed_semantic_frame"]["player_repeat_semantic_frame"];
            repeat["mode"]["enum_value"] = payload["mode"].asString();
            repeat["disable_nlg"]["bool_value"] = true;

            auto& analytics = semanticFramePayload["analytics"];
            analytics["origin"] = "RemoteControl";
            analytics["purpose"] = "repeat";

            const auto requestId = request["id"].asString();
            auto request = std::make_shared<YandexIO::VinsRequest>(event, YandexIO::VinsRequest::createSoftwareDirectiveEventSource(), requestId);
            sdk_->getAliceCapability()->startRequest(std::move(request), nullptr);
        } else if (command == "sendText") {
            status = SUCCESS;

            try {
                Json::Value event;
                event["type"] = "text_input";
                event["text"] = payload["text"].asString();

                auto request = std::make_shared<YandexIO::VinsRequest>(event, YandexIO::VinsRequest::createSoftwareDirectiveEventSource());
                auto requestEvents = std::make_shared<AliceRequestEventsHandler>();

                sdk_->getAliceCapability()->startRequest(std::move(request), requestEvents);

                auto aliceResponse = requestEvents->getResponse();
                if (!aliceResponse) {
                    throw std::runtime_error("aliceResponse is null");
                }

                response["vinsResponse"] = parseJson(aliceResponse->vins_call_response().vins_response());
                response["errorCode"] = aliceResponse->vins_call_response().error_code();
                response["errorText"] = aliceResponse->vins_call_response().error_text();
                response["errorTextLang"] = aliceResponse->vins_call_response().error_text_lang();
                status = statusToString(aliceResponse->vins_call_response());
            } catch (const std::exception& e) {
                YIO_LOG_ERROR_EVENT("Glagol.FailedCommand.SendText", "Exception during sending vinsCallMessage command: " << e.what());
                status = FAILURE;
                response["errorCode"] = "GlagolDeviceInternalTimeout";
                response["errorText"] = "Произошла какая-то ошибка. Спросите попозже, пожалуйста.";
                response["errorTextLang"] = "ru";
            }
        } else if (command == "setOverridedChannel") {
            status = SUCCESS;
            if (stereoPairProvider_) {
                auto channel = StereoPairState::parseChannel(tryGetString(payload, "channel"));
                stereoPairProvider_->overrideChannel(channel);
                response["extra"]["channel"] = StereoPairState::channelName(channel);
            } else {
                response["errorCode"] = "MultiroomIsUnavailable";
                response["errorText"] = "multiroom is unavailable";
                response["errorTextLang"] = "en";
                status = FAILURE;
            }
        }

        response["requestId"] = request["id"];
        response["requestSentTime"] = request["sentTime"];
        response["sentTime"] = static_cast<int64_t>(std::chrono::duration_cast<std::chrono::milliseconds>(std::chrono::system_clock::now().time_since_epoch()).count());
        response["processingTime"] = static_cast<int64_t>(std::chrono::duration_cast<std::chrono::milliseconds>(std::chrono::steady_clock::now() - startTs).count());
        response["status"] = status;

        YIO_LOG_INFO("responding with message { id=" << response["id"] << ", requestId=" << response["requestId"] << ", status=" << response["status"] << "}");
        if (extraLogging_) {
            YIO_LOG_INFO("Full response " << response);
        }

        wsServer_.send(hdl, jsonToString(response));
    } catch (websocketpp::exception const& e) {
        YIO_LOG_WARN("Echo failed because: "
                     << "(" << e.what() << ")");
    } catch (Json::Exception const& e) {
        YIO_LOG_WARN("Invalid JSON because: "
                     << "(" << e.what() << ")");
    } catch (std::runtime_error& e) {
        YIO_LOG_ERROR_EVENT("Glagol.HandleGlagolWsMessage.Exception", e.what());
    } catch (...) {
        YIO_LOG_ERROR_EVENT("Glagol.HandleGlagolWsMessage.UnknownError", "Unknown error");
    }
}

bool Glagol::buildControlRequest(const Json::Value& payload, QuasarMessage& messageToBuild, const std::string& vinsRequestId) {
    if (payload["action"].isNull() || !payload["action"].isString()) {
        return false;
    }

    NavigationMessage* navigation = nullptr;
    std::string actionStr = payload["action"].asString();
    if ("go_left" == actionStr) {
        navigation = messageToBuild.mutable_control_request()->mutable_navigation_request()->mutable_go_left();
    } else if ("go_right" == actionStr) {
        navigation = messageToBuild.mutable_control_request()->mutable_navigation_request()->mutable_go_right();
    } else if ("go_up" == actionStr) {
        navigation = messageToBuild.mutable_control_request()->mutable_navigation_request()->mutable_go_up();
    } else if ("go_down" == actionStr) {
        navigation = messageToBuild.mutable_control_request()->mutable_navigation_request()->mutable_go_down();
    } else if ("click_action" == actionStr) {
        messageToBuild.mutable_control_request()->mutable_action_request();
    } else {
        // unexpected action
        return false;
    }

    messageToBuild.mutable_control_request()->set_vins_request_id(TString(vinsRequestId));
    messageToBuild.mutable_control_request()->set_origin(ControlRequest_Origin_TOUCH);
    if (!navigation) {
        return true;
    }

    // set up navigation details, if we need it
    NavigationScrollAmount amount;
    if (!payload["scrollAmount"].isNull()) {
        if (!buildNavigationScrollAmount(payload, amount)) {
            return false;
        }
        navigation->mutable_scroll_amount()->CopyFrom(amount);
    }

    if (!payload["mode"].isNull() && payload["mode"].isString()) {
        std::string mode = payload["mode"].asString();
        if (!mode.empty()) {
            if (mode == "visual") {
                navigation->mutable_visual_mode();
            } else if (mode == "history") {
                navigation->mutable_historical_mode();
            } else if (mode == "native") {
                navigation->mutable_native_mode();
            } else {
                YIO_LOG_ERROR_EVENT("Glagol.InvalidNavigationMode", "Unexpected command mode in payload: " << mode);
                return false;
            }
        }
    }

    return true;
}

bool Glagol::buildNavigationScrollAmount(const Json::Value& payload, NavigationScrollAmount& messageToBuild) {
    if (payload["scrollAmount"].isNull() || !payload["scrollAmount"].isString()) {
        return false;
    }

    std::string scrollAmountStr = payload["scrollAmount"].asString();
    if ("few" == scrollAmountStr) {
        messageToBuild.mutable_few();
    } else if ("many" == scrollAmountStr) {
        messageToBuild.mutable_many();
    } else if ("till_end" == scrollAmountStr) {
        messageToBuild.mutable_till_end();
    } else if ("exact" == scrollAmountStr && !payload["scrollExactValue"].isNull() && payload["scrollExactValue"].isInt()) {
        messageToBuild.mutable_exact()->set_value(payload["scrollExactValue"].asInt());
    } else {
        // unexpected scroll amount
        return false;
    }

    return true;
}

Json::Value makeHdmiState(bool capable, bool present) {
    Json::Value result(Json::objectValue);
    result["capable"] = capable;
    result["present"] = present;
    return result;
}

Json::Value Glagol::getState() const {
    std::lock_guard lock(stateMutex_);

    Json::Value result(state_); // copy
    result["timeSinceLastVoiceActivity"] = static_cast<int64_t>(std::chrono::duration_cast<std::chrono::seconds>(std::chrono::steady_clock::now() - lastAliceActivity_).count());
    const bool isHdmiCapable = appState_.has_screen_state();
    const bool isHdmiConnected = isHdmiCapable && appState_.screen_state().is_hdmi_connected();
    result["hdmi"] = makeHdmiState(isHdmiCapable, isHdmiConnected);
    return result;
}

void Glagol::notifyHeartbeat() {
    auto response = jsonToString(getResponseMessageCore(proto::GlagoldState::YANDEXIO_DEVICE));
    YIO_LOG_TRACE("broadcasting heartbeat " << response);
    wsServer_.sendAll(response,
                      [](auto /*hdl*/, auto type) {
                          return type != proto::GlagoldState::CLUSTER;
                      });
}

// prepares common part of response message, with all the data and stuff
// TODO: make the `extra` part depend on actual request and / or subscriber.
Json::Value Glagol::getResponseMessageCore(proto::GlagoldState::ConnectionType connectionType) const {
    Json::Value response;
    response["id"] = makeUUID();
    response["sentTime"] = static_cast<int64_t>(std::chrono::duration_cast<std::chrono::milliseconds>(std::chrono::system_clock::now().time_since_epoch()).count());

    // TODO: send these not always but when they change only
    if (connectionType != proto::GlagoldState::CLUSTER) {
        response["state"] = getState();

        std::lock_guard lock(quasarStateMutex_);
        response["supported_features"] = FeaturesConfig::formatFeatures(featuresConfig_.getSupportedFeatures());
        response["unsupported_features"] = FeaturesConfig::formatFeatures(featuresConfig_.getUnsupportedFeatures());
        response["experiments"] = FeaturesConfig::formatExperiments(featuresConfig_.getExperiments());
        TString appStateString;
        Y_PROTOBUF_SUPPRESS_NODISCARD appState_.SerializeToString(&appStateString);
        response["extra"]["appState"] = base64Encode(appStateString.c_str(), appStateString.length());

        TString watchedVideoStateString;
        Y_PROTOBUF_SUPPRESS_NODISCARD watchedVideoState_.SerializeToString(&watchedVideoStateString);
        response["extra"]["watchedVideoState"] = base64Encode(watchedVideoStateString.c_str(), watchedVideoStateString.length());

        TString localDeviceInfoString;
        Y_PROTOBUF_SUPPRESS_NODISCARD environmentState_.SerializeToString(&localDeviceInfoString);
        response["extra"]["environmentState"] = base64Encode(localDeviceInfoString.c_str(), localDeviceInfoString.length());
    }
    return response;
}

std::string Glagol::statusToString(const VinsCallResponse& vinsCallResponse) {
    if (!vinsCallResponse.has_status() || vinsCallResponse.status() == VinsCallResponse_Status_OK) {
        return SUCCESS;
    } else {
        return FAILURE;
    }
}

std::string Glagol::mapState(quasar::proto::AliceState_State state) {
    switch (state) {
        case AliceState_State_IDLE:
            return "IDLE";
        case AliceState_State_BUSY:
            return "BUSY";
        case AliceState_State_LISTENING:
            return "LISTENING";
        case AliceState_State_SHAZAM:
            return "SHAZAM";
        case AliceState_State_SPEAKING:
            return "SPEAKING";
        default:
            return "UNKNOWN";
    }
}

void Glagol::reportConnectionsChange(quasar::proto::GlagoldState::ConnectionType type,
                                     const ConnectionsTypeCounters& oldConnections,
                                     const ConnectionsTypeCounters& newConnections,
                                     const std::string& metricaEventNamePrefix) const {
    const unsigned long oldCnt = oldConnections[type];
    const unsigned long newCnt = newConnections[type];
    /* NOTE: Send event for each connected/disconnected device */
    if (oldCnt > newCnt) {
        const auto eventName = metricaEventNamePrefix + "_DISCONNECTED";
        for (unsigned long i = 0; i < oldCnt - newCnt; ++i) {
            device_->telemetry()->reportEvent(eventName);
        }
    } else if (oldCnt < newCnt) {
        const auto eventName = metricaEventNamePrefix + "_CONNECTED";
        for (unsigned long i = 0; i < newCnt - oldCnt; ++i) {
            device_->telemetry()->reportEvent(eventName);
        }
    }
}

Glagol::ConnectionsTypeCounters Glagol::buildConnectionsCounters(const quasar::proto::GlagoldState& state)
{
    ConnectionsTypeCounters result = {};
    for (const auto& conn : state.connections()) {
        result[conn.type()] += 1;
    }
    return result;
}

void Glagol::sendConnectionsMetrica(const quasar::proto::GlagoldState& oldState,
                                    const quasar::proto::GlagoldState& newState) const {
    device_->telemetry()->putAppEnvironmentValue("glagold_connections_count", std::to_string(newState.connections_size()));
    /* Sanity check for protobuf change */
    static_assert(GlagoldState::ConnectionType_ARRAYSIZE == 4, "Connections Type enum changed! Check that all necessary values are handled!");

    const auto newConnections = buildConnectionsCounters(newState);
    const auto oldConnections = buildConnectionsCounters(oldState);
    reportConnectionsChange(GlagoldState::ConnectionType::GlagoldState_ConnectionType_GLAGOL_APP, oldConnections, newConnections, "GLAGOL_APP");
    reportConnectionsChange(GlagoldState::ConnectionType::GlagoldState_ConnectionType_YANDEXIO_DEVICE, oldConnections, newConnections, "YANDEXIO_DEVICE");
    reportConnectionsChange(GlagoldState::ConnectionType::GlagoldState_ConnectionType_OTHER, oldConnections, newConnections, "OTHER");
    reportConnectionsChange(GlagoldState::ConnectionType::GlagoldState_ConnectionType_CLUSTER, oldConnections, newConnections, "CLUSTER");
}

void Glagol::waitWsServerStart() const {
    wsServer_.waitServerStart();
}
